const { PrismaClient } = require('@prisma/client');
const fs = require('fs');
const path = require('path');

const prisma = new PrismaClient();

async function main() {
  // Read teachers.json
  const teachersPath = path.join(process.cwd(), 'src', 'constants', 'teachers.json');
  const teachersData = JSON.parse(fs.readFileSync(teachersPath, 'utf-8'));

  console.log(`📚 Starting to seed ${teachersData.length} teachers...`);

  let created = 0;
  let skipped = 0;
  let errors = 0;

  for (const teacher of teachersData) {
    try {
      // Generate slug from name
      const slug = teacher.name
        .toLowerCase()
        .replace(/\s+/g, '-')
        .replace(/[^\w-]/g, '');

      // Check if teacher already exists
      const existing = await prisma.teacher.findUnique({
        where: { slug },
      });

      if (existing) {
        console.log(`⏭️  Skipping ${teacher.name} (already exists)`);
        skipped++;
        continue;
      }

      // Parse experience (e.g., "4 years" -> 4)
      const experienceMatch = teacher.experience?.match(/(\d+)/);
      const experience = experienceMatch ? parseInt(experienceMatch[1]) : null;

      // Create teacher
      const newTeacher = await prisma.teacher.create({
        data: {
          name: teacher.name,
          slug,
          title: teacher.gender === 'male' ? 'Mr.' : 'Mrs.',
          position: teacher.subject || 'Teacher',
          subject: teacher.subject,
          gender: teacher.gender,
          bio: teacher.bio,
          qualifications: teacher.achievement,
          image: teacher.image ? `/uploads/teachers/${teacher.image.split('/').pop()}` : null,
          experience,
          active: true,
        },
      });

      console.log(`✅ Created: ${newTeacher.name}`);
      created++;
    } catch (error) {
      console.error(`❌ Error creating ${teacher.name}:`, error.message);
      errors++;
    }
  }

  console.log('\n📊 Seed Results:');
  console.log(`✅ Created: ${created}`);
  console.log(`⏭️  Skipped: ${skipped}`);
  console.log(`❌ Errors: ${errors}`);

  // Seed leadership
  await seedLeadership();
}

async function seedLeadership() {
  const leadershipData = [
    {
      name: 'Azizullah Paeyndah',
      position: 'Founder & Director',
      shortDescription:
        'Visionary leader who established Setaragan Zirak to provide quality education during challenging times.',
      bio: 'Azizullah Paeyndah is a university lecturer and education advocate with over a decade of experience in educational leadership. His vision of accessible, quality education for all has been the driving force behind Setaragan Zirak School.',
      expertise: [
        'Educational Leadership',
        'Curriculum Development',
        'Academic Administration',
      ],
      image: null,
      order: 1,
      active: true,
    },
    {
      name: 'Qudsia Sharifi',
      position: 'Principal',
      shortDescription:
        'Educational leader focused on academic excellence and student development.',
      bio: 'Qudsia Sharifi brings exceptional leadership and educational expertise to her role as Principal. Her commitment to excellence and student-centered approach has transformed the educational experience at our school.',
      expertise: [
        'Academic Excellence',
        'Student Development',
        'Educational Innovation',
      ],
      image: null,
      order: 2,
      active: true,
    },
    {
      name: 'Mohammad Arif Payenda',
      position: 'Communication Coordinator',
      shortDescription:
        'Ensures effective communication between school, students, and families.',
      bio: 'Mohammad Arif Payenda is an experienced web application developer, researcher, and lecturer who ensures effective communication between all stakeholders and manages our digital presence.',
      expertise: [
        'Communication Strategy',
        'Technology Integration',
        'Community Outreach',
      ],
      image: null,
      order: 3,
      active: true,
    },
    {
      name: 'Zakia Ebrahimi',
      position: 'Vice Principal',
      shortDescription:
        'Supports academic programs and student achievement initiatives.',
      bio: 'Zakia Ebrahimi remains deeply committed to education and works tirelessly to ensure that every student receives the support and guidance they need to succeed academically and personally.',
      expertise: [
        'Student Support',
        'Academic Guidance',
        'Educational Policy',
      ],
      image: null,
      order: 4,
      active: true,
    },
    {
      name: 'Raihana Mohsenzada',
      position: 'Finance & Communications Manager',
      shortDescription:
        'Finance manager and educator with entrepreneurship background and extensive education experience.',
      bio: "Bachelor's in Physics from Kabul University. Finance manager and educator with entrepreneurship background. Certificate recipient from U.S. Embassy in Afghanistan.",
      expertise: [
        'Finance Management',
        'Communications',
        'Education & Entrepreneurship',
      ],
      image: null,
      order: 5,
      active: true,
    },
  ];

  console.log(`\n👥 Starting to seed ${leadershipData.length} leadership members...`);

  let created = 0;
  let skipped = 0;
  let errors = 0;

  for (const leader of leadershipData) {
    try {
      // Check if leadership member already exists
      const existing = await prisma.leadership.findFirst({
        where: { name: leader.name },
      });

      if (existing) {
        console.log(`⏭️  Skipping ${leader.name} (already exists)`);
        skipped++;
        continue;
      }

      // Create leadership member
      const newLeader = await prisma.leadership.create({
        data: {
          name: leader.name,
          position: leader.position,
          shortDescription: leader.shortDescription,
          bio: leader.bio,
          expertise: JSON.stringify(leader.expertise),
          image: leader.image,
          order: leader.order,
          active: leader.active,
        },
      });

      console.log(`✅ Created: ${newLeader.name}`);
      created++;
    } catch (error) {
      console.error(`❌ Error creating ${leader.name}:`, error.message);
      errors++;
    }
  }

  console.log('\n👥 Leadership Seed Results:');
  console.log(`✅ Created: ${created}`);
  console.log(`⏭️  Skipped: ${skipped}`);
  console.log(`❌ Errors: ${errors}`);
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
