'use client';

import { useState } from 'react';
import { useAdmissions, useDeleteAdmission, useUpdateAdmissionStatus } from '@/hooks';
import { User, Trash2, Eye } from 'lucide-react';
import Button from '@/components/common/ui/button';
import Modal from '@/components/common/ui/modal';
import Card from '@/components/common/ui/card';
import Table from '@/components/common/ui/table';

interface AdmissionsInquiry {
  id: string;
  studentName: string;
  gradeLevel: string;
  parentName: string;
  parentEmail: string;
  parentPhone: string;
  message?: string;
  hearAbout?: string;
  status: 'new' | 'contacted' | 'enrolled' | 'rejected';
  reviewed: boolean;
  createdAt: string;
}

export default function AdmissionsPage() {
  const [selectedInquiry, setSelectedInquiry] = useState<AdmissionsInquiry | null>(null);
  const [showModal, setShowModal] = useState(false);

  // Use custom hooks with React Query
  const { data, isLoading } = useAdmissions();
  const deleteAdmissionMutation = useDeleteAdmission();
  const updateStatusMutation = useUpdateAdmissionStatus();

  const handleViewInquiry = (inquiry: AdmissionsInquiry) => {
    setSelectedInquiry(inquiry);
    setShowModal(true);
  };

  const handleDelete = (id: string) => {
    if (confirm('Are you sure you want to delete this inquiry?')) {
      deleteAdmissionMutation.mutate(id);
    }
  };

  const handleStatusChange = (id: string, newStatus: string) => {
    updateStatusMutation.mutate({ id, status: newStatus });
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-gray-600">Loading admissions...</div>
      </div>
    );
  }

  const inquiries = data?.admissions || [];
  const newCount = inquiries.filter((a: AdmissionsInquiry) => !a.reviewed).length;
  const enrolledCount = inquiries.filter((a: AdmissionsInquiry) => a.status === 'enrolled').length;

  return (
    <div className="space-y-6">
      {/* Header Stats */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Admissions Inquiries</h1>
        <div className="grid grid-cols-3 gap-4 mt-4">
          <Card className="p-4">
            <p className="text-sm text-gray-600">Total</p>
            <p className="text-2xl font-bold text-gray-900">{inquiries.length}</p>
          </Card>
          <Card className="p-4 border-blue-200 bg-blue-50">
            <p className="text-sm text-blue-600">New</p>
            <p className="text-2xl font-bold text-blue-900">{newCount}</p>
          </Card>
          <Card className="p-4 border-green-200 bg-green-50">
            <p className="text-sm text-green-600">Enrolled</p>
            <p className="text-2xl font-bold text-green-900">{enrolledCount}</p>
          </Card>
        </div>
      </div>

      {/* Inquiries List */}
      {inquiries.length === 0 ? (
        <Card className="p-12 text-center">
          <User className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No admission inquiries yet</p>
        </Card>
      ) : (
        <Table
          data={inquiries}
          variant="primary"
          columns={[
            {
              key: 'studentName',
              label: 'Student',
              render: (value, row: AdmissionsInquiry) => (
                <div className="flex items-center gap-2">
                  {!row.reviewed && <div className="w-2 h-2 bg-blue-500 rounded-full"></div>}
                  <div>
                    <p className="font-medium">{value}</p>
                    <p className="text-xs text-gray-500">Grade: {row.gradeLevel}</p>
                  </div>
                </div>
              ),
            },
            {
              key: 'parentName',
              label: 'Parent',
              render: (value) => <span className="text-sm">{value}</span>,
            },
            {
              key: 'parentEmail',
              label: 'Email',
              render: (value) => (
                <a href={`mailto:${value}`} className="text-primary-600 hover:underline text-sm">
                  {value}
                </a>
              ),
            },
            {
              key: 'status',
              label: 'Status',
              render: (value: string) => {
                const statusColors = {
                  new: 'bg-blue-100 text-blue-800',
                  contacted: 'bg-yellow-100 text-yellow-800',
                  enrolled: 'bg-green-100 text-green-800',
                  rejected: 'bg-red-100 text-red-800',
                };
                return (
                  <span
                    className={`inline-flex px-2.5 py-1 rounded-full text-xs font-medium ${
                      statusColors[value as keyof typeof statusColors] || 'bg-gray-100 text-gray-800'
                    }`}
                  >
                    {value.charAt(0).toUpperCase() + value.slice(1)}
                  </span>
                );
              },
            },
            {
              key: 'createdAt',
              label: 'Date',
              sortable: true,
              render: (value) => (
                <span className="text-sm text-gray-600">
                  {new Date(value).toLocaleDateString()}
                </span>
              ),
            },
            {
              key: 'id',
              label: 'Actions',
              render: (value, row: AdmissionsInquiry) => (
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => handleViewInquiry(row)}
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded transition"
                    title="View details"
                  >
                    <Eye className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => handleDelete(value)}
                    className="p-2 text-red-600 hover:bg-red-50 rounded transition"
                    disabled={deleteAdmissionMutation.isPending}
                    title="Delete inquiry"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>
              ),
            },
          ]}
        />
      )}

      {/* Inquiry Modal */}
      {showModal && selectedInquiry && (
        <Modal
          isOpen={showModal}
          onClose={() => {
            setShowModal(false);
            setSelectedInquiry(null);
          }}
          title={`Inquiry from ${selectedInquiry.studentName}`}
        >
          <div className="space-y-4">
            {/* Student Info */}
            <div className="bg-gray-50 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 mb-2">Student</h4>
              <p className="text-gray-700">{selectedInquiry.studentName}</p>
              <p className="text-sm text-gray-600">Grade: {selectedInquiry.gradeLevel}</p>
            </div>

            {/* Parent Info */}
            <div className="bg-gray-50 p-4 rounded-lg">
              <h4 className="font-semibold text-gray-900 mb-3">Parent Information</h4>
              <div className="space-y-2">
                <div>
                  <p className="text-sm text-gray-600">Name</p>
                  <p className="font-semibold text-gray-900">{selectedInquiry.parentName}</p>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-gray-600">Email</p>
                    <a
                      href={`mailto:${selectedInquiry.parentEmail}`}
                      className="text-primary-600 hover:underline break-all"
                    >
                      {selectedInquiry.parentEmail}
                    </a>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Phone</p>
                    <a
                      href={`tel:${selectedInquiry.parentPhone}`}
                      className="text-primary-600 hover:underline"
                    >
                      {selectedInquiry.parentPhone}
                    </a>
                  </div>
                </div>
              </div>
            </div>

            {/* How They Heard */}
            {selectedInquiry.hearAbout && (
              <div>
                <p className="text-sm text-gray-600 mb-2">How did they hear about us?</p>
                <p className="text-gray-900 bg-gray-50 p-3 rounded">{selectedInquiry.hearAbout}</p>
              </div>
            )}

            {/* Message */}
            {selectedInquiry.message && (
              <div>
                <p className="text-sm text-gray-600 mb-2">Message</p>
                <div className="bg-gray-50 p-4 rounded text-gray-800 whitespace-pre-wrap text-sm">
                  {selectedInquiry.message}
                </div>
              </div>
            )}

            {/* Status Update */}
            <div className="bg-blue-50 border border-blue-200 rounded p-4">
              <p className="text-sm text-gray-600 mb-3">Update Status</p>
              <div className="grid grid-cols-2 gap-2">
                {['new', 'contacted', 'enrolled', 'rejected'].map((status) => (
                  <button
                    key={status}
                    onClick={() => handleStatusChange(selectedInquiry.id, status)}
                    className={`p-2 rounded text-sm font-medium transition ${
                      selectedInquiry.status === status
                        ? 'bg-primary-600 text-white'
                        : 'bg-white border border-gray-300 text-gray-700 hover:border-primary-500'
                    }`}
                    disabled={updateStatusMutation.isPending}
                  >
                    {status.charAt(0).toUpperCase() + status.slice(1)}
                  </button>
                ))}
              </div>
            </div>

            {/* Close Button */}
            <div className="flex justify-end pt-4 border-t">
              <Button onClick={() => setShowModal(false)}>Close</Button>
            </div>
          </div>
        </Modal>
      )}
    </div>
  );
}
