'use client';

import { useState } from 'react';
import { Mail, Trash2, Eye, Check, Send } from 'lucide-react';
import Button from '@/components/common/ui/button';
import Modal from '@/components/common/ui/modal';
import Card from '@/components/common/ui/card';
import Table from '@/components/common/ui/table';
import {
  useContacts,
  useDeleteContact,
  useMarkContactAsRead,
  useSendContactReply,
} from '@/hooks';

interface ContactMessage {
  id: string;
  name: string;
  email: string;
  phone?: string;
  subject: string;
  message: string;
  read: boolean;
  replied: boolean;
  replyMessage?: string;
  repliedAt?: string;
  createdAt: string;
}

export default function ContactsPage() {
  const [selectedContact, setSelectedContact] = useState<ContactMessage | null>(null);
  const [showModal, setShowModal] = useState(false);
  const [replyText, setReplyText] = useState('');

  // Use custom hooks with React Query
  const { data, isLoading } = useContacts();
  const deleteContactMutation = useDeleteContact();
  const markAsReadMutation = useMarkContactAsRead();
  const sendReplyMutation = useSendContactReply();

  const handleViewMessage = (contact: ContactMessage) => {
    setSelectedContact(contact);
    setShowModal(true);
    if (!contact.read) {
      markAsReadMutation.mutate(contact.id);
    }
  };

  const handleDelete = (id: string) => {
    if (confirm('Are you sure you want to delete this message?')) {
      deleteContactMutation.mutate(id);
    }
  };

  const handleReply = (replyMessage: string) => {
    if (!selectedContact || !replyMessage.trim()) return;
    sendReplyMutation.mutate(
      { id: selectedContact.id, message: replyMessage },
      {
        onSuccess: () => {
          setReplyText('');
          setSelectedContact(null);
          setShowModal(false);
        },
      }
    );
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-gray-600">Loading contacts...</div>
      </div>
    );
  }

  const contacts = data?.contacts || [];
  const unreadCount = contacts.filter((c: ContactMessage) => !c.read).length;
  console.log('ContactsPage render with contacts:', contacts, data);
  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Contact Messages</h1>
        <p className="text-gray-600 mt-2">
          {contacts.length} message{contacts.length !== 1 ? 's' : ''} (
          <span className="font-semibold text-primary-600">{unreadCount} unread</span>)
        </p>
      </div>

      {/* Messages List */}
      {contacts.length === 0 ? (
        <Card className="p-12 text-center">
          <Mail className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No contact messages yet</p>
        </Card>
      ) : (
        <Table
          data={contacts}
          variant="primary"
          columns={[
            {
              key: 'name',
              label: 'Name',
              render: (value, row: ContactMessage) => (
                <div className="flex items-center gap-2">
                  {!row.read && <div className="w-2 h-2 bg-blue-500 rounded-full"></div>}
                  <span className="font-medium">{value}</span>
                </div>
              ),
            },
            {
              key: 'email',
              label: 'Email',
              render: (value) => (
                <a href={`mailto:${value}`} className="text-primary-600 hover:underline text-sm">
                  {value}
                </a>
              ),
            },
            {
              key: 'subject',
              label: 'Subject',
              render: (value) => <span className="text-sm text-gray-700">{value}</span>,
            },
            {
              key: 'read',
              label: 'Status',
              render: (value, row: ContactMessage) => (
                <div className="flex items-center gap-2">
                  {row.replied && (
                    <span className="inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-700">
                      <Check className="w-3 h-3" />
                      Replied
                    </span>
                  )}
                  {!value && !row.replied && (
                    <span className="inline-flex px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-700">
                      Unread
                    </span>
                  )}
                  {value && !row.replied && (
                    <span className="inline-flex px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-700">
                      Read
                    </span>
                  )}
                </div>
              ),
            },
            {
              key: 'createdAt',
              label: 'Date',
              sortable: true,
              render: (value) => (
                <span className="text-sm text-gray-600">
                  {new Date(value).toLocaleDateString()}
                </span>
              ),
            },
            {
              key: 'id',
              label: 'Actions',
              render: (value, row: ContactMessage) => (
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => handleViewMessage(row)}
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded transition"
                    title="View details"
                  >
                    <Eye className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => handleDelete(value)}
                    className="p-2 text-red-600 hover:bg-red-50 rounded transition"
                    disabled={deleteContactMutation.isPending}
                    title="Delete message"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>
              ),
            },
          ]}
        />
      )}

      {/* Message Modal */}
      {showModal && selectedContact && (
        <Modal
          isOpen={showModal}
          onClose={() => {
            setShowModal(false);
            setSelectedContact(null);
          }}
          title={`Message from ${selectedContact.name}`}
        >
          <div className="space-y-4">
            {/* Contact Info */}
            <div className="bg-gray-50 p-4 rounded-lg space-y-2">
              <div>
                <p className="text-sm text-gray-600">Name</p>
                <p className="font-semibold text-gray-900">{selectedContact.name}</p>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-gray-600">Email</p>
                  <a
                    href={`mailto:${selectedContact.email}`}
                    className="text-primary-600 hover:underline"
                  >
                    {selectedContact.email}
                  </a>
                </div>
                {selectedContact.phone && (
                  <div>
                    <p className="text-sm text-gray-600">Phone</p>
                    <a
                      href={`tel:${selectedContact.phone}`}
                      className="text-primary-600 hover:underline"
                    >
                      {selectedContact.phone}
                    </a>
                  </div>
                )}
              </div>
              <div>
                <p className="text-sm text-gray-600">Date</p>
                <p className="text-gray-900">
                  {new Date(selectedContact.createdAt).toLocaleString()}
                </p>
              </div>
            </div>

            {/* Subject and Message */}
            <div>
              <p className="text-sm text-gray-600 mb-2">Subject</p>
              <p className="font-semibold text-gray-900 bg-gray-50 p-3 rounded">
                {selectedContact.subject}
              </p>
            </div>

            <div>
              <p className="text-sm text-gray-600 mb-2">Original Message</p>
              <div className="bg-blue-50 border border-blue-200 p-4 rounded text-gray-800 whitespace-pre-wrap">
                {selectedContact.message}
              </div>
            </div>

            {/* Admin Reply (if sent) */}
            {selectedContact.replied && selectedContact.replyMessage && (
              <div className="bg-green-50 border border-green-200 rounded p-4">
                <p className="text-sm text-green-700 font-semibold mb-2 flex items-center gap-2">
                  <Check className="w-4 h-4" /> Reply sent
                </p>
                <div className="text-gray-800 whitespace-pre-wrap text-sm mb-2">
                  {selectedContact.replyMessage}
                </div>
                <p className="text-xs text-gray-600">
                  {new Date(selectedContact.repliedAt!).toLocaleString()}
                </p>
              </div>
            )}

            {/* Reply Form - Always available for new replies */}
            <div className="border-t pt-4">
              <label className="block text-sm font-medium text-gray-700 mb-2">
                {selectedContact.replied ? 'Add Another Reply' : 'Send a Reply'}
              </label>
              <textarea
                value={replyText}
                onChange={(e) => setReplyText(e.target.value)}
                placeholder="Type your reply message here..."
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 resize-none"
                rows={4}
              />
              <div className="flex justify-end gap-2 mt-3">
                <Button
                  variant="outlined"
                  onClick={() => {
                    setReplyText('');
                    setShowModal(false);
                  }}
                >
                  Close
                </Button>
                <Button
                  onClick={() => handleReply(replyText)}
                  disabled={!replyText.trim() || sendReplyMutation.isPending}
                  className="flex items-center gap-2"
                >
                  <Send className="w-4 h-4" />
                  {sendReplyMutation.isPending ? 'Sending...' : 'Send Reply'}
                </Button>
              </div>
            </div>
          </div>
        </Modal>
      )}
    </div>
  );
}
