'use client';

import { useState } from 'react';
import { useDonations, useDeleteDonation } from '@/hooks';
import { Heart, Trash2, Eye, Clock, CheckCircle, AlertCircle } from 'lucide-react';
import Button from '@/components/common/ui/button';
import Modal from '@/components/common/ui/modal';
import Card from '@/components/common/ui/card';
import Table from '@/components/common/ui/table';
import ManualDonationForm from '@/components/admin/forms/ManualDonationForm';

interface Donation {
  id: string;
  name: string;
  email: string;
  phone?: string;
  amount: number;
  currency: string;
  category?: string;
  message?: string;
  paymentMethod?: string;
  anonymous: boolean;
  paymentStatus: 'pending' | 'completed' | 'failed';
  transactionId?: string;
  createdAt: string;
}

export default function DonationsPage() {
  const [selectedDonation, setSelectedDonation] = useState<Donation | null>(null);
  const [showModal, setShowModal] = useState(false);

  // Use custom hooks with React Query
  const { data, isLoading } = useDonations();
  const deleteDonationMutation = useDeleteDonation();

  const handleViewDonation = (donation: Donation) => {
    setSelectedDonation(donation);
    setShowModal(true);
  };

  const handleDelete = (id: string) => {
    if (confirm('Are you sure you want to delete this donation record?')) {
      deleteDonationMutation.mutate(id);
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-gray-600">Loading donations...</div>
      </div>
    );
  }

  const donations = data?.donations || [];
  const totalAmount = donations.reduce((sum, d) => sum + Number(d.amount), 0);
  const completedCount = donations.filter((d) => d.paymentStatus === 'completed').length;
  const pendingCount = donations.filter((d) => d.paymentStatus === 'pending').length;

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-yellow-600" />;
      case 'failed':
        return <AlertCircle className="w-4 h-4 text-red-600" />;
      default:
        return null;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="space-y-6">
      {/* Header Stats */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Donations</h1>
        <div className="grid grid-cols-4 gap-4 mt-4">
          <Card className="p-4">
            <p className="text-sm text-gray-600">Total Donations</p>
            <p className="text-2xl font-bold text-gray-900">{donations.length}</p>
          </Card>
          <Card className="p-4 border-green-200 bg-green-50">
            <p className="text-sm text-green-600">Completed</p>
            <p className="text-2xl font-bold text-green-900">{completedCount}</p>
          </Card>
          <Card className="p-4 border-yellow-200 bg-yellow-50">
            <p className="text-sm text-yellow-600">Pending</p>
            <p className="text-2xl font-bold text-yellow-900">{pendingCount}</p>
          </Card>
          <Card className="p-4 border-primary-200 bg-primary-50">
            <p className="text-sm text-primary-600">Total Amount</p>
            <p className="text-2xl font-bold text-primary-900">${totalAmount.toFixed(2)}</p>
          </Card>
        </div>
      </div>

      {/* Add Manual Donation Form */}
      <ManualDonationForm onSuccess={() => {}} />

      {/* Donations List */}
      {donations.length === 0 ? (
        <Card className="p-12 text-center">
          <Heart className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600">No donations yet</p>
        </Card>
      ) : (
        <Table
          data={donations}
          variant="primary"
          columns={[
            {
              key: 'name' as const,
              label: 'Donor',
              render: (value, row) => (
                <span className="flex items-center gap-2">
                  {row.anonymous ? (
                    <>
                      <Heart className="w-4 h-4 text-red-500" />
                      Anonymous
                    </>
                  ) : (
                    <>
                      <Heart className="w-4 h-4 text-red-500" />
                      {value}
                    </>
                  )}
                </span>
              ),
            },
            {
              key: 'email' as const,
              label: 'Email',
              render: (value, row) => row.anonymous ? '—' : value,
            },
            {
              key: 'amount' as const,
              label: 'Amount',
              sortable: true,
              render: (value, row) => (
                <span className="font-semibold text-primary-600">
                  {row.currency} {Number(value).toFixed(2)}
                </span>
              ),
            },
            {
              key: 'category' as const,
              label: 'Category',
              render: (value) => value || 'General',
            },
            {
              key: 'paymentStatus' as const,
              label: 'Status',
              render: (value) => (
                <span className={`text-xs px-2 py-1 rounded font-medium flex items-center gap-1 w-fit ${getStatusColor(value)}`}>
                  {getStatusIcon(value)}
                  {value.charAt(0).toUpperCase() + value.slice(1)}
                </span>
              ),
            },
            {
              key: 'createdAt' as const,
              label: 'Date',
              sortable: true,
              render: (value) => new Date(value).toLocaleDateString(),
            },
            {
              key: 'id' as const,
              label: 'Actions',
              render: (value, row) => (
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => handleViewDonation(row)}
                    className="p-2 text-blue-600 hover:bg-blue-50 rounded transition"
                    title="View details"
                  >
                    <Eye className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => handleDelete(row.id)}
                    className="p-2 text-red-600 hover:bg-red-50 rounded transition"
                    disabled={deleteDonationMutation.isPending}
                    title="Delete donation"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>
              ),
            },
          ]}
          rowClassName="cursor-pointer"
        />
      )}

      {/* Donation Modal */}
      {showModal && selectedDonation && (
        <Modal
          isOpen={showModal}
          onClose={() => {
            setShowModal(false);
            setSelectedDonation(null);
          }}
          title={`Donation from ${selectedDonation.anonymous ? 'Anonymous' : selectedDonation.name}`}
        >
          <div className="space-y-4">
            {/* Donor Info */}
            {!selectedDonation.anonymous && (
              <div className="bg-gray-50 p-4 rounded-lg space-y-2">
                <div>
                  <p className="text-sm text-gray-600">Name</p>
                  <p className="font-semibold text-gray-900">{selectedDonation.name}</p>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-gray-600">Email</p>
                    <a
                      href={`mailto:${selectedDonation.email}`}
                      className="text-primary-600 hover:underline break-all"
                    >
                      {selectedDonation.email}
                    </a>
                  </div>
                  {selectedDonation.phone && (
                    <div>
                      <p className="text-sm text-gray-600">Phone</p>
                      <a
                        href={`tel:${selectedDonation.phone}`}
                        className="text-primary-600 hover:underline"
                      >
                        {selectedDonation.phone}
                      </a>
                    </div>
                  )}
                </div>
              </div>
            )}

            {/* Donation Details */}
            <div className="bg-primary-50 border border-primary-200 rounded p-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-gray-600">Amount</p>
                  <p className="text-2xl font-bold text-primary-900">
                    ${Number(selectedDonation?.amount)?.toFixed(2)}
                  </p>
                </div>
                <div>
                  <p className="text-sm text-gray-600">Status</p>
                  <div className={`inline-flex items-center gap-2 px-3 py-1 rounded-full font-semibold text-sm ${getStatusColor(selectedDonation.paymentStatus)}`}>
                    {getStatusIcon(selectedDonation.paymentStatus)}
                    {selectedDonation.paymentStatus.charAt(0).toUpperCase() + selectedDonation.paymentStatus.slice(1)}
                  </div>
                </div>
              </div>
              {selectedDonation.category && (
                <div className="mt-3">
                  <p className="text-sm text-gray-600">Category</p>
                  <p className="font-semibold text-gray-900">{selectedDonation.category}</p>
                </div>
              )}
            </div>

            {/* Payment Info */}
            <div className="bg-gray-50 p-4 rounded-lg space-y-2">
              <div>
                <p className="text-sm text-gray-600">Payment Method</p>
                <p className="font-semibold text-gray-900">
                  {selectedDonation.paymentMethod || 'Not specified'}
                </p>
              </div>
              {selectedDonation.transactionId && (
                <div>
                  <p className="text-sm text-gray-600">Transaction ID</p>
                  <p className="font-mono text-sm text-gray-900 break-all">
                    {selectedDonation.transactionId}
                  </p>
                </div>
              )}
              <div>
                <p className="text-sm text-gray-600">Date</p>
                <p className="text-gray-900">
                  {new Date(selectedDonation.createdAt).toLocaleString()}
                </p>
              </div>
            </div>

            {/* Message */}
            {selectedDonation.message && (
              <div>
                <p className="text-sm text-gray-600 mb-2">Donor Message</p>
                <div className="bg-gray-50 p-4 rounded text-gray-800 whitespace-pre-wrap text-sm">
                  {selectedDonation.message}
                </div>
              </div>
            )}

            {/* Close Button */}
            <div className="flex justify-end pt-4 border-t">
              <Button onClick={() => setShowModal(false)}>Close</Button>
            </div>
          </div>
        </Modal>
      )}
    </div>
  );
}
