'use client';

import { useState } from 'react';
import Link from 'next/link';
import { useLeadership, useDeleteLeadership, type Leadership } from '@/hooks/useLeadership';
import Table from '@/components/common/ui/table';
import TablePagination from '@/components/common/ui/table-pagination';
import { Avatar } from '@/components/common/ui';
import { Plus, Trash2, Edit } from 'lucide-react';

export default function LeadershipPage() {
  const { data: leadership = [], isLoading, error, refetch } = useLeadership();
  const deleteLeadership = useDeleteLeadership();
  const [searchTerm, setSearchTerm] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 10;

  const filteredLeadership = leadership.filter(leader =>
    leader.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    leader.position.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const totalPages = Math.ceil(filteredLeadership.length / itemsPerPage);
  const paginatedLeadership = filteredLeadership.slice(
    (currentPage - 1) * itemsPerPage,
    currentPage * itemsPerPage
  );

  const handleSearch = (term: string) => {
    setSearchTerm(term);
    setCurrentPage(1);
  };

  const handleDelete = (id: string, name: string) => {
    if (confirm(`Are you sure you want to delete ${name}?`)) {
      deleteLeadership.mutate(id, {
        onSuccess: () => {
          refetch();
        },
      });
    }
  };

  const totalMembers = leadership.length;
  const activeMembers = leadership.filter(m => m.active !== false).length;

  const stats = [
    { label: 'Total Members', value: totalMembers },
    { label: 'Active', value: activeMembers },
  ];

  if (error) {
    return (
      <div className="p-6 bg-red-50 text-red-700 rounded-lg">
        Error loading leadership members: {error instanceof Error ? error.message : 'Unknown error'}
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold text-gray-900">Leadership Management</h1>
        <Link href="/admin/leadership/create">
          <button className="flex items-center gap-2 bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700 transition">
            <Plus size={20} />
            Add Member
          </button>
        </Link>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        {stats.map((stat) => (
          <div
            key={stat.label}
            className="bg-white p-4 rounded-lg shadow-sm border-l-4 border-primary-600"
          >
            <p className="text-sm text-gray-600">{stat.label}</p>
            <p className="text-3xl font-bold text-gray-900 mt-1">{stat.value}</p>
          </div>
        ))}
      </div>

      <div>
        <input
          type="text"
          placeholder="Search by name or position..."
          value={searchTerm}
          onChange={(e) => handleSearch(e.target.value)}
          className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
        />
      </div>

      {isLoading ? (
        <div className="p-8 text-center text-gray-500">Loading members...</div>
      ) : filteredLeadership.length === 0 ? (
        <div className="p-8 text-center text-gray-500 bg-white rounded-lg">
          {leadership.length === 0 ? 'No leadership members yet. Add your first member!' : 'No members match your search.'}
        </div>
      ) : (
        <div className="space-y-4">
          <Table
            data={paginatedLeadership}
            columns={[
              {
                key: 'name',
                label: 'Name',
                render: (value: string, row: Leadership) => (
                  <div className="flex items-center gap-3">
                    <Avatar
                      src={row.image}
                      name={row.name}
                      size="sm"
                      alt={`${row.name} avatar`}
                    />
                    <p className="font-medium text-gray-900">{row.name}</p>
                  </div>
                ),
              },
              {
                key: 'position',
                label: 'Position',
                render: (value: string) => <span className="text-gray-600">{value}</span>,
              },
              {
                key: 'shortDescription',
                label: 'Description',
                render: (value: string | undefined) => (
                  <span className="text-gray-600">
                    {value ? value.substring(0, 50) + '...' : '-'}
                  </span>
                ),
              },
              {
                key: 'active',
                label: 'Status',
                render: (value: boolean | undefined) => (
                  <span
                    className={`px-3 py-1 rounded-full text-xs font-semibold ${
                      value !== false
                        ? 'bg-green-100 text-green-800'
                        : 'bg-gray-100 text-gray-800'
                    }`}
                  >
                    {value !== false ? 'Active' : 'Inactive'}
                  </span>
                ),
              },
              {
                key: 'id',
                label: 'Actions',
                render: (value: string, row: Leadership) => (
                  <div className="flex items-center gap-2 justify-end">
                    <Link href={`/admin/leadership/${row.id}/edit`}>
                      <button
                        className="text-primary-600 hover:text-primary-700 p-1"
                        title="Edit"
                      >
                        <Edit size={18} />
                      </button>
                    </Link>
                    <button
                      onClick={() => handleDelete(row.id, row.name)}
                      disabled={deleteLeadership.isPending}
                      className="text-red-600 hover:text-red-700 p-1 disabled:opacity-50"
                      title="Delete"
                    >
                      <Trash2 size={18} />
                    </button>
                  </div>
                ),
              },
            ]}
            variant="primary"
          />

            <TablePagination
              currentPage={currentPage}
              totalPages={totalPages}
              onPageChange={setCurrentPage}
              itemsPerPage={itemsPerPage}
              totalItems={filteredLeadership.length}
              variant="footer"
            />
        </div>
      )}

      {/* No modal form needed - using dedicated pages */}
    </div>
  );
}
