"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useFormik } from "formik";
import { useCreateNewsArticle } from "@/hooks/useNews";
import { useAdminUsers, type AdminUser } from "@/hooks/useAdminUsers";
import { InputField, RichTextEditor, FileUpload, Button, SearchableDropdown } from "@/components/common/ui";
import { ArrowLeft } from "lucide-react";
import { NEWS_ARTICLE_SCHEMA, NEWS_CATEGORIES, NEWS_THEMES } from "@/lib/admin/schemas";

export default function CreateNewsPage() {
  const router = useRouter();
  const createArticle = useCreateNewsArticle();
  const { data: adminUsersData } = useAdminUsers();
  const [featuredImage, setFeaturedImage] = useState<File | null>(null);
  const [adminUsers, setAdminUsers] = useState<AdminUser[]>([]);
  const [useExternalAuthor, setUseExternalAuthor] = useState(false);

  useEffect(() => {
    if (adminUsersData?.users) {
      setAdminUsers(adminUsersData.users);
    }
  }, [adminUsersData]);

  const formik = useFormik({
    initialValues: {
      title: "",
      excerpt: "",
      content: "",
      category: "",
      author: "",
      authorId: "",
      theme: "primary",
      tags: [] as string[],
      readTime: 3,
      status: "draft" as "draft" | "published",
    },
    validationSchema: NEWS_ARTICLE_SCHEMA,
    onSubmit: async (values, { setSubmitting, setStatus }) => {
      // create slug
      const slug = values.title.toLowerCase().replace(/\s+/g, "-");
      try {
        // Upload featured image if selected
        let imageUrl = undefined;
        if (featuredImage) {
          const formData = new FormData();
          formData.append("file", featuredImage);
          formData.append("folder", "news");

          const uploadResponse = await fetch("/api/admin/upload", {
            method: "POST",
            body: formData,
          });

          if (!uploadResponse.ok) {
            throw new Error("Failed to upload image");
          }

          const uploadData = await uploadResponse.json();
          imageUrl = uploadData.url;
        }

        await createArticle.mutateAsync({
          ...values,
          author: values.author || undefined,
          authorId: values.authorId || undefined,
          featuredImage: imageUrl,
          slug,
        });
        router.push("/admin/news");
      } catch (err) {
        setStatus((err as Error).message || "Failed to create article");
      } finally {
        setSubmitting(false);
      }
    },
  });

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 via-blue-50 to-gray-100 py-8 px-4 sm:px-6 lg:px-8">
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Button
            variant="text"
            color="dark"
            onClick={() => router.back()}
            // className="flex items-center gap-2 text-blue-600 hover:text-blue-700 mb-4 font-medium transition"
          >
            <ArrowLeft className="w-5 h-5" />
            Back
          </Button>
          <h1 className="text-4xl font-bold text-gray-900">Create Article</h1>
          <p className="mt-2 text-gray-600">Write and publish a new article</p>
        </div>

        {/* Form */}
        <div className="bg-white rounded-lg shadow-lg border border-gray-100 p-8">
          {formik.status && (
            <div className="mb-6 p-4 bg-red-50 border-l-4 border-red-500 rounded-lg">
              <p className="text-red-700 font-medium">{formik.status}</p>
            </div>
          )}

          <form onSubmit={formik.handleSubmit} className="space-y-6 overflow-visible">
            {/* Title */}
            <div>
              <InputField
                label="Article Title"
                name="title"
                value={formik.values.title}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                placeholder="Enter article title"
                required
                error={formik.touched.title && formik.errors.title ? formik.errors.title : ""}
                inputClassName="border-gray-300 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>

            {/* Featured Image */}
            <FileUpload
              label="Featured Image"
              accept="image/jpeg,image/jpg,image/png,image/webp"
              maxSize={2}
              value={featuredImage}
              onChange={setFeaturedImage}
              previewWidth={800}
              previewHeight={400}
              helpText="PNG, JPG, or WebP (max 2MB)"
            />

            {/* Excerpt */}
            <div>
              <label htmlFor="excerpt" className="block text-sm font-semibold text-gray-800 mb-2">
                Excerpt
              </label>
              <textarea
                id="excerpt"
                name="excerpt"
                value={formik.values.excerpt}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                placeholder="Brief summary of the article"
                rows={3}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent outline-none transition bg-gray-50 hover:bg-white"
                required
              />
              {formik.touched.excerpt && formik.errors.excerpt && (
                <p className="mt-2 text-sm text-red-600 font-medium">{formik.errors.excerpt}</p>
              )}
            </div>

            {/* Content */}
            <div>
              <label htmlFor="content" className="block text-sm font-semibold text-gray-800 mb-2">
                Content
              </label>
              <RichTextEditor
                value={formik.values.content}
                onChange={(html) => formik.setFieldValue("content", html)}
              />
              {formik.touched.content && formik.errors.content && (
                <p className="mt-2 text-sm text-red-600 font-medium">{formik.errors.content}</p>
              )}
            </div>

            {/* Category and Theme Row */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Category */}
              <SearchableDropdown
                label="Category"
                placeholder="Select a category"
                value={formik.values.category}
                onChange={(value) => {
                  formik.setFieldValue("category", value);
                  formik.setFieldTouched("category", true);
                }}
                options={NEWS_CATEGORIES.map((cat) => ({
                  label: cat,
                  value: cat,
                }))}
                required
                error={formik.touched.category && formik.errors.category ? formik.errors.category : ""}
              />

              {/* Author - Toggle between system user and external */}
              <div>
                <div className="flex items-center justify-between mb-2">
                  <label className="block text-sm font-semibold text-gray-800">
                    Author
                  </label>
                  <button
                    type="button"
                    onClick={() => {
                      setUseExternalAuthor(!useExternalAuthor);
                      formik.setFieldValue("author", "");
                      formik.setFieldValue("authorId", "");
                    }}
                    className="text-xs text-blue-600 hover:text-blue-700 font-medium"
                  >
                    {useExternalAuthor ? "Use System User" : "Use External Author"}
                  </button>
                </div>

                {useExternalAuthor ? (
                  // External author text input
                  <InputField
                    placeholder="Enter author name"
                    value={formik.values.author}
                    onChange={(e) => {
                      formik.setFieldValue("author", e.target.value);
                      formik.setFieldValue("authorId", ""); // No user ID for external authors
                    }}
                    onBlur={formik.handleBlur}
                    inputClassName="border-gray-300 focus:ring-blue-500 focus:border-blue-500"
                  />
                ) : (
                  // System user dropdown
                  <SearchableDropdown
                    placeholder="Select author from team"
                    value={formik.values.author}
                    onChange={(value) => {
                      // Find the selected user and set both author name and authorId
                      const selectedUser = adminUsers.find((u) => u.name === value);
                      formik.setFieldValue("author", value);
                      formik.setFieldValue("authorId", selectedUser?.id || "");
                      formik.setFieldTouched("author", true);
                    }}
                    options={adminUsers.map((user) => ({
                      label: user.name,
                      value: user.name,
                      subtitle: user.email,
                    }))}
                  />
                )}
                {formik.touched.author && formik.errors.author && (
                  <p className="mt-2 text-sm text-red-600 font-medium">{formik.errors.author}</p>
                )}
              </div>
            </div>

            {/* Theme and Read Time Row */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Theme */}
              <SearchableDropdown
                label="Theme Color"
                placeholder="Select a theme"
                value={formik.values.theme}
                onChange={(value) => {
                  formik.setFieldValue("theme", value);
                  formik.setFieldTouched("theme", true);
                }}
                options={NEWS_THEMES.map((theme) => ({
                  label: theme.charAt(0).toUpperCase() + theme.slice(1),
                  value: theme,
                  subtitle: `${theme} color theme`,
                }))}
                error={formik.touched.theme && formik.errors.theme ? formik.errors.theme : ""}
              />

              {/* Read Time */}
              <InputField
                label="Read Time (minutes)"
                name="readTime"
                type="number"
                value={formik.values.readTime}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                placeholder="Estimated read time"
                inputClassName="border-gray-300 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>

            {/* Tags */}
            <SearchableDropdown
              label="Tags"
              placeholder="Select or type tags..."
              value={formik.values.tags}
              onChange={(value) => {
                formik.setFieldValue("tags", Array.isArray(value) ? value : []);
                formik.setFieldTouched("tags", true);
              }}
              options={[
                { label: "Science", value: "Science" },
                { label: "Competition", value: "Competition" },
                { label: "Achievement", value: "Achievement" },
                { label: "Students", value: "Students" },
                { label: "Technology", value: "Technology" },
                { label: "Learning", value: "Learning" },
                { label: "Facilities", value: "Facilities" },
                { label: "Innovation", value: "Innovation" },
                { label: "Teachers", value: "Teachers" },
                { label: "International", value: "International" },
                { label: "Faculty", value: "Faculty" },
                { label: "Education", value: "Education" },
                { label: "Sports", value: "Sports" },
                { label: "Athletics", value: "Athletics" },
                { label: "Community", value: "Community" },
                { label: "Community Service", value: "Community Service" },
                { label: "Social Responsibility", value: "Social Responsibility" },
                { label: "Charity", value: "Charity" },
                { label: "Art", value: "Art" },
                { label: "Exhibition", value: "Exhibition" },
                { label: "Creativity", value: "Creativity" },
              ]}
              multiple
              clearable
              limitTags={3}
              error={
                formik.touched.tags && formik.errors.tags
                  ? typeof formik.errors.tags === "string"
                    ? formik.errors.tags
                    : undefined
                  : ""
              }
            />

            {/* Read Time */}
            <div>
              <label htmlFor="readTime" className="block text-sm font-semibold text-gray-800 mb-2">
                Estimated Read Time (minutes)
              </label>
              <input
                type="number"
                id="readTime"
                name="readTime"
                value={formik.values.readTime}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                min="1"
                max="60"
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent outline-none transition bg-gray-50 hover:bg-white"
              />
              {formik.touched.readTime && formik.errors.readTime && (
                <p className="mt-2 text-sm text-red-600 font-medium">{formik.errors.readTime}</p>
              )}
            </div>

            {/* Status */}
            <div>
              <label htmlFor="status" className="block text-sm font-semibold text-gray-800 mb-2">
                Status
              </label>
              <select
                id="status"
                name="status"
                value={formik.values.status}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent outline-none transition bg-gray-50 hover:bg-white"
              >
                <option value="draft">Draft</option>
                <option value="published">Published</option>
              </select>
              {formik.touched.status && formik.errors.status && (
                <p className="mt-2 text-sm text-red-600 font-medium">{formik.errors.status}</p>
              )}
            </div>

            {/* Submit Buttons */}
            <div className="flex gap-4 justify-end pt-6 border-t border-gray-200">
              <Button
                type="button"
                variant="outlined"
                color="primary"
                onClick={() => router.back()}
                className=""
              >
                Cancel
              </Button>
              <Button
                type="submit"
                variant="contained"
                color="primary"
                disabled={formik.isSubmitting || createArticle.isPending}
                className="px-6 py-2 "
              >
                {formik.isSubmitting || createArticle.isPending ? "Creating..." : "Create Article"}
              </Button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
