"use client";

import { useState } from "react";
import Link from "next/link";
import { useNewsArticles, useDeleteNewsArticle } from "@/hooks/useNews";
import { useDebounce } from "@/hooks/useDebounce";
import { Button, InputField, Modal, Table } from "@/components/common/ui";
import { Trash2, Edit, Plus, Loader } from "lucide-react";

interface NewsArticleRow {
  id: string;
  title: string;
  excerpt: string;
  status: "draft" | "published";
  category?: string;
  readTime?: number;
  author: string;
  createdAt: string;
}

export default function NewsPage() {
  const [page, setPage] = useState(1);
  const [search, setSearch] = useState("");
  const debouncedSearch = useDebounce(search, 700);
  const { data, isLoading, error } = useNewsArticles(page, debouncedSearch);
  const deleteArticle = useDeleteNewsArticle();
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [articleToDelete, setArticleToDelete] = useState<string | null>(null);

  const handleDelete = async (id: string) => {
    setArticleToDelete(id);
    setDeleteModalOpen(true);
  };

  const handleSearch = (value: string) => {
    setSearch(value);
    setPage(1);
  };

  const handleDeleteArticle = async () => {
    if (!articleToDelete) return;
    try {
      await deleteArticle.mutateAsync(articleToDelete);
      setDeleteModalOpen(false);
      setArticleToDelete(null);
    } catch {
      alert("Failed to delete article");
    }
  };

  const articles = data?.data?.articles || [];
  const pagination = data?.data?.pagination || { page: 1, pages: 1, total: 0 };

  const columns = [
    {
      key: "title" as const,
      label: "Title",
      sortable: true,
      render: (value: string, row: NewsArticleRow) => (
        <div>
          <p className="font-medium text-gray-900">{value}</p>
          <p className="text-sm text-gray-600 line-clamp-1">{row.excerpt}</p>
        </div>
      ),
    },
    {
      key: "category" as const,
      label: "Category",
      sortable: true,
      render: (value: string) => (
        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
          {value || "—"}
        </span>
      ),
    },
    {
      key: "readTime" as const,
      label: "Read Time",
      sortable: true,
      render: (value: number) => (
        <span className="text-sm text-gray-700 font-medium">
          {value || 3} min
        </span>
      ),
    },
    {
      key: "author" as const,
      label: "Author",
      sortable: false,
      render: (value: string) => (
        <span className="text-sm text-gray-700">{value || "—"}</span>
      ),
    },
    {
      key: "status" as const,
      label: "Status",
      sortable: true,
      render: (value: string) => (
        <span
          className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${
            value === "published"
              ? "bg-green-100 text-green-800"
              : "bg-yellow-100 text-yellow-800"
          }`}
        >
          {value}
        </span>
      ),
    },
    {
      key: "createdAt" as const,
      label: "Created",
      sortable: true,
      render: (value: string) => new Date(value).toLocaleDateString(),
    },
    {
      key: "id" as const,
      label: "Actions",
      sortable: false,
      render: (value: string, row: NewsArticleRow) => (
        <div className="flex justify-end gap-2">
          <Link
            href={`/admin/news/${row.id}`}
            className="p-2 text-blue-600 hover:bg-blue-50 rounded transition"
            title="Edit article"
          >
            <Edit className="w-5 h-5" />
          </Link>
          <button
            onClick={() => handleDelete(row.id)}
            disabled={deleteArticle.isPending}
            className="p-2 text-red-600 hover:bg-red-50 rounded transition disabled:opacity-50"
            title="Delete article"
          >
            <Trash2 className="w-5 h-5" />
          </button>
        </div>
      ),
    },
  ];

  return (
    <div className="min-h-screen  py-8 px-4 sm:px-6 lg:px-8">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8 flex justify-between items-start">
          <div>
            <h1 className="text-4xl font-bold text-gray-900">News Articles</h1>
            <p className="mt-2 text-gray-600">
              Manage all news and blog articles
            </p>
          </div>
          <Link
            href="/admin/news/create"
            className="flex items-center gap-2 px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium shadow-md hover:shadow-lg"
          >
            <Plus className="w-5 h-5" />
            New Article
          </Link>
        </div>

        {/* Search */}
        <div className="mb-6">
          <InputField
            placeholder="Search articles..."
            value={search}
            onChange={(e) => handleSearch(e.target.value)}
            icon={<span>🔍</span>}
            iconPosition="start"
          />
        </div>

        {/* Articles Table */}
        <div className="bg-white rounded-lg shadow-lg border border-gray-100">
          {isLoading ? (
            <div className="flex justify-center items-center h-64">
              <Loader className="w-8 h-8 text-blue-600 animate-spin" />
            </div>
          ) : error ? (
            <div className="p-8 text-center">
              <div className="bg-red-50 border-l-4 border-red-500 p-4 rounded-lg inline-block">
                <p className="text-red-700 font-medium">
                  Failed to load articles
                </p>
              </div>
            </div>
          ) : articles.length === 0 ? (
            <div className="p-8 text-center text-gray-500">
              <p className="mb-4">No articles found</p>
              <Link
                href="/admin/news/create"
                className="inline-block px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium"
              >
                Create your first article
              </Link>
            </div>
          ) : (
            <>
              <Table<NewsArticleRow>
                data={articles}
                columns={columns}
                variant="primary"
              />

              {/* Pagination */}
              <div className="px-6 py-4 border-t border-gray-200 flex justify-between items-center bg-gray-50">
                <p className="text-sm text-gray-600">
                  Showing {(page - 1) * 10 + 1} to{" "}
                  {Math.min(page * 10, pagination.total)} of {pagination.total}{" "}
                  articles
                </p>
                <div className="flex gap-2">
                  <button
                    onClick={() => setPage(page - 1)}
                    disabled={page === 1}
                    className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 transition"
                  >
                    Previous
                  </button>
                  <span className="px-4 py-2 text-sm font-medium text-gray-700">
                    Page {page} of {pagination.pages}
                  </span>
                  <button
                    onClick={() => setPage(page + 1)}
                    disabled={page >= pagination.pages}
                    className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 transition"
                  >
                    Next
                  </button>
                </div>
              </div>
            </>
          )}
        </div>
        <Modal
          isOpen={deleteModalOpen}
          onClose={() => setDeleteModalOpen(false)}
          title="Confirm Deletion"
        >
          <div className="py-4">
            <p>
              Are you sure you want to delete this article? This action cannot
              be undone.
            </p>
          </div>
          <div className="py-4 flex justify-end">
            <Button
              variant="outlined"
              onClick={() => setDeleteModalOpen(false)}
            >
              Cancel
            </Button>
            <Button
              variant="contained"
              color="error"
              onClick={handleDeleteArticle}
              className="ml-2"
            >
              Delete
            </Button>
          </div>
        </Modal>
      </div>
    </div>
  );
}
