'use client';

import { useState } from 'react';
import { 
  useNewsletterSubscribers,
  useUpdateSubscriberStatus,
  useDeleteSubscriber,
  useBulkDeleteSubscribers,
  useExportSubscribers,
  NewsletterSubscriber,
} from '@/hooks/useNewsletter';
import NewsletterStatsComponent from '@/components/admin/newsletter/NewsletterStats';
import { Badge, Button, InputField, Table } from '@/components/common/ui';
import { Mail, Download, Trash2, MailX, Search } from 'lucide-react';
import { toast } from 'react-toastify';

export default function NewsletterPage() {
  const [selectedIds, setSelectedIds] = useState<string[]>([]);
  const [deleteConfirm, setDeleteConfirm] = useState<string | null>(null);

  const {
    data,
    isLoading,
    error,
    filters,
    setSearch,
    setStatus,
    setPage,
  } = useNewsletterSubscribers();

  const updateStatusMutation = useUpdateSubscriberStatus();
  const deleteMutation = useDeleteSubscriber();
  const bulkDeleteMutation = useBulkDeleteSubscribers();
  const exportMutation = useExportSubscribers();

  // Handle status toggle
  const handleStatusToggle = (id: string, currentStatus: boolean) => {
    updateStatusMutation.mutate(
      { id, subscribed: !currentStatus },
      {
        onSuccess: () => {
          toast.success(`Subscriber ${!currentStatus ? 'subscribed' : 'unsubscribed'} successfully`);
        },
        onError: (error) => {
          toast.error(`Failed to update status: ${error.message}`);
        },
      }
    );
  };

  // Handle delete
  const handleDelete = (id: string) => {
    deleteMutation.mutate(id, {
      onSuccess: () => {
        toast.success('Subscriber deleted successfully');
        setDeleteConfirm(null);
        setSelectedIds(selectedIds.filter((sid) => sid !== id));
      },
      onError: (error) => {
        toast.error(`Failed to delete: ${error.message}`);
      },
    });
  };

  // Handle bulk delete
  const handleBulkDelete = () => {
    if (selectedIds.length === 0) return;

    if (window.confirm(`Delete ${selectedIds.length} subscriber(s)?`)) {
      bulkDeleteMutation.mutate(selectedIds, {
        onSuccess: () => {
          toast.success(`Deleted ${selectedIds.length} subscriber(s)`);
          setSelectedIds([]);
        },
        onError: (error) => {
          toast.error(`Failed to delete: ${error.message}`);
        },
      });
    }
  };

  // Handle export
  const handleExport = () => {
    exportMutation.mutate(
      { status: filters.status, ids: selectedIds.length > 0 ? selectedIds : undefined },
      {
        onSuccess: () => {
          toast.success('Subscribers exported successfully');
        },
        onError: (error) => {
          toast.error(`Export failed: ${error.message}`);
        },
      }
    );
  };

  if (error) {
    return (
      <div className="p-8">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4 text-red-800">
          <p className="font-medium">Error loading newsletter subscribers</p>
          <p className="text-sm mt-1">{error.message}</p>
        </div>
      </div>
    );
  }

  const columns = [
    {
      key: 'email' as const,
      label: 'Email',
      sortable: true,
    },
    {
      key: 'firstName' as const,
      label: 'Name',
      render: (value: string | null) => value || 'N/A',
    },
    {
      key: 'subscribed' as const,
      label: 'Status',
      render: (value: boolean) =>
        value ? (
          <Badge variant="success">Subscribed</Badge>
        ) : (
          <Badge variant="error">Unsubscribed</Badge>
        ),
    },
    {
      key: 'subscribedAt' as const,
      label: 'Subscribed Date',
      render: (value: Date) => new Date(value).toLocaleDateString(),
    },
    {
      key: 'id' as const,
      label: 'Actions',
      render: (_: string, row: NewsletterSubscriber) => (
        <div className="flex items-center gap-2">
          {/* Toggle Subscribe/Unsubscribe */}
          <button
            onClick={() => handleStatusToggle(row.id, row.subscribed)}
            disabled={updateStatusMutation.isPending}
            className={`p-2 rounded-lg transition-colors ${
              row.subscribed
                ? 'text-orange-600 hover:bg-orange-50'
                : 'text-green-600 hover:bg-green-50'
            } disabled:opacity-50`}
            title={row.subscribed ? 'Unsubscribe' : 'Re-subscribe'}
          >
            {row.subscribed ? <MailX className="h-4 w-4" /> : <Mail className="h-4 w-4" />}
          </button>

          {/* Delete */}
          {deleteConfirm === row.id ? (
            <div className="flex gap-2">
              <button
                onClick={() => handleDelete(row.id)}
                disabled={deleteMutation.isPending}
                className="px-2 py-1 text-xs bg-red-600 text-white rounded hover:bg-red-700"
              >
                Confirm
              </button>
              <button
                onClick={() => setDeleteConfirm(null)}
                className="px-2 py-1 text-xs bg-gray-200 rounded hover:bg-gray-300"
              >
                Cancel
              </button>
            </div>
          ) : (
            <button
              onClick={() => setDeleteConfirm(row.id)}
              className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
              title="Delete"
            >
              <Trash2 className="h-4 w-4" />
            </button>
          )}
        </div>
      ),
    },
  ];

  return (
    <div className="p-8">
      {/* Header */}
      <div className="mb-8">
        <div className="flex items-center gap-3 mb-2">
          <div className="p-3 bg-gradient-to-br from-purple-500 to-blue-600 rounded-lg">
            <Mail className="h-8 w-8 text-white" />
          </div>
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Newsletter Management</h1>
            <p className="text-gray-600 mt-1">Manage and track newsletter subscribers</p>
          </div>
        </div>
      </div>

      {/* Stats */}
      <NewsletterStatsComponent
        stats={data?.stats || { total: 0, active: 0, unsubscribed: 0 }}
        isLoading={isLoading}
      />

      {/* Toolbar */}
      <div className="bg-white rounded-lg shadow p-4 mb-4">
        <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-center justify-between">
          {/* Search */}
          <div className="relative flex-1 max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
            <InputField
              type="text"
              placeholder="Search by email or name..."
              value={filters.search || ''}
              onChange={(e) => setSearch(e.target.value)}
              className="w-full pl-10 pr-4 py-2"
            />
          </div>

          {/* Actions */}
          <div className="flex gap-3">
            {/* Status Filter */}
            <select
              value={filters.status || 'all'}
              onChange={(e) => setStatus(e.target.value as 'all' | 'subscribed' | 'unsubscribed')}
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
            >
              <option value="all">All Status</option>
              <option value="subscribed">Subscribed</option>
              <option value="unsubscribed">Unsubscribed</option>
            </select>

            {/* Export */}
            <Button
              onClick={handleExport}
              disabled={exportMutation.isPending}
              loading={exportMutation.isPending}
              variant="contained"
              color='primary'
              className=""
            >
              <Download className="h-4 w-4" />
              Export CSV
            </Button>

            {/* Bulk Delete */}
            {selectedIds.length > 0 && (
              <button
                onClick={handleBulkDelete}
                disabled={bulkDeleteMutation.isPending}
                className="flex items-center gap-2 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
              >
                <Trash2 className="h-4 w-4" />
                Delete ({selectedIds.length})
              </button>
            )}
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="bg-white rounded-lg shadow overflow-hidden">
        {isLoading ? (
          <div className="p-12 text-center text-gray-500">Loading subscribers...</div>
        ) : (
          <Table<NewsletterSubscriber>
            data={data?.subscribers || []}
            columns={columns}
            variant="default"
          />
        )}
        
        {/* Pagination */}
        {data && data.pagination.totalPages > 1 && (
          <div className="px-6 py-4 border-t flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Showing {(data.pagination.page - 1) * data.pagination.limit + 1} to{' '}
              {Math.min(data.pagination.page * data.pagination.limit, data.pagination.total)} of{' '}
              {data.pagination.total} subscribers
            </div>
            <div className="flex gap-2">
              <button
                onClick={() => setPage(data.pagination.page - 1)}
                disabled={data.pagination.page === 1}
                className="px-3 py-1 border rounded-lg hover:bg-gray-50 disabled:opacity-50"
              >
                Previous
              </button>
              <button
                onClick={() => setPage(data.pagination.page + 1)}
                disabled={data.pagination.page === data.pagination.totalPages}
                className="px-3 py-1 border rounded-lg hover:bg-gray-50 disabled:opacity-50"
              >
                Next
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
