"use client";

import { useState } from "react";
import { useUserProfile } from "@/hooks/useUserProfile";
import { Button, InputField } from "@/components/common/ui";
import { ImageUploadField } from "@/components/common/ui/ImageUploadField";
import { useFormik } from "formik";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import * as Yup from "yup";

// Validation schemas
const profileSchema = Yup.object().shape({
  name: Yup.string()
    .required("Name is required")
    .min(2, "Name must be at least 2 characters"),
  email: Yup.string()
    .required("Email is required")
    .email("Invalid email address"),
  phone: Yup.string().optional().min(10, "Phone must be at least 10 digits"),
  department: Yup.string().optional(),
  bio: Yup.string().optional().max(500, "Bio must not exceed 500 characters"),
  avatar: Yup.string().optional(),
});

const passwordSchema = Yup.object().shape({
  currentPassword: Yup.string().required("Current password is required"),
  newPassword: Yup.string()
    .required("New password is required")
    .min(8, "Password must be at least 8 characters")
    .matches(/[A-Z]/, "Password must contain at least one uppercase letter")
    .matches(/[0-9]/, "Password must contain at least one number"),
  confirmPassword: Yup.string()
    .required("Please confirm your password")
    .oneOf([Yup.ref("newPassword")], "Passwords must match"),
});

// API mutation functions
const updateProfile = async (data: {
  name: string;
  email: string;
  phone?: string;
  department?: string;
  bio?: string;
  avatar?: string;
}) => {
  const response = await fetch("/api/admin/profile", {
    method: "PUT",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(data),
  });
  const result = await response.json();
  if (!response.ok)
    throw new Error(result.message || "Failed to update profile");
  return result;
};

const updatePassword = async (data: {
  currentPassword: string;
  newPassword: string;
}) => {
  const response = await fetch("/api/admin/password", {
    method: "PUT",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(data),
  });
  const result = await response.json();
  if (!response.ok)
    throw new Error(result.message || "Failed to change password");
  return result;
};

export default function SettingsPage() {
  const [successMessage, setSuccessMessage] = useState("");
  const queryClient = useQueryClient();

  // Fetch user profile with React Query
  const {
    data: profile,
    isLoading,
    error: profileError,
    refetch: refetchProfile,
  } = useUserProfile();

  // Profile update mutation with auto-refetch
  const profileMutation = useMutation({
    mutationFn: updateProfile,
    onSuccess: async () => {
      setSuccessMessage("Profile updated successfully!");
      // Invalidate and refetch profile data
      await queryClient.invalidateQueries({ queryKey: ["user-profile"] });
      await refetchProfile();
      setTimeout(() => {
        setSuccessMessage("");
      }, 2000);
    },
  });

  // Password update mutation
  const passwordMutation = useMutation({
    mutationFn: updatePassword,
    onSuccess: () => {
      setSuccessMessage("Password changed successfully!");
      passwordFormik.resetForm();
      setTimeout(() => setSuccessMessage(""), 2000);
    },
  });

  // Profile form
  const profileFormik = useFormik({
    initialValues: {
      name: profile?.name || "",
      email: profile?.email || "",
      phone: profile?.phone || "",
      department: profile?.department || "",
      bio: profile?.bio || "",
      avatar: profile?.avatar || "",
    },
    validationSchema: profileSchema,
    onSubmit: (values) => profileMutation.mutate(values),
    enableReinitialize: true, // Re-initialize form when profile data loads
  });

  // Password form
  const passwordFormik = useFormik({
    initialValues: {
      currentPassword: "",
      newPassword: "",
      confirmPassword: "",
    },
    validationSchema: passwordSchema,
    onSubmit: (values) => {
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      const { confirmPassword, ...data } = values;
      passwordMutation.mutate(data);
    },
  });

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 py-12 px-4 sm:px-6 lg:px-8 flex items-center justify-center">
        <div className="text-center">
          <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
          <p className="mt-4 text-gray-600">Loading your profile...</p>
        </div>
      </div>
    );
  }

  if (profileError) {
    return (
      <div className="min-h-screen bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
        <div className="max-w-5xl mx-auto">
          <div className="p-6 bg-red-50 border border-red-200 rounded-lg">
            <h1 className="text-2xl font-bold text-red-900 mb-2">
              Error Loading Profile
            </h1>
            <p className="text-red-800">{profileError.message}</p>
            <button
              onClick={() => refetchProfile()}
              className="mt-4 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
            >
              Try Again
            </button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
      <div className="max-w-5xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Settings</h1>
          <p className="mt-2 text-gray-600">
            Manage your account and preferences
          </p>
        </div>

        {/* Success Message */}
        {successMessage && (
          <div className="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg">
            <p className="text-green-800 font-medium">✓ {successMessage}</p>
          </div>
        )}

        <div className="grid grid-cols-1 gap-6 lg:grid-cols-3">
          {/* Profile Picture & Bio */}
          <div className="bg-white rounded-lg shadow p-6">
            <h2 className="text-xl font-semibold text-gray-900 mb-6">
              Profile Picture
            </h2>

            <form onSubmit={profileFormik.handleSubmit} className="space-y-4">
              {/* Profile Image Upload */}
              <ImageUploadField
                label="Profile Picture"
                value={profileFormik.values.avatar}
                onChange={(url) => profileFormik.setFieldValue("avatar", url)}
                onError={(error) => console.error(error)}
                oldImage={profile?.avatar}
              />

              {/* Bio */}
              <div>
                <label
                  htmlFor="bio"
                  className="block text-sm font-medium text-gray-900 mb-2"
                >
                  Bio
                </label>
                <textarea
                  id="bio"
                  name="bio"
                  value={profileFormik.values.bio}
                  onChange={profileFormik.handleChange}
                  onBlur={profileFormik.handleBlur}
                  placeholder="Tell us about yourself..."
                  rows={4}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
                {profileFormik.touched.bio && profileFormik.errors.bio && (
                  <p className="text-sm text-red-600 mt-1">
                    {profileFormik.errors.bio}
                  </p>
                )}
                <p className="text-xs text-gray-500 mt-1">
                  {profileFormik.values.bio.length}/500 characters
                </p>
              </div>

              {/* Save Button */}
              <Button
                type="submit"
                disabled={profileMutation.isPending}
                variant="contained"
                color="primary"
                fullWidth
              >
                {profileMutation.isPending ? "Saving..." : "Save Profile"}
              </Button>

              {profileMutation.isError && (
                <div className="p-3 bg-red-50 border border-red-200 rounded-lg">
                  <p className="text-red-800 text-sm">
                    {profileMutation.error instanceof Error
                      ? profileMutation.error.message
                      : "Failed to update profile"}
                  </p>
                </div>
              )}
            </form>
          </div>

          {/* Basic Information */}
          <div className="bg-white rounded-lg shadow p-6 lg:col-span-2">
            <h2 className="text-xl font-semibold text-gray-900 mb-6">
              Basic Information
            </h2>

            <form onSubmit={profileFormik.handleSubmit} className="space-y-4">
              {/* Current User Info */}
              <div className="p-4 bg-gray-50 rounded-lg mb-6">
                <p className="text-sm text-gray-600">Current Role</p>
                <p className="font-medium text-gray-900 capitalize">
                  {profile?.role || "Admin"}
                </p>
              </div>

              {/* Name */}
              <InputField
                id="name"
                name="name"
                type="text"
                label="Full Name *"
                value={profileFormik.values.name}
                onChange={profileFormik.handleChange}
                onBlur={profileFormik.handleBlur}
                placeholder="Enter your full name"
                error={
                  profileFormik.touched.name && profileFormik.errors.name
                    ? profileFormik.errors.name
                    : ""
                }
              />

              {/* Email */}
              <InputField
                id="email"
                name="email"
                type="email"
                label="Email Address *"
                value={profileFormik.values.email}
                onChange={profileFormik.handleChange}
                onBlur={profileFormik.handleBlur}
                placeholder="Enter your email"
                error={
                  profileFormik.touched.email && profileFormik.errors.email
                    ? profileFormik.errors.email
                    : ""
                }
              />

              {/* Phone */}
              <InputField
                id="phone"
                name="phone"
                type="tel"
                label="Phone Number"
                value={profileFormik.values.phone}
                onChange={profileFormik.handleChange}
                onBlur={profileFormik.handleBlur}
                placeholder="Enter your phone number"
                error={
                  profileFormik.touched.phone && profileFormik.errors.phone
                    ? profileFormik.errors.phone
                    : ""
                }
              />

              {/* Department */}
              <InputField
                id="department"
                name="department"
                type="text"
                label="Department / Role"
                value={profileFormik.values.department}
                onChange={profileFormik.handleChange}
                onBlur={profileFormik.handleBlur}
                placeholder="e.g., Academics, Administration"
                error={
                  profileFormik.touched.department &&
                  profileFormik.errors.department
                    ? profileFormik.errors.department
                    : ""
                }
              />

              {/* Profile Mutation Error */}
              {profileMutation.isError && (
                <div className="p-3 bg-red-50 border border-red-200 rounded-lg">
                  <p className="text-red-800 text-sm">
                    {profileMutation.error instanceof Error
                      ? profileMutation.error.message
                      : "Failed to update profile"}
                  </p>
                </div>
              )}

              {/* Submit Button */}
              <div className="pt-4">
                <Button
                  type="submit"
                  disabled={profileMutation.isPending || !profileFormik.isValid}
                  variant="contained"
                  color="primary"
                  fullWidth
                >
                  {profileMutation.isPending ? "Updating..." : "Update Profile"}
                </Button>
              </div>
            </form>
          </div>
        </div>

        {/* Change Password */}
        <div className="mt-6 bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-semibold text-gray-900 mb-6">
            Change Password
          </h2>

          <form
            onSubmit={passwordFormik.handleSubmit}
            className="grid grid-cols-1 gap-6 lg:grid-cols-3"
          >
            {/* Current Password */}
            <InputField
              id="currentPassword"
              name="currentPassword"
              type="password"
              label="Current Password"
              value={passwordFormik.values.currentPassword}
              onChange={passwordFormik.handleChange}
              onBlur={passwordFormik.handleBlur}
              placeholder="Enter current password"
              showPasswordToggle={true}
              error={
                passwordFormik.touched.currentPassword &&
                passwordFormik.errors.currentPassword
                  ? passwordFormik.errors.currentPassword
                  : ""
              }
            />

            {/* New Password */}
            <InputField
              id="newPassword"
              name="newPassword"
              type="password"
              label="New Password"
              value={passwordFormik.values.newPassword}
              onChange={passwordFormik.handleChange}
              onBlur={passwordFormik.handleBlur}
              placeholder="Enter new password"
              showPasswordToggle={true}
              helperText="Min 8 chars, 1 uppercase, 1 number"
              error={
                passwordFormik.touched.newPassword &&
                passwordFormik.errors.newPassword
                  ? passwordFormik.errors.newPassword
                  : ""
              }
            />

            {/* Confirm Password */}
            <InputField
              id="confirmPassword"
              name="confirmPassword"
              type="password"
              label="Confirm Password"
              value={passwordFormik.values.confirmPassword}
              onChange={passwordFormik.handleChange}
              onBlur={passwordFormik.handleBlur}
              placeholder="Confirm new password"
              showPasswordToggle={true}
              error={
                passwordFormik.touched.confirmPassword &&
                passwordFormik.errors.confirmPassword
                  ? passwordFormik.errors.confirmPassword
                  : ""
              }
            />

            {/* Password Mutation Error */}
            {passwordMutation.isError && (
              <div className="lg:col-span-3 p-3 bg-red-50 border border-red-200 rounded-lg">
                <p className="text-red-800 text-sm">
                  {passwordMutation.error instanceof Error
                    ? passwordMutation.error.message
                    : "Failed to change password"}
                </p>
              </div>
            )}

            {/* Submit Button */}
            <div className="lg:col-span-3">
              <Button
                type="submit"
                disabled={passwordMutation.isPending || !passwordFormik.isValid}
                variant="contained"
                color="primary"
              >
                {passwordMutation.isPending ? "Updating..." : "Change Password"}
              </Button>
            </div>
          </form>
        </div>

        {/* Account Summary */}
        <div className="mt-6 bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">
            Account Summary
          </h2>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <div>
              <p className="text-sm text-gray-600">Email</p>
              <p className="font-medium text-gray-900 break-all">
                {profile?.email}
              </p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Role</p>
              <p className="font-medium text-gray-900 capitalize">
                {profile?.role || "Admin"}
              </p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Department</p>
              <p className="font-medium text-gray-900">
                {profile?.department || "—"}
              </p>
            </div>
            <div>
              <p className="text-sm text-gray-600">Status</p>
              <p className="font-medium text-green-600">Active</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
