"use client";

import { useState } from "react";
import Link from "next/link";
import { useStudents, useDeleteStudent, Student } from "@/hooks";
import { useStudentInNeed, useDeleteStudentInNeed } from "@/hooks/useStudentInNeed";
import { StudentInNeed } from "@/types/types";
import Table from "@/components/common/ui/table";
import TablePagination from "@/components/common/ui/table-pagination";
import { Plus, Trash2, Edit, Search } from "lucide-react";
import { Avatar, Button, Modal } from "@/components/common/ui";
import Card from "@/components/common/ui/card";

export default function StudentsPage() {
  const { data: students = [], isLoading, error, refetch } = useStudents();
  const deleteStudent = useDeleteStudent();
  const [searchTerm, setSearchTerm] = useState("");
  const [categoryFilter, setCategoryFilter] = useState<string>("");
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 10;

  // Tab state
  const [activeTab, setActiveTab] = useState<"stories" | "in-need">("stories");

  // Delete modal state
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [deleteStudentId, setDeleteStudentId] = useState<string | null>(null);
  const [isDeleting, setIsDeleting] = useState(false);

  // Students in Need state
  const { data: studentInNeed = [], isLoading: inNeedLoading } = useStudentInNeed();
  const deleteStudentInNeed = useDeleteStudentInNeed();
  const [inNeedSearchTerm, setInNeedSearchTerm] = useState("");
  const [inNeedCurrentPage, setInNeedCurrentPage] = useState(1);
  const inNeedItemsPerPage = 10;

  // In-Need delete modal state
  const [deleteInNeedModalOpen, setDeleteInNeedModalOpen] = useState(false);
  const [deleteInNeedId, setDeleteInNeedId] = useState<string | null>(null);
  const [isDeletingInNeed, setIsDeletingInNeed] = useState(false);

  const filteredStudents = students.filter((student) => {
    const matchesSearch =
      student.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      student.grade.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesCategory =
      categoryFilter === "" ||
      categoryFilter === "all" ||
      student.category === categoryFilter;

    return matchesSearch && matchesCategory;
  });

  // Pagination logic
  const totalPages = Math.ceil(filteredStudents.length / itemsPerPage);
  const paginatedStudents = filteredStudents.slice(
    (currentPage - 1) * itemsPerPage,
    currentPage * itemsPerPage
  );

  // Reset to page 1 when filters change
  const handleSearch = (term: string) => {
    setSearchTerm(term);
    setCurrentPage(1);
  };

  const handleCategoryFilter = (category: string) => {
    setCategoryFilter(category);
    setCurrentPage(1);
  };

  const handleDelete = (id: string) => {
    setDeleteStudentId(id);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!deleteStudentId) return;
    setIsDeleting(true);
    try {
      await deleteStudent.mutateAsync(deleteStudentId);
      setDeleteModalOpen(false);
      setDeleteStudentId(null);
      refetch();
    } catch (error) {
      console.error("Delete failed:", error);
    } finally {
      setIsDeleting(false);
    }
  };

  const handleDeleteInNeed = (id: string) => {
    setDeleteInNeedId(id);
    setDeleteInNeedModalOpen(true);
  };

  const confirmDeleteInNeed = async () => {
    if (!deleteInNeedId) return;
    setIsDeletingInNeed(true);
    try {
      await deleteStudentInNeed.mutateAsync(deleteInNeedId);
      setDeleteInNeedModalOpen(false);
      setDeleteInNeedId(null);
    } catch (error) {
      console.error("Delete failed:", error);
    } finally {
      setIsDeletingInNeed(false);
    }
  };

  // Stats
  const totalStudents = students.length;
  const featuredStudents = students.filter((s) => s.featured).length;
  const academicStories = students.filter((s) => s.category === "academic").length;

  const stats = [
    { label: "Total Stories", value: totalStudents },
    { label: "Featured", value: featuredStudents },
    { label: "Academic", value: academicStories },
  ];

  // Students in Need data and filtering
  const filteredInNeed = studentInNeed.filter((student: StudentInNeed) =>
    student.name.toLowerCase().includes(inNeedSearchTerm.toLowerCase())
  );
  const totalInNeedPages = Math.ceil(filteredInNeed.length / inNeedItemsPerPage);
  const inNeedStartIdx = (inNeedCurrentPage - 1) * inNeedItemsPerPage;
  const paginatedInNeed = filteredInNeed.slice(inNeedStartIdx, inNeedStartIdx + inNeedItemsPerPage);

  const inNeedStats = [
    { label: "Total Students", value: studentInNeed.length },
    { label: "Featured", value: studentInNeed.filter((s: StudentInNeed) => s.featured).length },
    {
      label: "Education Need",
      value: studentInNeed.filter((s: StudentInNeed) => s.needCategory === "education").length,
    },
    {
      label: "Other Needs",
      value: studentInNeed.filter((s: StudentInNeed) => s.needCategory !== "education").length,
    },
  ];

  if (error) {
    return (
      <div className="p-6 bg-red-50 text-red-700 rounded-lg">
        Error loading students:{" "}
        {error instanceof Error ? error.message : "Unknown error"}
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold text-gray-900">
          {activeTab === "stories" ? "Student Stories" : "Students in Need"} Management
        </h1>
        <Link href={activeTab === "stories" ? "/admin/students/create" : "/admin/students/in-need/create"}>
          <Button variant="contained" color="primary">
            <Plus size={20} />
            {activeTab === "stories" ? "Add Student Story" : "Add Student in Need"}
          </Button>
        </Link>
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <div className="flex gap-8">
          <button
            onClick={() => setActiveTab("stories")}
            className={`py-3 px-1 font-medium transition border-b-2 ${
              activeTab === "stories"
                ? "text-primary-600 border-primary-600"
                : "text-gray-600 border-transparent hover:text-gray-900"
            }`}
          >
            Student Stories
          </button>
          <button
            onClick={() => setActiveTab("in-need")}
            className={`py-3 px-1 font-medium transition border-b-2 ${
              activeTab === "in-need"
                ? "text-primary-600 border-primary-600"
                : "text-gray-600 border-transparent hover:text-gray-900"
            }`}
          >
            Students in Need
          </button>
        </div>
      </div>

      {/* Tab Content */}
      {activeTab === "stories" ? (
        <>
          {/* STUDENT STORIES TAB */}
          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            {stats.map((stat) => (
              <div
                key={stat.label}
                className="bg-white p-4 rounded-lg shadow-sm border-l-4 border-primary-600"
              >
                <p className="text-sm text-gray-600">{stat.label}</p>
                <p className="text-3xl font-bold text-gray-900 mt-1">
                  {stat.value}
                </p>
              </div>
            ))}
          </div>

      {/* Filters */}
      <div className="space-y-4">
            <div>
              <input
                type="text"
                placeholder="Search by student name or grade..."
                value={searchTerm}
                onChange={(e) => handleSearch(e.target.value)}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
              />
            </div>

            {/* Category Filter */}
            <div className="flex gap-2 flex-wrap">
              <button
                onClick={() => handleCategoryFilter("")}
                className={`px-4 py-2 rounded-lg transition ${
                  categoryFilter === ""
                    ? "bg-primary-600 text-white"
                    : "bg-gray-200 text-gray-700 hover:bg-gray-300"
                }`}
              >
                All Categories
              </button>
              <button
                onClick={() => handleCategoryFilter("academic")}
                className={`px-4 py-2 rounded-lg transition ${
                  categoryFilter === "academic"
                    ? "bg-primary-600 text-white"
                    : "bg-gray-200 text-gray-700 hover:bg-gray-300"
                }`}
              >
                Academic
              </button>
              <button
                onClick={() => handleCategoryFilter("sports")}
                className={`px-4 py-2 rounded-lg transition ${
                  categoryFilter === "sports"
                    ? "bg-primary-600 text-white"
                    : "bg-gray-200 text-gray-700 hover:bg-gray-300"
                }`}
              >
                Sports
              </button>
              <button
                onClick={() => handleCategoryFilter("arts")}
                className={`px-4 py-2 rounded-lg transition ${
                  categoryFilter === "arts"
                    ? "bg-primary-600 text-white"
                    : "bg-gray-200 text-gray-700 hover:bg-gray-300"
                }`}
              >
                Arts
              </button>
              <button
                onClick={() => handleCategoryFilter("other")}
                className={`px-4 py-2 rounded-lg transition ${
                  categoryFilter === "other"
                    ? "bg-primary-600 text-white"
                    : "bg-gray-200 text-gray-700 hover:bg-gray-300"
                }`}
              >
                Other
              </button>
            </div>
          </div>

          {/* Table */}
          {isLoading ? (
            <div className="p-8 text-center text-gray-500">Loading students...</div>
          ) : filteredStudents.length === 0 ? (
            <div className="p-8 text-center text-gray-500 bg-white rounded-lg">
              {students.length === 0
                ? "No student stories yet. Add your first story!"
                : "No stories match your search or filter."}
            </div>
          ) : (
            <div className="space-y-4">
              <Table
                data={paginatedStudents}
                columns={[
                  {
                    key: "name",
                    label: "Student Name",
                    render: (value: string, row: Student) => (
                      <div className="flex items-center gap-3">
                        <Avatar
                          src={
                            Array.isArray(row.images) && row.images.length > 0
                              ? row.images[0]
                              : undefined
                          }
                          name={row.name}
                          size="sm"
                          className="rounded"
                        />
                        <div>
                          <p className="font-medium text-gray-900">{row.name}</p>
                          <p className="text-xs text-gray-500">{row.grade}</p>
                        </div>
                      </div>
                    ),
                  },
                  {
                    key: "category",
                    label: "Category",
                    render: (value: string) => {
                      const categoryStyles: Record<string, string> = {
                        academic: "bg-blue-100 text-blue-800",
                        sports: "bg-green-100 text-green-800",
                        arts: "bg-purple-100 text-purple-800",
                        other: "bg-gray-100 text-gray-800",
                      };
                      return (
                        <span
                          className={`inline-block px-3 py-1 rounded-full text-xs font-medium ${
                            categoryStyles[value || "other"] || "bg-gray-100 text-gray-800"
                          }`}
                        >
                          {value ? value.charAt(0).toUpperCase() + value.slice(1) : "Other"}
                        </span>
                      );
                    },
                  },
                  {
                    key: "featured",
                    label: "Featured",
                    render: (value: boolean) => (
                      <span className="inline-flex items-center gap-2">
                        {value ? (
                          <>
                            <span className="inline-block w-2 h-2 bg-green-500 rounded-full"></span>
                            <span className="text-sm text-green-600">Yes</span>
                          </>
                        ) : (
                          <>
                            <span className="inline-block w-2 h-2 bg-gray-300 rounded-full"></span>
                            <span className="text-sm text-gray-600">No</span>
                          </>
                        )}
                      </span>
                    ),
                  },
                  {
                    key: "supportNeeded",
                    label: "Support",
                    render: (value: string) => (
                      <span className="text-sm text-gray-600">
                        {value ? value.substring(0, 20) + "..." : "-"}
                      </span>
                    ),
                  },
                  {
                    key: "id",
                    label: "Actions",
                    render: (_, row: Student) => (
                      <div className="flex items-center gap-2">
                        <Link href={`/admin/students/${row.id}/edit`}>
                          <button
                            className="p-2 hover:bg-blue-100 text-blue-600 rounded-lg transition"
                            title="Edit"
                          >
                            <Edit size={18} />
                          </button>
                        </Link>
                        <button
                          onClick={() => handleDelete(row.id)}
                          className="p-2 hover:bg-red-100 text-red-600 rounded-lg transition"
                          title="Delete"
                        >
                          <Trash2 size={18} />
                        </button>
                      </div>
                    ),
                  },
                ]}
              />

              {/* Pagination */}
              {totalPages > 1 && (
                <TablePagination
                  currentPage={currentPage}
                  totalPages={totalPages}
                  onPageChange={setCurrentPage}
                  itemsPerPage={itemsPerPage}
                  totalItems={filteredStudents.length}
                />
              )}
            </div>
          )}
        </>
      ) : (
        <>
          {/* STUDENTS IN NEED TAB */}
          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            {inNeedStats.map((stat) => (
              <div
                key={stat.label}
                className="bg-white p-4 rounded-lg shadow-sm border-l-4 border-primary-600"
              >
                <p className="text-sm text-gray-600">{stat.label}</p>
                <p className="text-3xl font-bold text-gray-900 mt-1">
                  {stat.value}
                </p>
              </div>
            ))}
          </div>

          {/* Search */}
          <Card className="p-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder="Search by name..."
                value={inNeedSearchTerm}
                onChange={(e) => {
                  setInNeedSearchTerm(e.target.value);
                  setInNeedCurrentPage(1);
                }}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500"
              />
            </div>
          </Card>

          {/* Table */}
          {inNeedLoading ? (
            <div className="p-8 text-center text-gray-500">Loading students...</div>
          ) : filteredInNeed.length === 0 ? (
            <div className="p-8 text-center text-gray-500 bg-white rounded-lg">
              {studentInNeed.length === 0
                ? "No students in need yet. Add your first student!"
                : "No students match your search."}
            </div>
          ) : (
            <div className="space-y-4">
              <Card className="overflow-hidden">
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead className="bg-gray-50 border-b border-gray-200">
                      <tr>
                        <th className="px-6 py-3 text-left text-sm font-semibold text-gray-900">
                          Name
                        </th>
                        <th className="px-6 py-3 text-left text-sm font-semibold text-gray-900">
                          Need
                        </th>
                        <th className="px-6 py-3 text-left text-sm font-semibold text-gray-900">
                          Grade
                        </th>
                        <th className="px-6 py-3 text-left text-sm font-semibold text-gray-900">
                          Featured
                        </th>
                        <th className="px-6 py-3 text-left text-sm font-semibold text-gray-900">
                          Actions
                        </th>
                      </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200">
                      {paginatedInNeed.map((student: StudentInNeed) => (
                        <tr key={student.id} className="hover:bg-gray-50">
                          <td className="px-6 py-4">
                            <p className="font-medium text-gray-900">{student.name}</p>
                          </td>
                          <td className="px-6 py-4">
                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 capitalize">
                              {student.needCategory}
                            </span>
                          </td>
                          <td className="px-6 py-4 text-sm text-gray-600">
                            {student.grade || "N/A"}
                          </td>
                          <td className="px-6 py-4">
                            {student.featured ? (
                              <span className="text-green-600 font-medium">✓ Yes</span>
                            ) : (
                              <span className="text-gray-400">-</span>
                            )}
                          </td>
                          <td className="px-6 py-4">
                            <div className="flex gap-2">
                              <Link href={`/admin/students/in-need/${student.id}/edit`}>
                                <button className="p-2 hover:bg-blue-50 rounded-lg text-blue-600">
                                  <Edit className="w-4 h-4" />asd
                                </button>
                              </Link>
                              <button
                                onClick={() => handleDeleteInNeed(student.id)}
                                className="p-2 hover:bg-red-50 rounded-lg text-red-600"
                              >
                                <Trash2 className="w-4 h-4" />
                              </button>
                            </div>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </Card>

              {/* Pagination */}
              {totalInNeedPages > 1 && (
                <TablePagination
                  currentPage={inNeedCurrentPage}
                  totalPages={totalInNeedPages}
                  itemsPerPage={inNeedItemsPerPage}
                  totalItems={filteredInNeed.length}
                  onPageChange={setInNeedCurrentPage}
                />
              )}
            </div>
          )}
        </>
      )}

      {/* Delete Student Story Modal */}
      <Modal
        isOpen={deleteModalOpen}
        onClose={() => {
          setDeleteModalOpen(false);
          setDeleteStudentId(null);
        }}
        title="Delete Student Story"
      >
        <div className="space-y-4">
          <p className="text-gray-600">
            Are you sure you want to delete this student story? This action cannot be undone.
          </p>
          <div className="flex gap-3 justify-end">
            <Button
              variant="outlined"
              onClick={() => {
                setDeleteModalOpen(false);
                setDeleteStudentId(null);
              }}
            >
              Cancel
            </Button>
            <Button
              variant="contained"
              onClick={confirmDelete}
              disabled={isDeleting}
            >
              {isDeleting ? "Deleting..." : "Delete"}
            </Button>
          </div>
        </div>
      </Modal>

      {/* Delete Student in Need Modal */}
      <Modal
        isOpen={deleteInNeedModalOpen}
        onClose={() => {
          setDeleteInNeedModalOpen(false);
          setDeleteInNeedId(null);
        }}
        title="Delete Student in Need"
      >
        <div className="space-y-4">
          <p className="text-gray-600">
            Are you sure you want to delete this student? This action cannot be undone.
          </p>
          <div className="flex gap-3 justify-end">
            <Button
              variant="outlined"
              onClick={() => {
                setDeleteInNeedModalOpen(false);
                setDeleteInNeedId(null);
              }}
            >
              Cancel
            </Button>
            <Button
              variant="contained"
              onClick={confirmDeleteInNeed}
              disabled={isDeletingInNeed}
            >
              {isDeletingInNeed ? "Deleting..." : "Delete"}
            </Button>
          </div>
        </div>
      </Modal>
    </div>
  );
}
