"use client";

import { useState } from "react";
import { useTeachers, useDeleteTeacher, Teacher } from "@/hooks";
import TeacherForm from "@/components/admin/forms/TeacherForm";
import Table from "@/components/common/ui/table";
import TablePagination from "@/components/common/ui/table-pagination";
import { Plus, Trash2, Edit, Eye } from "lucide-react";
import Link from "next/link";
import { Avatar } from "@/components/common/ui";

export default function TeachersPage() {
  const { data: teachers = [], isLoading, error, refetch } = useTeachers();
  const deleteTeacher = useDeleteTeacher();
  const [showForm, setShowForm] = useState(false);
  const [selectedTeacher, setSelectedTeacher] = useState<Teacher | undefined>();
  const [searchTerm, setSearchTerm] = useState("");
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 10;

  const filteredTeachers = teachers.filter(
    (teacher) =>
      teacher.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      teacher.position.toLowerCase().includes(searchTerm.toLowerCase()) ||
      teacher.email?.toLowerCase().includes(searchTerm.toLowerCase())
  );

  // Pagination logic
  const totalPages = Math.ceil(filteredTeachers.length / itemsPerPage);
  const paginatedTeachers = filteredTeachers.slice(
    (currentPage - 1) * itemsPerPage,
    currentPage * itemsPerPage
  );

  // Reset to page 1 when search term changes
  const handleSearch = (term: string) => {
    setSearchTerm(term);
    setCurrentPage(1);
  };

  const handleEdit = (teacher: Teacher) => {
    setSelectedTeacher(teacher);
    setShowForm(true);
  };

  const handleDelete = (id: string) => {
    if (confirm("Are you sure you want to delete this teacher?")) {
      deleteTeacher.mutate(id, {
        onSuccess: () => {
          refetch();
        },
      });
    }
  };

  const handleCloseForm = () => {
    setShowForm(false);
    setSelectedTeacher(undefined);
  };

  // Stats
  const totalTeachers = teachers.length;
  const activeTeachers = teachers.filter((t) => t.active !== false).length;
  const maleTeachers = teachers.filter((t) => t.gender === "male").length;
  const femaleTeachers = teachers.filter((t) => t.gender === "female").length;

  const stats = [
    { label: "Total Teachers", value: totalTeachers },
    { label: "Active", value: activeTeachers },
    { label: "Male", value: maleTeachers },
    { label: "Female", value: femaleTeachers },
  ];

  if (error) {
    return (
      <div className="p-6 bg-red-50 text-red-700 rounded-lg">
        Error loading teachers:{" "}
        {error instanceof Error ? error.message : "Unknown error"}
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold text-gray-900">
          Teachers Management
        </h1>
        <button
          onClick={() => {
            setSelectedTeacher(undefined);
            setShowForm(true);
          }}
          className="flex items-center gap-2 bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700 transition"
        >
          <Plus size={20} />
          Add Teacher
        </button>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        {stats.map((stat) => (
          <div
            key={stat.label}
            className="bg-white p-4 rounded-lg shadow-sm border-l-4 border-primary-600"
          >
            <p className="text-sm text-gray-600">{stat.label}</p>
            <p className="text-3xl font-bold text-gray-900 mt-1">
              {stat.value}
            </p>
          </div>
        ))}
      </div>

      {/* Search */}
      <div>
        <input
          type="text"
          placeholder="Search teachers by name, position, or email..."
          value={searchTerm}
          onChange={(e) => handleSearch(e.target.value)}
          className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
        />
      </div>

      {/* Table */}
      {isLoading ? (
        <div className="p-8 text-center text-gray-500">Loading teachers...</div>
      ) : filteredTeachers.length === 0 ? (
        <div className="p-8 text-center text-gray-500 bg-white rounded-lg">
          {teachers.length === 0
            ? "No teachers yet. Add your first teacher!"
            : "No teachers match your search."}
        </div>
      ) : (
        <div className="space-y-4">
          <Table
            data={paginatedTeachers}
            columns={[
              {
                key: "name",
                label: "Name",
                render: (value: string, row: Teacher) => (
                  <div className="flex items-center gap-3">
                    <Avatar
                      src={row.image}
                      name={row.name}
                      size="sm"
                      alt={`${row.name} avatar`}
                    />
                    <div>
                      <p className="font-medium text-gray-900">{row.name}</p>
                      <p className="text-xs text-gray-500">{row.title}</p>
                    </div>
                  </div>
                ),
              },
              {
                key: "position",
                label: "Position",
                render: (value: string) => (
                  <span className="text-gray-600">{value}</span>
                ),
              },
              {
                key: "email",
                label: "Email",
                render: (value: string | undefined) => (
                  <span className="text-gray-600">{value || "-"}</span>
                ),
              },
              {
                key: "experience",
                label: "Experience",
                render: (value: number | undefined) => (
                  <span className="text-gray-600">
                    {value ? `${value} yrs` : "-"}
                  </span>
                ),
              },
              {
                key: "active",
                label: "Status",
                render: (value: boolean | undefined) => (
                  <span
                    className={`px-3 py-1 rounded-full text-xs font-semibold ${
                      value !== false
                        ? "bg-green-100 text-green-800"
                        : "bg-gray-100 text-gray-800"
                    }`}
                  >
                    {value !== false ? "Active" : "Inactive"}
                  </span>
                ),
              },
              {
                key: "id",
                label: "Actions",
                render: (value: string, row: Teacher) => (
                  <div className="flex items-center gap-2 justify-end">
                    <Link
                      href={`/team/${row.slug}`}
                      target="_blank"
                      className="text-blue-600 hover:text-blue-700 p-1"
                      title="View profile"
                    >
                      <Eye size={18} />
                    </Link>
                    <button
                      onClick={() => handleEdit(row)}
                      className="text-primary-600 hover:text-primary-700 p-1"
                      title="Edit"
                    >
                      <Edit size={18} />
                    </button>
                    <button
                      onClick={() => handleDelete(row.id)}
                      disabled={deleteTeacher.isPending}
                      className="text-red-600 hover:text-red-700 p-1 disabled:opacity-50"
                      title="Delete"
                    >
                      <Trash2 size={18} />
                    </button>
                  </div>
                ),
              },
            ]}
            variant="primary"
          />

          <TablePagination
            currentPage={currentPage}
            totalPages={totalPages}
            onPageChange={setCurrentPage}
            itemsPerPage={itemsPerPage}
            totalItems={filteredTeachers.length}
            variant="footer"
          />
        </div>
      )}

      {/* Teacher Form Modal */}
      {showForm && (
        <TeacherForm
          teacher={selectedTeacher}
          onClose={handleCloseForm}
          onSuccess={() => {
            handleCloseForm();
            refetch();
          }}
        />
      )}
    </div>
  );
}
