'use client';

import { useState } from 'react';
import { useAdminUsers, useApproveUser, useRejectUser, useSuspendUser, useDeleteUser, useUpdateUserPermissions, AdminUser } from '@/hooks/useAdminUserManagement';
import Table from '@/components/common/ui/table';
import { Badge, Modal } from '@/components/common/ui';
import { toast } from 'react-toastify';
import { useAuth } from '@/context/auth-context';
import { roleLabels, roleBadgeColors } from '@/lib/admin/permissions';
import EditPermissionsModal from '@/components/admin/modals/EditPermissionsModal';
import UserApprovalModal from '@/components/admin/forms/UserApprovalModal';
import PermissionSummary from '@/components/admin/ui/PermissionSummary';
import { UserPagePermissions } from '@/lib/admin/page-permissions';

export default function UsersPage() {
  const { user: currentUser } = useAuth();
  const [statusFilter, setStatusFilter] = useState<'pending' | 'approved' | 'rejected' | 'suspended' | ''>('');
  const [searchQuery, setSearchQuery] = useState('');
  const [actionModal, setActionModal] = useState<{ type: 'reject' | 'suspend' | 'delete'; userId: string; userEmail?: string } | null>(null);
  const [approvalModal, setApprovalModal] = useState<{ userId: string; userEmail: string } | null>(null);
  const [editModal, setEditModal] = useState<{ user: AdminUser } | null>(null);
  const [rejectionReason, setRejectionReason] = useState('');

  const { data, isLoading } = useAdminUsers({
    status: statusFilter || undefined,
    search: searchQuery || undefined,
    limit: 50,
  });

  const approveUser = useApproveUser();
  const rejectUser = useRejectUser();
  const suspendUser = useSuspendUser();
  const deleteUser = useDeleteUser();
  const updateUserPermissions = useUpdateUserPermissions();

  const users = data?.users || [];

  // Check if the user is trying to act on themselves
  const isCurrentUser = (userId: string) => {
    return currentUser?.id === userId;
  };

  const handleApprove = (role: 'admin' | 'editor' | 'viewer') => {
    if (!approvalModal) return;
    
    if (isCurrentUser(approvalModal.userId)) {
      toast.error('❌ You cannot approve your own account');
      return;
    }
    
    // Approve with role, permissions will be null (must be set via Edit button)
    approveUser.mutate({ userId: approvalModal.userId, role, permissions: null }, {
      onSuccess: (data) => {
        const email = data?.data?.email || 'User';
        const assignedRole = data?.data?.role || role;
        toast.success(`✅ ${email} has been approved as ${assignedRole}. Use "Edit" to set permissions.`);
        setApprovalModal(null);
      },
      onError: (error) => {
        toast.error(`❌ Failed to approve user: ${error.message}`);
      },
    });
  };

  const handleReject = (userId: string) => {
    if (isCurrentUser(userId)) {
      toast.error('❌ You cannot reject your own account');
      return;
    }
    rejectUser.mutate(
      { userId, reason: rejectionReason },
      {
        onSuccess: (data) => {
          const email = data?.data?.email || 'User';
          toast.success(`🚫 ${email} has been rejected`);
          setActionModal(null);
          setRejectionReason('');
        },
        onError: (error) => {
          toast.error(`❌ Failed to reject user: ${error.message}`);
          setActionModal(null);
        },
      }
    );
  };

  const handleSuspend = (userId: string) => {
    if (isCurrentUser(userId)) {
      toast.error('❌ You cannot suspend your own account');
      return;
    }
    suspendUser.mutate(
      { userId, reason: rejectionReason },
      {
        onSuccess: (data) => {
          const email = data?.data?.email || 'User';
          toast.warning(`⚠️ ${email} has been suspended`);
          setActionModal(null);
          setRejectionReason('');
        },
        onError: (error) => {
          toast.error(`❌ Failed to suspend user: ${error.message}`);
          setActionModal(null);
        },
      }
    );
  };

  const handleDelete = (userId: string) => {
    if (isCurrentUser(userId)) {
      toast.error('❌ You cannot delete your own account');
      return;
    }
    deleteUser.mutate(userId, {
      onSuccess: (data) => {
        const email = data?.data?.deletedEmail || 'User';
        toast.success(`🗑️ ${email} has been permanently deleted`);
        setActionModal(null);
        setRejectionReason('');
      },
      onError: (error) => {
        toast.error(`❌ Failed to delete user: ${error.message}`);
        setActionModal(null);
      },
    });
  };

  const handleUpdatePermissions = (role: 'admin' | 'editor' | 'viewer', permissions: UserPagePermissions | null) => {
    if (!editModal) return;
    
    if (isCurrentUser(editModal.user.id)) {
      toast.error('❌ You cannot modify your own permissions');
      return;
    }

    updateUserPermissions.mutate(
      { 
        userId: editModal.user.id, 
        role, 
        permissions 
      },
      {
        onSuccess: (data) => {
          const email = data?.data?.email || 'User';
          toast.success(`✅ Permissions updated for ${email}`);
          setEditModal(null);
        },
        onError: (error) => {
          toast.error(`❌ Failed to update permissions: ${error.message}`);
        },
      }
    );
  };

  const getStatusColor = (status: string): 'warning' | 'success' | 'error' | 'default' | 'primary' => {
    switch (status) {
      case 'pending':
        return 'warning';
      case 'approved':
        return 'success';
      case 'rejected':
      case 'suspended':
        return 'error';
      default:
        return 'default';
    }
  };

  const columns = [
    {
      key: 'name' as const,
      label: 'Name',
      sortable: true,
    },
    {
      key: 'email' as const,
      label: 'Email',
      sortable: true,
    },
    {
      key: 'role' as const,
      label: 'Role',
      render: (value: string) => {
        const role = value as 'admin' | 'editor' | 'viewer';
        const color = roleBadgeColors[role] || roleBadgeColors.viewer;
        return (
          <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${color}`}>
            {roleLabels[role] || value}
          </span>
        );
      },
    },
    {
      key: 'permissions' as const,
      label: 'Permissions',
      render: (_: unknown, row: AdminUser) => {
        if (row.status !== 'approved') {
          return <span className="text-xs text-gray-400">—</span>;
        }
        return (
          <PermissionSummary 
            role={row.role as 'admin' | 'editor' | 'viewer'} 
            permissions={row.permissions as UserPagePermissions | null}
            compact
          />
        );
      },
    },
    {
      key: 'status' as const,
      label: 'Status',
      render: (value: string) => (
        <Badge variant={getStatusColor(value)}>{value}</Badge>
      ),
    },
    {
      key: 'createdAt' as const,
      label: 'Created',
      render: (value: string) => new Date(value).toLocaleDateString(),
    },
    {
      key: 'id' as const,
      label: 'Actions',
      render: (_: unknown, row: AdminUser) => {
        const isSelf = isCurrentUser(row.id);
        
        if (isSelf) {
          return (
            <span className="text-xs text-gray-500 italic">You</span>
          );
        }
        
        return (
          <div className="flex flex-wrap gap-2">
            {row.status === 'pending' && (
              <>
                <button
                  onClick={() => {
                    setApprovalModal({ userId: row.id, userEmail: row.email });
                  }}
                  className="px-2 py-1 text-xs font-medium text-green-600 hover:bg-green-50 rounded transition-colors"
                >
                  Approve
                </button>
                <button
                  onClick={() => {
                    setActionModal({ type: 'reject', userId: row.id, userEmail: row.email });
                  }}
                  className="px-2 py-1 text-xs font-medium text-red-600 hover:bg-red-50 rounded transition-colors"
                >
                  Reject
                </button>
              </>
            )}
            {row.status === 'approved' && (
              <>
                <button
                  onClick={() => {
                    setEditModal({ user: row });
                  }}
                  className="px-2 py-1 text-xs font-medium text-blue-600 hover:bg-blue-50 rounded transition-colors"
                >
                  Manage Permissions
                </button>
                <button
                  onClick={() => {
                    setActionModal({ type: 'suspend', userId: row.id, userEmail: row.email });
                  }}
                  className="px-2 py-1 text-xs font-medium text-orange-600 hover:bg-orange-50 rounded transition-colors"
                >
                  Suspend
                </button>
              </>
            )}
            {row.status === 'suspended' && (
              <>
                <button
                  onClick={() => {
                    setApprovalModal({ userId: row.id, userEmail: row.email });
                  }}
                  className="px-2 py-1 text-xs font-medium text-green-600 hover:bg-green-50 rounded transition-colors"
                >
                  Reactivate
                </button>
                <button
                  onClick={() => {
                    setActionModal({ type: 'delete', userId: row.id, userEmail: row.email });
                  }}
                  className="px-2 py-1 text-xs font-medium text-red-600 hover:bg-red-50 rounded transition-colors"
                >
                  Delete
                </button>
              </>
            )}
            {row.status === 'rejected' && (
              <button
                onClick={() => {
                  setActionModal({ type: 'delete', userId: row.id, userEmail: row.email });
                }}
                className="px-2 py-1 text-xs font-medium text-red-600 hover:bg-red-50 rounded transition-colors"
              >
                Delete
              </button>
            )}
          </div>
        );
      },
    },
  ];

  return (
    <>
      <div className="min-h-screen bg-gray-50 p-4 md:p-8">
        {/* Header */}
        <div className="mb-6">
          <h1 className="text-2xl md:text-3xl font-bold text-gray-900">User Management</h1>
          <p className="text-gray-600 mt-1">
            Manage admin user accounts and approvals. Total: {data?.pagination.total || 0}
          </p>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow p-4 md:p-6 mb-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Search */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Search</label>
              <input
                type="text"
                placeholder="Search by name or email..."
                value={searchQuery}
                onChange={(e: React.ChangeEvent<HTMLInputElement>) => setSearchQuery(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>

            {/* Status Filter */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Filter by Status</label>
              <select
                value={statusFilter}
                onChange={(e: React.ChangeEvent<HTMLSelectElement>) => {
                  const value = e.target.value as '' | 'pending' | 'approved' | 'rejected' | 'suspended';
                  setStatusFilter(value);
                }}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
              >
                <option value="">All Users</option>
                <option value="pending">Pending</option>
                <option value="approved">Approved</option>
                <option value="rejected">Rejected</option>
                <option value="suspended">Suspended</option>
              </select>
            </div>
          </div>
        </div>

        {/* Table - Responsive (Desktop Table / Mobile Cards) */}
        <div className="bg-white rounded-lg shadow overflow-hidden">
          {isLoading ? (
            <div className="p-8 text-center text-gray-500">Loading users...</div>
          ) : users.length === 0 ? (
            <div className="p-8 text-center text-gray-500">No users found</div>
          ) : (
            <>
              {/* Desktop Table View with Horizontal Scroll */}
              <div className="w-full">
                <Table<AdminUser> data={users} columns={columns} variant="default" />
              </div>

            </>
          )}
        </div>

        {/* Pagination */}
        {data?.pagination && data.pagination.pages > 1 && (
          <div className="mt-4 text-center text-sm text-gray-600">
            Page 1 of {data.pagination.pages} ({data.pagination.total} total users)
          </div>
        )}
      </div>

      {/* Action Modal (Reject/Suspend/Delete) */}
      {actionModal && (
        <Modal
          isOpen={true}
          onClose={() => {
            setActionModal(null);
            setRejectionReason('');
          }}
          title={
            actionModal.type === 'reject'
              ? 'Reject User'
              : actionModal.type === 'suspend'
                ? 'Suspend User'
                : 'Delete User Permanently'
          }
        >
          <div className="space-y-4">
            {actionModal.userEmail && (
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
                <p className="text-sm font-medium text-blue-900">
                  User: <span className="font-semibold">{actionModal.userEmail}</span>
                </p>
              </div>
            )}

            <p className="text-gray-600">
              {actionModal.type === 'reject'
                ? 'This user will not be able to login.'
                : actionModal.type === 'suspend'
                  ? 'This user will be blocked from logging in.'
                  : '⚠️ This action cannot be undone. All user data will be permanently deleted.'}
            </p>

            {(actionModal.type === 'reject' || actionModal.type === 'suspend') && (
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Reason (optional)</label>
                <textarea
                  value={rejectionReason}
                  onChange={(e: React.ChangeEvent<HTMLTextAreaElement>) => setRejectionReason(e.target.value)}
                  placeholder="Provide a reason..."
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
            )}

            <div className="flex gap-3 justify-end">
              <button
                onClick={() => {
                  setActionModal(null);
                  setRejectionReason('');
                }}
                className="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={() => {
                  if (actionModal.type === 'reject') {
                    handleReject(actionModal.userId);
                  } else if (actionModal.type === 'suspend') {
                    handleSuspend(actionModal.userId);
                  } else if (actionModal.type === 'delete') {
                    handleDelete(actionModal.userId);
                  }
                }}
                disabled={rejectUser.isPending || suspendUser.isPending || deleteUser.isPending}
                className={`px-4 py-2 text-sm font-medium text-white rounded-lg transition-colors disabled:opacity-50 ${
                  actionModal.type === 'reject' || actionModal.type === 'delete'
                    ? 'bg-red-600 hover:bg-red-700'
                    : 'bg-orange-600 hover:bg-orange-700'
                }`}
              >
                {rejectUser.isPending || suspendUser.isPending || deleteUser.isPending
                  ? 'Processing...'
                  : actionModal.type === 'delete'
                    ? 'Delete Permanently'
                    : 'Confirm'}
              </button>
            </div>
          </div>
        </Modal>
      )}

      {/* Approval Modal (with Permissions) */}
      {approvalModal && (
        <UserApprovalModal
          isOpen={true}
          onClose={() => setApprovalModal(null)}
          onApprove={handleApprove}
          userEmail={approvalModal.userEmail}
          isPending={approveUser.isPending}
        />
      )}

      {/* Edit Permissions Modal */}
      {editModal && (
        <EditPermissionsModal
          isOpen={true}
          onClose={() => setEditModal(null)}
          onSave={handleUpdatePermissions}
          userName={editModal.user.name}
          userEmail={editModal.user.email}
          currentRole={editModal.user.role}
          currentPermissions={editModal.user.permissions || null}
          isPending={updateUserPermissions.isPending}
        />
      )}
    </>
  );
}
