"use client";

import { signIn } from "next-auth/react";
import { useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { GraduationCap } from "lucide-react";
import { InputField } from "@/components/common/ui";
import { useCsrfToken } from "@/hooks/useCsrfToken";

export default function LoginPage() {
  const { token, loading: tokenLoading } = useCsrfToken();
  const router = useRouter();
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [error, setError] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError("");
    setLoading(true);

    try {
      const result = await signIn("credentials", {
        email,
        password,
        csrfToken: token || undefined,
        redirect: false,
      });

      if (result?.error) {
        // Map error codes to user-friendly messages
        let errorMessage = "Invalid email or password. Please try again.";
        console.log("Sign in error:", result, result.error);
        
        // Check for specific account status errors
        if (result.code === "ACCOUNT_PENDING_APPROVAL") {
          errorMessage = "⏳ Your account is pending admin approval. Please wait for an administrator to approve your account.";
        } else if (result.code === "ACCOUNT_REJECTED") {
          errorMessage = "🚫 Your account has been rejected. Please contact the administrator for more information.";
        } else if (result.code === "ACCOUNT_SUSPENDED") {
          errorMessage = "⚠️ Your account has been suspended. Please contact the administrator.";
        } else if (result.code === "ACCOUNT_INACTIVE") {
          errorMessage = "❌ Your account is inactive. Please contact the administrator.";
        }
        
        setError(errorMessage);
      } else if (result?.ok) {
        router.push("/admin");
      }
    } catch (err) {
      console.error("Sign in error:", err);
      setError("An unexpected error occurred. Please try again.");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen w-full flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100 px-4 sm:px-6 lg:px-8">
      <div className="w-full max-w-md">
        {/* Logo */}
        <div className="flex justify-center mb-8">
          <div className="flex items-center space-x-2">
            <div className="p-2 bg-gradient-to-br from-blue-600 to-purple-600 rounded-lg">
              <GraduationCap className="w-8 h-8 text-white" />
            </div>
            <span className="text-2xl font-bold text-gray-900">Admin Panel</span>
          </div>
        </div>

        {/* Card */}
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          {/* Header */}
          <div className="bg-gradient-to-r from-blue-600 to-purple-600 px-6 py-8">
            <h1 className="text-2xl font-bold text-white">Welcome Back</h1>
            <p className="text-blue-100 mt-1">Sign in to your admin account</p>
          </div>

          {/* Form */}
          <form onSubmit={handleSubmit} className="px-6 py-8 space-y-6">
            {/* Error Message */}
            {error && (
              <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                <p className="text-sm text-red-700">{error}</p>
              </div>
            )}

            {/* Email Field */}
            <InputField
              id="email"
              name="email"
              type="email"
              label="Email Address"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              placeholder="admin@school.com"
              disabled={loading}
              required
            />

            {/* Password Field */}
            <InputField
              id="password"
              name="password"
              type="password"
              label="Password"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              placeholder="••••••••"
              disabled={loading}
              showPasswordToggle={true}
              required
            />

            {/* Forgot Password Link */}
            <div className="flex justify-end">
              <Link
                href="/auth/forgot-password"
                className="text-sm text-blue-600 hover:text-blue-700 font-medium"
              >
                Forgot password?
              </Link>
            </div>

            {/* Submit Button */}
            <button
              type="submit"
              disabled={loading || tokenLoading || !token}
              className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 disabled:from-gray-400 disabled:to-gray-500 text-white font-medium py-2 px-4 rounded-lg transition-all duration-200"
            >
              {loading ? "Signing in..." : tokenLoading ? "Initializing..." : "Sign In"}
            </button>
          </form>

          {/* Footer */}
          <div className="bg-gray-50 px-6 py-4 border-t border-gray-200 text-center">
            <p className="text-sm text-gray-600">
              Need help?{" "}
              <Link href="/contact" className="text-blue-600 hover:text-blue-700 font-medium">
                Contact support
              </Link>
            </p>
          </div>
        </div>

        {/* Test Credentials Note */}
        <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
          <p className="text-xs text-blue-700">
            <strong>Demo:</strong> Use test credentials provided by your administrator
          </p>
        </div>
      </div>
    </div>
  );
}
