import React from "react";
import Image from "next/image";
import { Section, ResponsiveGrid, AppLayout } from "@/components/common/layout";
import Badge from "@/components/common/ui/badge";
import Card from "@/components/common/ui/card";
import {
  Users,
  Target,
  Heart,
  Award,
  Calendar,
  MapPin,
  ArrowRight,
} from "lucide-react";

const AboutPage = () => {
  const values = [
    {
      icon: Users,
      title: "Gender Equality",
      description:
        "Our motto is to teach and implement gender equality to our students inside the school and through them, in the community.",
      color: "text-blue-600",
    },
    {
      icon: Heart,
      title: "Women Empowerment",
      description:
        "We provide different activities for our female students to have access to various opportunities and resources.",
      color: "text-red-600",
    },
    {
      icon: Target,
      title: "Poverty Reduction",
      description:
        "Our efforts are concentrated on children who do not have the financial capacity to attend school, believing education is the only way to reduce poverty.",
      color: "text-green-600",
    },
    {
      icon: Award,
      title: "Quality Education",
      description:
        "We are committed to providing affordable, high-quality education implementing principles of innovation in pedagogy and Sustainable Development Goals.",
      color: "text-primary-600",
    },
  ];

  const milestones = [
    {
      year: "2022-Present",
      title: "Adapting to Challenges",
      description:
        "Despite restrictions on girls' education, the school continues offering practical skills courses including English, computer skills, painting, sewing, and literacy programs for the community, especially women.",
    },
    {
      year: "2019-2021",
      title: "Winter Programs & COVID Response",
      description:
        "Organized comprehensive winter programs: 530 students (2019), 320 female students grades 7-12 (2020), with Mathematics, Physics, Chemistry, and English courses, including online classes during COVID-19.",
    },
    {
      year: "2019",
      title: "Setaragan Zirak School Established",
      description:
        "Azizullah Paeyndeh founded Setaragan Zirak School with support from the Zirak Charity Foundation, based in Colorado, USA, bringing innovation and technology-driven education to continue the legacy.",
    },
    {
      year: "2000s",
      title: "International Partnership & Expansion",
      description:
        "A Swedish charity began supporting primary education for girls (grades 1-6), while the school attracted students from neighboring villages like Mohajerin, Bosayed, and Loman with separate shifts for boys and girls.",
    },
    {
      year: "1990",
      title: "Salehi Zirak School Founded",
      description:
        "Mr. Ezzatullah Salehi established the first school in Zirak village, starting with classes in temporary tents beneath trees, eventually growing to a comprehensive high school with 48 classrooms serving thousands of students.",
    },
  ];

  return (
    <AppLayout>
    <div className="min-h-screen bg-gradient-to-br from-gray-50 via-white to-primary-50/20">
      {/* Enhanced Modern Hero Header */}
      <div className="relative bg-gradient-to-br from-primary-50/80 via-white to-secondary-50/80 border-b border-gray-100/50 overflow-hidden backdrop-blur-sm">
        {/* Enhanced Background Pattern */}
        <div className="absolute inset-0 opacity-20">
          <div className="absolute top-0 left-0 w-96 h-96 bg-gradient-to-br from-primary-300 via-primary-200 to-transparent rounded-full -translate-x-48 -translate-y-48 animate-pulse"></div>
          <div className="absolute bottom-0 right-0 w-[500px] h-[500px] bg-gradient-to-br from-secondary-300 via-secondary-200 to-transparent rounded-full translate-x-64 translate-y-64 animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-32 h-32 bg-gradient-to-br from-primary-400 to-secondary-400 rounded-full -translate-x-16 -translate-y-16 opacity-30 animate-bounce delay-2000"></div>
        </div>

        <div className="relative max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 py-16 sm:py-20 lg:py-24">
          {/* Enhanced Hero Content */}
          <div className="max-w-5xl">
            <div className="relative">
              <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-black tracking-tight leading-none text-gray-900 mb-8">
                About{" "}
                <span className="relative inline-block">
                  <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent animate-pulse">
                    Setaragan Zirak
                  </span>
                  <div className="absolute inset-0 bg-gradient-to-r from-primary-600 to-secondary-600 opacity-10 blur-xl"></div>
                </span>{" "}
                School
              </h1>
              <div className="w-24 h-1 bg-gradient-to-r from-primary-500 to-secondary-500 mb-6 rounded-full"></div>
            </div>
            <p className="text-base sm:text-lg md:text-xl lg:text-2xl leading-relaxed text-gray-700 mb-10 max-w-4xl">
              Discover our{" "}
              <span className="font-semibold text-primary-600">mission</span>,
              <span className="font-semibold text-secondary-600"> values</span>,
              and commitment to
              <span className="font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                {" "}
                educational excellence
              </span>{" "}
              that has shaped generations of successful students.
            </p>

            {/* Enhanced Quick Stats */}
            <div className="grid grid-cols-2 lg:grid-cols-4 gap-6 mt-16">
              <div className="relative group">
                <div className="text-center p-6 rounded-2xl bg-white/80 backdrop-blur-sm border border-gray-100/50 shadow-lg hover:shadow-2xl transition-all duration-300 hover:scale-105">
                  <div className="text-2xl sm:text-3xl md:text-4xl font-black bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-3">
                    30+
                  </div>
                  <div className="text-xs sm:text-sm font-semibold tracking-wide text-gray-600">
                    Years of Excellence
                  </div>
                  <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-300 -z-10"></div>
                </div>
              </div>
              <div className="relative group">
                <div className="text-center p-6 rounded-2xl bg-white/80 backdrop-blur-sm border border-gray-100/50 shadow-lg hover:shadow-2xl transition-all duration-300 hover:scale-105">
                  <div className="text-2xl sm:text-3xl md:text-4xl font-black bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-3">
                    500+
                  </div>
                  <div className="text-xs sm:text-sm font-semibold tracking-wide text-gray-600">
                    Graduates
                  </div>
                  <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-300 -z-10"></div>
                </div>
              </div>
              <div className="relative group">
                <div className="text-center p-6 rounded-2xl bg-white/80 backdrop-blur-sm border border-gray-100/50 shadow-lg hover:shadow-2xl transition-all duration-300 hover:scale-105">
                  <div className="text-2xl sm:text-3xl md:text-4xl font-black bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-3">
                    25+
                  </div>
                  <div className="text-xs sm:text-sm font-semibold tracking-wide text-gray-600">
                    Expert Teachers
                  </div>
                  <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-300 -z-10"></div>
                </div>
              </div>
              <div className="relative group">
                <div className="text-center p-6 rounded-2xl bg-white/80 backdrop-blur-sm border border-gray-100/50 shadow-lg hover:shadow-2xl transition-all duration-300 hover:scale-105">
                  <div className="text-4xl font-black bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-3">
                    1990
                  </div>
                  <div className="text-sm text-gray-600 font-semibold">
                    Established
                  </div>
                  <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-300 -z-10"></div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Enhanced Our Story Section */}
      <Section
        background="white"
        padding="xl"
        className="scroll-mt-20 bg-gradient-to-br from-white via-primary-50/30 to-secondary-50/30"
        id="our-story"
      >
        <div className="text-center mb-16 sm:mb-20">
          <div className="inline-flex items-center px-4 py-2 rounded-full bg-gradient-to-r from-primary-100 to-secondary-100 border border-primary-200/50 mb-6">
            <span className="text-sm font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Our Story
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-8">
            From Humble Beginnings to{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Educational Excellence
            </span>
          </h2>
          <div className="w-24 h-1 bg-gradient-to-r from-primary-500 to-secondary-500 mx-auto mb-8 rounded-full"></div>
          <p className="text-base sm:text-lg md:text-xl leading-relaxed text-gray-600 max-w-4xl mx-auto">
            The story of Setaragan Zirak School begins{" "}
            <span className="font-semibold text-primary-600">
              thirty years ago
            </span>{" "}
            in the village of Zirak, located in the Jaghori district of Ghazni
            province, where severe obstacles blocked access to{" "}
            <span className="font-semibold text-secondary-600">education</span>.
          </p>
        </div>

        <div className="max-w-5xl mx-auto mt-16">
          <div className="grid gap-8 lg:gap-12">
            {/* Section 1: Tackling Educational Challenges */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-lg sm:text-xl lg:text-2xl font-bold leading-snug tracking-tight text-gray-900 mb-4">
                      Tackling Educational Challenges: A Vision For Progress
                    </h3>
                    <p className="text-sm sm:text-base leading-relaxed text-gray-700 group-hover:text-gray-800 transition-colors duration-300">
                      Boys had to endure a grueling{" "}
                      <span className="font-semibold text-primary-600">
                        three-hour journey
                      </span>{" "}
                      on difficult terrain to reach the sole school in the area,
                      situated in the village of Angori. For girls, attending
                      school was virtually impossible due to cultural
                      restrictions and a lack of transportation. These barriers
                      deprived many children of their basic right to education.
                    </p>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/ab1.jpg"
                      alt="Educational challenges in Zirak village"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Section 2: A Transformative Effort - Founding Salehi Zirak School */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-lg sm:text-xl lg:text-2xl font-bold leading-snug tracking-tight text-gray-900 mb-4">
                      A Transformative Effort: Founding Salehi Zirak School
                    </h3>
                    <p className="text-sm sm:text-base leading-relaxed text-gray-700 group-hover:text-gray-800 transition-colors duration-300">
                      In{" "}
                      <span className="font-bold text-primary-600">1990</span>,
                      Mr. Ezzatullah Salehi, a native of Zirak, took decisive
                      action to address this dire situation. After returning
                      from the United States and witnessing the community&apos;s
                      deep desire for education, he established the{" "}
                      <span className="font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                        Salehi Zirak School
                      </span>{" "}
                      in collaboration with local teachers. This initiative
                      marked a pivotal moment in the educational history of the
                      region.
                    </p>
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300 mt-4">
                      Initially, the school operated under challenging
                      conditions, with classes held in temporary tents beneath
                      the shade of trees. Despite these limitations, it began by
                      offering education up to the third grade. As
                      infrastructure developed, the school expanded to include
                      grades up to the ninth, ultimately transforming into a
                      comprehensive high school.
                    </p>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/ab2.jpg"
                      alt="Founding of Salehi Zirak School in 1990"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Section 3: School Growth and Strategic Location */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300">
                      Thanks to its strategic location, Salehi Zirak School
                      attracted students from neighboring villages such as{" "}
                      <span className="font-semibold text-primary-600">
                        Mohajerin, Bosayed, and Loman
                      </span>
                      . The school adopted an{" "}
                      <span className="font-semibold text-primary-600">
                        inclusive model
                      </span>{" "}
                      with separate shifts:{" "}
                      <span className="font-semibold text-secondary-600">
                        morning classes for girls and afternoon classes for boys
                      </span>
                      . Mr. Salehi fully funded the construction and teacher
                      salaries for the{" "}
                      <span className="font-semibold text-secondary-600">
                        48-classroom facility
                      </span>
                      .
                    </p>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/ab3.jpg"
                      alt="Salehi Zirak School expansion and student life"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Section 4: Expanding Educational Opportunities */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4">
                      Expanding Educational Opportunities
                    </h3>
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300">
                      Over time, a{" "}
                      <span className="font-semibold text-primary-600">
                        Swedish charity
                      </span>{" "}
                      began supporting the primary education of girls (grades
                      one through six), while Mr. Salehi continued to fund the
                      remaining sections of the school. This partnership helped
                      break down numerous cultural and economic barriers,
                      creating a welcoming environment for girls&apos;
                      education. To date,{" "}
                      <span className="font-semibold text-secondary-600">
                        thousands of students
                      </span>
                      —both male and female—have graduated from Salehi Zirak
                      School. Many have pursued higher education across various
                      fields, contributing significantly to the social and
                      economic development of the region.
                    </p>
                  </div>
                  <div className="lg:w-1/3">
                    <div className="space-y-4">
                      <div className="text-center p-4 bg-gradient-to-r from-primary-50 to-secondary-50 rounded-lg">
                        <div className="text-2xl font-bold text-primary-600">
                          1000+
                        </div>
                        <div className="text-sm text-gray-600">
                          Students Graduated
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-2 text-center">
                        <div className="p-2 bg-blue-50 rounded">
                          <div className="text-lg font-bold text-blue-600">
                            448+
                          </div>
                          <div className="text-xs text-gray-600">Female</div>
                        </div>
                        <div className="p-2 bg-green-50 rounded">
                          <div className="text-lg font-bold text-green-600">
                            399+
                          </div>
                          <div className="text-xs text-gray-600">Male</div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Section 5: Specialized Educational Programs */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4">
                      Specialized Educational And Reinforcement Programs
                    </h3>
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300 mb-4">
                      Salehi Zirak School has implemented customized programs to
                      enhance learning opportunities for its students:
                    </p>
                    <div className="space-y-3">
                      <div className="p-3 bg-blue-50 rounded-lg">
                        <div className="font-semibold text-blue-800">
                          For boys:
                        </div>
                        <div className="text-gray-700">
                          Free courses in Physics, Chemistry, Mathematics, and
                          preparation for the university entrance exam (Konkur).
                        </div>
                      </div>
                      <div className="p-3 bg-pink-50 rounded-lg">
                        <div className="font-semibold text-pink-800">
                          For girls:
                        </div>
                        <div className="text-gray-700">
                          English language training and skill-building courses
                          tailored to meet their specific needs.
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/ab5.jpg"
                      alt="Specialized educational programs"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Section 6: A New Era - Setaragan Zirak School */}
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-primary-500 to-secondary-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-primary-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4">
                      A New Era: Setaragan Zirak School
                    </h3>
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300 mb-4">
                      In{" "}
                      <span className="font-bold text-secondary-600">2019</span>
                      , Azizullah Paeyndeh established the{" "}
                      <span className="font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                        Setaragan Zirak School
                      </span>{" "}
                      with financial support from the Zirak Charity Foundation,
                      based in Colorado, USA. Drawing inspiration from the
                      success of Salehi Zirak School, Setaragan Zirak School is
                      committed to fostering{" "}
                      <span className="font-semibold text-primary-600">
                        innovation
                      </span>
                      ,{" "}
                      <span className="font-semibold text-secondary-600">
                        inclusivity
                      </span>
                      , and{" "}
                      <span className="font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                        technology-driven education
                      </span>
                      .
                    </p>
                    <p className="text-gray-700 leading-relaxed text-lg group-hover:text-gray-800 transition-colors duration-300">
                      Despite significant challenges, including the COVID-19
                      pandemic and regional instability, the school has achieved
                      noteworthy progress. Its core objectives include:
                      Eliminating gender-based disparities in education,
                      identifying and nurturing talents, and fostering
                      leadership and civic responsibility among students.
                    </p>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/new-era.jpg"
                      alt="Setaragan Zirak School - A New Era"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </Section>

      {/* Mission & Vision Section */}
      <Section background="gray" padding="lg">
        <div className="text-center mb-12 sm:mb-16">
          <div className="inline-flex items-center px-4 py-2 rounded-full bg-gradient-to-r from-primary-100 to-secondary-100 border border-primary-200/50 mb-6">
            <span className="text-sm font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Foundation
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Why We Exist &{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Where We&apos;re Going
            </span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
            Our mission drives our daily actions, while our vision guides our
            long-term commitment to educational excellence.
          </p>
        </div>

        <ResponsiveGrid cols={{ default: 1, lg: 2 }} gap="xl">
          {/* Mission */}
          <Card className="p-6 sm:p-8 lg:p-10 h-full border-l-4 border-l-primary-600">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-primary-100 rounded-lg flex items-center justify-center">
                  <Target className="w-6 h-6 text-primary-600" />
                </div>
              </div>
              <div>
                <h2 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4">
                  Our Mission
                </h2>
                <p className="text-base sm:text-lg text-gray-600 leading-relaxed mb-4">
                  To provide equitable, high-quality education that empowers
                  students and their families by removing barriers and promoting
                  inclusivity.
                </p>
                <ul className="text-gray-600 space-y-2">
                  <li>• Eliminating gender-based disparities in education</li>
                  <li>• Identifying and nurturing student talents</li>
                  <li>• Fostering leadership and civic responsibility</li>
                  <li>• Implementing innovative pedagogical approaches</li>
                </ul>
              </div>
            </div>
          </Card>

          {/* Vision */}
          <Card className="p-6 sm:p-8 lg:p-10 h-full border-l-4 border-l-secondary-600">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-secondary-100 rounded-lg flex items-center justify-center">
                  <Heart className="w-6 h-6 text-secondary-600" />
                </div>
              </div>
              <div>
                <h2 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4">
                  Our Vision
                </h2>
                <p className="text-base sm:text-lg text-gray-600 leading-relaxed mb-4">
                  To foster positive social change, eliminate inequalities, and
                  cultivate a lifelong culture of learning and civic
                  responsibility for sustainable development.
                </p>
                <p className="text-gray-600 leading-relaxed">
                  &ldquo;Quality education is the only path to the salvation of
                  nations&rdquo; - a principle that guides our commitment to
                  shaping a brighter, more equitable future for generations to
                  come.
                </p>
              </div>
            </div>
          </Card>
        </ResponsiveGrid>
      </Section>

      {/* Enhanced Values Section */}
      <Section
        background="gray"
        padding="xl"
        className="scroll-mt-20 bg-gradient-to-br from-gray-50 via-white to-secondary-50/20"
        id="mission-values"
      >
        <div className="text-center mb-20">
          <div className="inline-flex items-center px-4 py-2 rounded-full bg-gradient-to-r from-secondary-100 to-primary-100 border border-secondary-200/50 mb-6">
            <span className="text-xs sm:text-sm font-semibold tracking-wide bg-gradient-to-r from-secondary-600 to-primary-600 bg-clip-text text-transparent">
              Our Values
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-8">
            What We{" "}
            <span className="bg-gradient-to-r from-secondary-600 to-primary-600 bg-clip-text text-transparent">
              Stand For
            </span>
          </h2>
          <div className="w-24 h-1 bg-gradient-to-r from-secondary-500 to-primary-500 mx-auto mb-8 rounded-full"></div>
          <p className="text-base sm:text-lg md:text-xl leading-relaxed text-gray-600 max-w-4xl mx-auto">
            Our core values guide every decision and shape the character of our{" "}
            <span className="font-semibold text-primary-600">
              educational community
            </span>
            .
          </p>
        </div>

        <ResponsiveGrid cols={{ default: 1, sm: 2, lg: 4 }} gap="lg">
          {values.map((value, index) => {
            const IconComponent = value.icon;
            return (
              <div key={index} className="group">
                <Card className="relative p-6 sm:p-8 text-center h-full hover:shadow-2xl transition-all duration-500 hover:scale-105 bg-white/80 backdrop-blur-sm border border-gray-200/50 overflow-hidden flex flex-col">
                  {/* Gradient border effect */}
                  <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-primary-500 to-secondary-500 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>

                  <div
                    className={`relative w-16 h-16 sm:w-18 sm:h-18 lg:w-20 lg:h-20 mx-auto mb-6 bg-gradient-to-br from-gray-100 to-gray-50 rounded-2xl flex items-center justify-center shadow-lg group-hover:shadow-xl group-hover:scale-110 transition-all duration-300`}
                  >
                    <IconComponent
                      className={`w-8 h-8 sm:w-9 sm:h-9 lg:w-10 lg:h-10 ${value.color} group-hover:scale-110 transition-transform duration-300`}
                    />
                    <div className="absolute inset-0 bg-gradient-to-br from-primary-50 to-secondary-50 rounded-2xl opacity-0 group-hover:opacity-50 transition-opacity duration-300"></div>
                  </div>

                  <div className="flex-1 flex flex-col justify-start">
                    <h3 className="text-lg sm:text-xl font-bold text-gray-900 mb-4 group-hover:text-primary-600 transition-colors duration-300 leading-tight">
                      {value.title}
                    </h3>

                    <p className="text-gray-600 leading-relaxed text-sm sm:text-base group-hover:text-gray-700 transition-colors duration-300 text-left px-2">
                      {value.description}
                    </p>
                  </div>

                  {/* Background gradient effect */}
                  <div className="absolute inset-0 bg-gradient-to-br from-primary-50/0 via-transparent to-secondary-50/0 group-hover:from-primary-50/30 group-hover:to-secondary-50/30 transition-all duration-500 -z-10 rounded-xl"></div>
                </Card>
              </div>
            );
          })}
        </ResponsiveGrid>
      </Section>

      {/* Enhanced History & Milestones */}
      <Section
        background="white"
        padding="xl"
        className="scroll-mt-20 bg-gradient-to-br from-white via-primary-50/20 to-secondary-50/20 overflow-x-hidden"
        id="timeline"
      >
        {/* Header Section */}
        <div className="text-center mb-16">
          {/* Tag */}
          <div className="inline-flex items-center px-4 py-1.5 rounded-full bg-gradient-to-r from-primary-100 to-secondary-100 border border-primary-200/40 mb-5">
            <span className="text-sm font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Our Journey
            </span>
          </div>

          {/* Title */}
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-extrabold text-gray-900 mb-4 leading-tight">
            School History &{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Milestones
            </span>
          </h2>

          {/* Underline */}
          <div className="w-28 h-0.5 bg-gradient-to-r from-primary-500 to-secondary-500 mx-auto rounded-full mb-6"></div>

          {/* Subtitle */}
          <p className="text-base sm:text-lg lg:text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            From{" "}
            <span className="font-semibold text-primary-600">
              humble beginnings
            </span>{" "}
            to{" "}
            <span className="font-semibold text-secondary-600">
              educational excellence
            </span>{" "}
            — discover our journey of growth and achievement.
          </p>
        </div>

        <div className="relative max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Enhanced Central Timeline Line - Desktop */}
          <div
            className="hidden lg:block absolute left-1/2 transform -translate-x-1/2 w-2 bg-gradient-to-b from-primary-300 via-primary-500 to-secondary-500 rounded-full shadow-lg"
            style={{ height: "calc(100% - 2rem)" }}
          ></div>

          {/* Left Timeline Line - Mobile/Tablet */}
          <div
            className="lg:hidden absolute left-6 top-12 w-1 bg-gradient-to-b from-primary-200 via-primary-500 to-primary-200 rounded-full"
            style={{ height: "calc(100% - 2rem)" }}
          ></div>

          <div className="space-y-8 lg:space-y-0">
            {milestones.map((milestone, index) => (
              <div
                key={index}
                className={`relative flex flex-col lg:flex-row ${
                  index % 2 === 0 ? "lg:flex-row" : "lg:flex-row-reverse"
                } lg:items-start`}
              >
                {/* Timeline Dot (no year text) */}
                <div
                  className={`absolute lg:left-1/2 -translate-x-1/2 w-6 h-6 lg:w-7 lg:h-7 bg-gradient-to-br from-primary-500 to-secondary-600 rounded-full shadow-md ring-4 ring-white hover:scale-110 transition-transform duration-300`}
                  style={{ top: "2.5rem" }}
                >
                  <div className="w-2.5 h-2.5 bg-white rounded-full m-auto"></div>
                </div>

                {/* Card */}
                <div
                  className={`w-full lg:w-[45%] mt-12 lg:mt-0 ${
                    index % 2 === 0 ? "lg:pr-10" : "lg:pl-10"
                  }`}
                >
                  <div className="group relative bg-white rounded-xl border border-primary-50 shadow hover:shadow-2xl transition-all duration-500 hover:-translate-y-1.5">
                    <div className="relative p-5 sm:p-7 lg:p-8 space-y-5">
                      {/* Year Badge */}
                      <span className="inline-flex items-center px-3 py-1 text-xs font-bold rounded-full bg-gradient-to-r from-primary-500 to-secondary-600 text-white shadow">
                        <Calendar className="w-3 h-3 mr-2" />
                        {milestone.year}
                      </span>

                      {/* Title */}
                      <h3 className="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-900 group-hover:text-primary-600 transition-colors duration-300">
                        {milestone.title}
                      </h3>

                      {/* Description */}
                      <p className="text-gray-600 leading-relaxed text-sm sm:text-base">
                        {milestone.description}
                      </p>

                      {/* Footer Dots */}
                      <div className="flex items-center justify-between pt-2">
                        <div className="flex space-x-2 opacity-60 group-hover:opacity-100 transition-opacity duration-300">
                          <div className="w-2 h-2 bg-gradient-to-r from-primary-400 to-primary-500 rounded-full"></div>
                          <div className="w-1.5 h-1.5 bg-gradient-to-r from-primary-300 to-primary-400 rounded-full mt-0.5"></div>
                          <div className="w-1 h-1 bg-gradient-to-r from-primary-200 to-primary-300 rounded-full mt-1"></div>
                        </div>
                        <ArrowRight className="w-5 h-5 text-primary-500 opacity-0 group-hover:opacity-100 transform translate-x-2 group-hover:translate-x-0 transition-all duration-300" />
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </Section>

      {/* Empowering Women And Youth Section */}
      <Section
        background="gray"
        padding="lg"
        className="scroll-mt-20 bg-gradient-to-br from-gray-50 via-white to-primary-50/20"
        id="empowerment"
      >
        <div className="text-center mb-12 sm:mb-16">
          <div className="inline-flex items-center px-4 py-2 rounded-full bg-gradient-to-r from-pink-100 to-purple-100 border border-pink-200/50 mb-6">
            <span className="text-sm font-semibold bg-gradient-to-r from-pink-600 to-purple-600 bg-clip-text text-transparent">
              Community Impact
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Empowering{" "}
            <span className="bg-gradient-to-r from-pink-600 to-purple-600 bg-clip-text text-transparent">
              Women And Youth
            </span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
            Setaragan Zirak School places a strong emphasis on empowerment and
            practical skill development for our community.
          </p>
        </div>

        <div className="max-w-5xl mx-auto">
          <div className="grid gap-8 lg:gap-12">
            <div className="relative group">
              <div className="absolute left-0 top-0 w-1 h-full bg-gradient-to-b from-pink-500 to-purple-500 rounded-full"></div>
              <div className="pl-8 p-6 bg-white/80 backdrop-blur-sm rounded-2xl border border-gray-100/50 shadow-lg hover:shadow-xl hover:border-pink-200/70 transition-all duration-300 hover:-translate-y-1">
                <div className="flex flex-col lg:flex-row lg:items-center gap-6">
                  <div className="lg:w-2/3">
                    <h3 className="text-2xl font-bold text-gray-900 mb-4">
                      Key Initiatives
                    </h3>
                    <div className="space-y-4">
                      <div className="p-4 bg-pink-50 rounded-lg">
                        <h4 className="font-semibold text-pink-800 mb-2">
                          Literacy courses for mothers
                        </h4>
                        <p className="text-gray-700">
                          Enabling them to acquire essential reading and writing
                          skills.
                        </p>
                      </div>
                      <div className="p-4 bg-purple-50 rounded-lg">
                        <h4 className="font-semibold text-purple-800 mb-2">
                          Sewing workshops for women
                        </h4>
                        <p className="text-gray-700">
                          Many have successfully launched small businesses with
                          the school&apos;s support, including the provision of
                          tools such as sewing machines and essential supplies.
                        </p>
                      </div>
                      <div className="p-4 bg-blue-50 rounded-lg">
                        <h4 className="font-semibold text-blue-800 mb-2">
                          COVID-19 Response Programs
                        </h4>
                        <p className="text-gray-700">
                          For students impacted by the pandemic, the school
                          offered remedial programs and free training in
                          computer skills, painting, photography, calligraphy,
                          and English language, equipping them with valuable
                          skills for personal and professional growth.
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="lg:w-1/3">
                    <Image
                      src="/images/about/about-2.jpg"
                      alt="Women empowerment and youth development programs"
                      width={400}
                      height={300}
                      className="rounded-xl shadow-lg object-cover w-full h-48 md:h-56 lg:h-48 object-center"
                      loading="lazy"
                    />
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </Section>

      {/* Educational Programs & Community Impact */}
      <Section
        background="white"
        padding="lg"
        className="scroll-mt-20 bg-gradient-to-br from-white via-gray-50/50 to-primary-50/20"
      >
        <div className="text-center mb-12 sm:mb-16">
          <div className="inline-flex items-center px-4 py-2 rounded-full bg-gradient-to-r from-primary-100 to-secondary-100 border border-primary-200/50 mb-6">
            <span className="text-sm font-semibold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Educational Programs
            </span>
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Empowering Through{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Education & Skills
            </span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
            Our comprehensive programs focus on both academic excellence and
            practical skill development for students and the community.
          </p>
        </div>

        <ResponsiveGrid cols={{ default: 1, lg: 2 }} gap="lg" className="mb-16">
          {/* Boys Programs */}
          <Card className="p-6 sm:p-8 h-full border-l-4 border-l-blue-500">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                  <Target className="w-6 h-6 text-blue-600" />
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-xl sm:text-2xl font-bold text-gray-900 mb-4">
                  Academic Excellence Programs
                </h3>
                <p className="text-gray-600 leading-relaxed mb-4">
                  Specialized courses designed to prepare students for higher
                  education and university entrance exams.
                </p>
                <ul className="text-gray-600 space-y-2">
                  <li>• Physics, Chemistry & Mathematics intensive courses</li>
                  <li>• University entrance exam (Konkur) preparation</li>
                  <li>• Advanced analytical and problem-solving skills</li>
                  <li>• Academic mentorship and guidance</li>
                </ul>
              </div>
            </div>
            <div className="mt-6">
              <Image
                src="/images/about/ab5.jpg"
                alt="Academic programs and student activities"
                width={400}
                height={250}
                className="rounded-lg shadow-md object-cover w-full h-40 md:h-48 lg:h-40 object-center"
                loading="lazy"
              />
            </div>
          </Card>

          {/* Girls Programs & Community */}
          <Card className="p-6 sm:p-8 h-full border-l-4 border-l-pink-500">
            <div className="flex items-start space-x-4">
              <div className="flex-shrink-0">
                <div className="w-12 h-12 bg-pink-100 rounded-lg flex items-center justify-center">
                  <Heart className="w-6 h-6 text-pink-600" />
                </div>
              </div>
              <div className="flex-1">
                <h3 className="text-xl sm:text-2xl font-bold text-gray-900 mb-4">
                  Empowerment & Life Skills
                </h3>
                <p className="text-gray-600 leading-relaxed mb-4">
                  Comprehensive programs focused on practical skills, language
                  development, and women&apos;s empowerment.
                </p>
                <ul className="text-gray-600 space-y-2">
                  <li>• English language training and communication</li>
                  <li>• Computer skills and digital literacy</li>
                  <li>• Sewing workshops and entrepreneurship</li>
                  <li>• Literacy programs for mothers and community</li>
                </ul>
              </div>
            </div>
            <div className="mt-6">
              <Image
                src="/images/about/about-2.jpg"
                alt="Women empowerment and community programs"
                width={400}
                height={250}
                className="rounded-lg shadow-md object-cover w-full h-40 md:h-48 lg:h-40 object-center"
                loading="lazy"
              />
            </div>
          </Card>
        </ResponsiveGrid>

        {/* Special Programs Highlight */}
        <div className="bg-gradient-to-r from-primary-50 to-secondary-50 rounded-2xl p-8 border border-primary-100">
          <div className="text-center mb-8">
            <h3 className="text-2xl sm:text-3xl font-bold text-gray-900 mb-4">
              Winter Educational Programs (2019-2021)
            </h3>
            <p className="text-gray-600 leading-relaxed">
              During challenging times, we provided free educational support to
              hundreds of students:
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 text-center">
            <div className="bg-white rounded-xl p-6 shadow-sm">
              <div className="text-3xl font-bold text-primary-600 mb-2">
                530
              </div>
              <div className="text-sm text-gray-600">
                Students supported in 2019
              </div>
              <div className="text-xs text-gray-500 mt-1">Grades 1-9</div>
            </div>
            <div className="bg-white rounded-xl p-6 shadow-sm">
              <div className="text-3xl font-bold text-secondary-600 mb-2">
                320
              </div>
              <div className="text-sm text-gray-600">
                Female students in 2020
              </div>
              <div className="text-xs text-gray-500 mt-1">
                Grades 7-12, COVID response
              </div>
            </div>
            <div className="bg-white rounded-xl p-6 shadow-sm">
              <div className="text-3xl font-bold text-green-600 mb-2">850+</div>
              <div className="text-sm text-gray-600">
                Total students across 3 years
              </div>
              <div className="text-xs text-gray-500 mt-1">
                Math, Physics, Chemistry, English
              </div>
            </div>
          </div>
        </div>
      </Section>

      {/* Graduate Testimonials */}
      <Section
        background="white"
        padding="lg"
        className="scroll-mt-20"
        id="testimonials"
      >
        <div className="text-center mb-12 sm:mb-16">
          <Badge className="mb-4">Student Success</Badge>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            First-Generation{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Graduate Testimonials
            </span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
            Hear from our first graduates about how Zirak High School shaped
            their lives and futures.
          </p>
        </div>

        <ResponsiveGrid cols={{ default: 1, lg: 3 }} gap="lg">
          <Card className="p-6 sm:p-8 h-full">
            <div className="mb-6">
              <div className="w-16 h-16 bg-primary-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Users className="w-8 h-8 text-primary-600" />
              </div>
              <h3 className="text-xl font-bold text-gray-900 text-center">
                Nooria
              </h3>
              <p className="text-center text-gray-600">Graduate</p>
            </div>
            <blockquote className="text-gray-600 leading-relaxed">
              &ldquo;Zirak high school has formed a fundamental part of my early
              education. Growing up in a traditional society in a war-torn
              country, school was the hope for a brighter future for many young
              girls like me. I would go to school smiling and come home happy
              and positive every day.&rdquo;
            </blockquote>
          </Card>

          <Card className="p-6 sm:p-8 h-full">
            <div className="mb-6">
              <div className="w-16 h-16 bg-secondary-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Award className="w-8 h-8 text-secondary-600" />
              </div>
              <h3 className="text-xl font-bold text-gray-900 text-center">
                Mohammad Hossein Hemat
              </h3>
              <p className="text-center text-gray-600">Project Manager</p>
            </div>
            <blockquote className="text-gray-600 leading-relaxed">
              &ldquo;In a very critical situation where there was a lack of
              school in Jaghori, Ghazni that had made the generation before me
              illiterate, Zirak High School was built. The facilities in
              suitable atmosphere of the school created enthusiasm and
              motivation in me and many others to study and try hard.&rdquo;
            </blockquote>
          </Card>

          <Card className="p-6 sm:p-8 h-full">
            <div className="mb-6">
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Target className="w-8 h-8 text-green-600" />
              </div>
              <h3 className="text-xl font-bold text-gray-900 text-center">
                Mohammad Jalil Ebrahimi
              </h3>
              <p className="text-center text-gray-600">Graduate</p>
            </div>
            <blockquote className="text-gray-600 leading-relaxed">
              &ldquo;Zirak High School has played a very crucial and valuable
              role for me in my journey of life. I would like to express my
              thanks and appreciation for the honest, most treasured, and
              compassionate efforts that Zirak High School and Mr. Ezatullah
              Salehi have done for me and all people like me.&rdquo;
            </blockquote>
          </Card>
        </ResponsiveGrid>
      </Section>

      {/* Contact CTA */}
      <Section background="white" padding="lg">
        <div className="text-center">
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Ready to Join{" "}
            <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
              Our Community?
            </span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
            Discover how Setaragan Zirak School can provide your child with an
            exceptional educational experience.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <a
              href="/contact"
              className="inline-flex items-center gap-3 bg-gradient-to-r from-primary-600 to-secondary-600 text-white px-8 py-4 rounded-xl font-bold text-lg hover:from-primary-700 hover:to-secondary-700 transition-all duration-200 hover:scale-105 shadow-lg hover:shadow-xl"
            >
              <MapPin className="w-5 h-5" />
              Contact Us
            </a>
            <a
              href="/admissions"
              className="inline-flex items-center gap-3 border-2 border-gray-300 text-gray-700 hover:border-primary-300 hover:text-primary-600 px-8 py-4 rounded-xl font-bold text-lg transition-all duration-200 hover:scale-105"
            >
              <Users className="w-5 h-5" />
              Learn About Admissions
            </a>
          </div>
        </div>
      </Section>
      </div>
    </AppLayout>
  );
};

export default AboutPage;
