import React from 'react';
import { Metadata } from 'next';
import Link from 'next/link';
import { 
  BookOpen, 
  Users, 
  FileText, 
  CheckCircle, 
  Phone, 
  Mail, 
  MapPin, 
  Clock,
  Award,
  Heart,
  Star,
  GraduationCap,
  UserCheck,
  DollarSign,
  AlertCircle
} from 'lucide-react';
import Card from '@/components/common/ui/card';
import Container from '@/components/common/ui/container';
import AdmissionsForm from '@/components/common/forms/AdmissionsForm';
import { Section, ResponsiveGrid, AppLayout } from '@/components/common/layout';

export const metadata: Metadata = {
  title: 'Admissions - Join Our Community | Setaragan Zirak School',
  description: 'Apply to Setaragan Zirak Private School. Learn about our admission process, requirements, scholarships, and how to secure your place in our exceptional educational community.',
  keywords: 'Setaragan Zirak admissions, school application, enrollment, admission requirements, scholarships, private school Kabul',
};

export default function AdmissionsPage() {
  const admissionSteps = [
    {
      step: 1,
      title: "Initial Application",
      description: "Submit your application form with required documents",
      icon: FileText,
      timeframe: "Ongoing",
      details: [
        "Complete application form",
        "Submit academic transcripts",
        "Provide identification documents",
        "Include recent photographs"
      ]
    },
    {
      step: 2,
      title: "Assessment & Interview",
      description: "Academic evaluation and family interview",
      icon: UserCheck,
      timeframe: "Within 1 week",
      details: [
        "Academic assessment test",
        "Student interview",
        "Parent/guardian meeting",
        "School tour opportunity"
      ]
    },
    {
      step: 3,
      title: "Application Review",
      description: "Our admission committee reviews your application",
      icon: CheckCircle,
      timeframe: "3-5 business days",
      details: [
        "Academic performance review",
        "Assessment result evaluation",
        "Family interview feedback",
        "Available capacity consideration"
      ]
    },
    {
      step: 4,
      title: "Decision & Enrollment",
      description: "Receive admission decision and complete enrollment",
      icon: GraduationCap,
      timeframe: "Upon acceptance",
      details: [
        "Admission decision notification",
        "Fee payment and scheduling",
        "Uniform and supply information",
        "Orientation session booking"
      ]
    }
  ];

  const requiredDocuments = [
    {
      document: "Completed Application Form",
      description: "Official school application with all sections filled",
      required: true,
      icon: FileText
    },
    {
      document: "Academic Transcripts",
      description: "Previous school records and grade reports",
      required: true,
      icon: Award
    },
    {
      document: "Birth Certificate",
      description: "Official birth certificate or equivalent ID",
      required: true,
      icon: UserCheck
    },
    {
      document: "Passport Photos",
      description: "Recent passport-size photographs (2-4 copies)",
      required: true,
      icon: Users
    },
    {
      document: "Medical Records",
      description: "Basic health information and vaccination records",
      required: false,
      icon: Heart
    },
    {
      document: "Parent/Guardian ID",
      description: "Valid identification of parent or legal guardian",
      required: true,
      icon: Users
    }
  ];

  const gradeRequirements = [
    {
      grade: "Grades 1-3 (Primary)",
      age: "Ages 6-8",
      requirements: [
        "Basic alphabet and number recognition",
        "Simple communication skills",
        "Toilet training independence",
        "Social interaction readiness"
      ],
      tuition: "Contact for current rates"
    },
    {
      grade: "Grades 4-6 (Elementary)",
      age: "Ages 9-11",
      requirements: [
        "Basic reading and writing skills",
        "Fundamental math concepts",
        "Previous school experience",
        "Academic assessment completion"
      ],
      tuition: "Contact for current rates"
    },
    {
      grade: "Grades 7-9 (Middle School)",
      age: "Ages 12-14",
      requirements: [
        "Strong foundation in core subjects",
        "Academic transcript review",
        "Placement exam completion",
        "Interview with academic counselor"
      ],
      tuition: "Contact for current rates"
    },
    {
      grade: "Grades 10-12 (High School)",
      age: "Ages 15-17",
      requirements: [
        "Completed middle school education",
        "Entrance examination",
        "Academic performance review",
        "Career counseling session"
      ],
      tuition: "Contact for current rates"
    }
  ];

  const scholarshipPrograms = [
    {
      title: "Academic Excellence Scholarship",
      description: "Merit-based support for outstanding academic performance",
      coverage: "Up to 50% tuition reduction",
      criteria: [
        "Exceptional academic record",
        "Standardized test scores",
        "Teacher recommendations",
        "Community involvement"
      ],
      icon: Award
    },
    {
      title: "Need-Based Financial Aid",
      description: "Support for families demonstrating financial need",
      coverage: "Varies based on need assessment",
      criteria: [
        "Family income verification",
        "Financial hardship documentation",
        "Academic potential demonstration",
        "Community ties consideration"
      ],
      icon: Heart
    },
    {
      title: "Sibling Discount",
      description: "Special rates for families with multiple students",
      coverage: "10-20% reduction per additional sibling",
      criteria: [
        "Multiple children enrolled",
        "Good academic standing",
        "Regular attendance record",
        "Fee payment compliance"
      ],
      icon: Users
    }
  ];

  return (
    <AppLayout>
    <div className="min-h-screen bg-white">
      {/* Hero Section */}
      <Section 
        background="white" 
        padding="xl" 
        className="bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600 text-white relative overflow-hidden"
      >
        {/* Background Pattern */}
        <div className="absolute inset-0 opacity-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-white rounded-full mix-blend-overlay filter blur-xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-secondary-300 rounded-full mix-blend-overlay filter blur-xl animate-pulse delay-1000"></div>
        </div>
        
        <Container className="relative">
          <div className="text-center max-w-4xl mx-auto">
            <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-6xl font-bold mb-6 leading-tight">
              Join Our{" "}
              <span className="text-transparent bg-clip-text bg-gradient-to-r from-yellow-200 to-yellow-100">
                Educational Journey
              </span>
            </h1>
            <p className="text-xl sm:text-2xl text-white/90 max-w-3xl mx-auto leading-relaxed mb-8">
              Take the first step towards excellence in education. Discover how Setaragan Zirak School 
              can provide your child with world-class learning opportunities and personal growth.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <a
                href="#application-form"
                className="inline-flex items-center gap-3 bg-white text-primary-600 px-8 py-4 rounded-xl font-bold text-lg hover:bg-gray-100 transition-all duration-200 hover:scale-105 shadow-lg hover:shadow-xl"
              >
                <FileText className="w-5 h-5" />
                Apply Now
              </a>
              <a
                href="#admission-process"
                className="inline-flex items-center gap-3 border-2 border-white/30 text-white hover:border-white hover:bg-white/10 px-8 py-4 rounded-xl font-bold text-lg transition-all duration-200 hover:scale-105"
              >
                <BookOpen className="w-5 h-5" />
                Learn More
              </a>
            </div>
          </div>
        </Container>
      </Section>

      {/* Quick Stats */}
      <Section background="white" padding="md" className="border-b border-gray-100">
        <Container>
          <ResponsiveGrid cols={{ default: 2, sm: 4 }} gap="md">
            <div className="text-center">
              <div className="text-3xl sm:text-4xl font-bold text-primary-600 mb-2">500+</div>
              <div className="text-gray-600 font-medium">Current Students</div>
            </div>
            <div className="text-center">
              <div className="text-3xl sm:text-4xl font-bold text-secondary-600 mb-2">95%</div>
              <div className="text-gray-600 font-medium">Satisfaction Rate</div>
            </div>
            <div className="text-center">
              <div className="text-3xl sm:text-4xl font-bold text-primary-600 mb-2">30+</div>
              <div className="text-gray-600 font-medium">Expert Teachers</div>
            </div>
            <div className="text-center">
              <div className="text-3xl sm:text-4xl font-bold text-secondary-600 mb-2">6</div>
              <div className="text-gray-600 font-medium">Years Excellence</div>
            </div>
          </ResponsiveGrid>
        </Container>
      </Section>

      {/* Admission Process */}
      <Section id="admission-process" background="gray" padding="xl">
        <Container>
          <div className="text-center mb-12 sm:mb-16">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Simple{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Admission Process
              </span>
            </h2>
            <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
              We&apos;ve designed a straightforward admission process to make joining our school community as smooth as possible.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 lg:gap-8">
            {admissionSteps.map((step) => (
              <Card key={step.step} className="p-6 sm:p-8 text-center relative overflow-hidden group hover:shadow-2xl transition-all duration-300">
                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-primary-500 to-secondary-500"></div>
                
                <div className="w-16 h-16 bg-gradient-to-br from-primary-100 to-secondary-100 rounded-full flex items-center justify-center mx-auto mb-6 group-hover:scale-110 transition-transform duration-300">
                  <step.icon className="w-8 h-8 text-primary-600" />
                </div>
                
                <div className="mb-4">
                  <div className="inline-flex items-center justify-center w-8 h-8 bg-primary-600 text-white rounded-full text-sm font-bold mb-3">
                    {step.step}
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-2">
                    {step.title}
                  </h3>
                  <p className="text-gray-600 mb-4">
                    {step.description}
                  </p>
                  <div className="inline-flex items-center gap-2 bg-primary-50 text-primary-700 px-3 py-1 rounded-full text-sm font-medium">
                    <Clock className="w-4 h-4" />
                    {step.timeframe}
                  </div>
                </div>

                <div className="text-left">
                  <h4 className="font-semibold text-gray-900 mb-2">Key Activities:</h4>
                  <ul className="text-sm text-gray-600 space-y-1">
                    {step.details.map((detail, idx) => (
                      <li key={idx} className="flex items-start gap-2">
                        <div className="w-1.5 h-1.5 bg-primary-400 rounded-full mt-2 flex-shrink-0"></div>
                        {detail}
                      </li>
                    ))}
                  </ul>
                </div>
              </Card>
            ))}
          </div>
        </Container>
      </Section>

      {/* Grade Requirements */}
      <Section background="white" padding="xl">
        <Container>
          <div className="text-center mb-12 sm:mb-16">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Grade Level{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Requirements
              </span>
            </h2>
            <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
              Each grade level has specific requirements to ensure students are prepared for academic success.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 lg:gap-8">
            {gradeRequirements.map((grade, index) => (
              <Card key={index} className="p-6 sm:p-8 hover:shadow-xl transition-shadow duration-300">
                <div className="flex items-start justify-between mb-6">
                  <div>
                    <h3 className="text-xl font-bold text-gray-900 mb-2">
                      {grade.grade}
                    </h3>
                    <div className="inline-flex items-center gap-2 bg-secondary-50 text-secondary-700 px-3 py-1 rounded-full text-sm font-medium">
                      <Users className="w-4 h-4" />
                      {grade.age}
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="text-sm text-gray-500 mb-1">Tuition</div>
                    <div className="font-semibold text-primary-600">{grade.tuition}</div>
                  </div>
                </div>

                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Entry Requirements:</h4>
                  <ul className="space-y-2">
                    {grade.requirements.map((requirement, idx) => (
                      <li key={idx} className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span className="text-gray-600">{requirement}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </Card>
            ))}
          </div>
        </Container>
      </Section>

      {/* Required Documents */}
      <Section background="gray" padding="xl">
        <Container>
          <div className="text-center mb-12 sm:mb-16">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Required{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Documents
              </span>
            </h2>
            <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
              Please prepare the following documents for your application. Required documents are marked with a star.
            </p>
          </div>

          <ResponsiveGrid cols={{ default: 1, sm: 2, lg: 3 }} gap="lg">
            {requiredDocuments.map((doc, index) => (
              <Card key={index} className="p-6 hover:shadow-lg transition-shadow duration-300">
                <div className="flex items-start gap-4">
                  <div className="w-12 h-12 bg-primary-100 rounded-lg flex items-center justify-center flex-shrink-0">
                    <doc.icon className="w-6 h-6 text-primary-600" />
                  </div>
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-2">
                      <h3 className="font-semibold text-gray-900">
                        {doc.document}
                      </h3>
                      {doc.required && (
                        <Star className="w-4 h-4 text-yellow-500 fill-current" />
                      )}
                    </div>
                    <p className="text-sm text-gray-600">
                      {doc.description}
                    </p>
                    <div className="mt-3">
                      <span className={`inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs font-medium ${
                        doc.required 
                          ? 'bg-red-50 text-red-700' 
                          : 'bg-green-50 text-green-700'
                      }`}>
                        {doc.required ? (
                          <>
                            <AlertCircle className="w-3 h-3" />
                            Required
                          </>
                        ) : (
                          <>
                            <CheckCircle className="w-3 h-3" />
                            Optional
                          </>
                        )}
                      </span>
                    </div>
                  </div>
                </div>
              </Card>
            ))}
          </ResponsiveGrid>
        </Container>
      </Section>

      {/* Scholarship Programs */}
      <Section background="white" padding="xl">
        <Container>
          <div className="text-center mb-12 sm:mb-16">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Scholarship{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Opportunities
              </span>
            </h2>
            <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
              We believe that financial circumstances should not limit access to quality education. 
              Explore our scholarship and financial aid programs.
            </p>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 lg:gap-8">
            {scholarshipPrograms.map((program, index) => (
              <Card key={index} className="p-6 sm:p-8 hover:shadow-xl transition-all duration-300 group">
                <div className="text-center mb-6">
                  <div className="w-16 h-16 bg-gradient-to-br from-primary-100 to-secondary-100 rounded-full flex items-center justify-center mx-auto mb-4 group-hover:scale-110 transition-transform duration-300">
                    <program.icon className="w-8 h-8 text-primary-600" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-2">
                    {program.title}
                  </h3>
                  <p className="text-gray-600 mb-4">
                    {program.description}
                  </p>
                  <div className="inline-flex items-center gap-2 bg-gradient-to-r from-primary-50 to-secondary-50 text-primary-700 px-4 py-2 rounded-full font-semibold">
                    <DollarSign className="w-4 h-4" />
                    {program.coverage}
                  </div>
                </div>

                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">Eligibility Criteria:</h4>
                  <ul className="space-y-2">
                    {program.criteria.map((criterion, idx) => (
                      <li key={idx} className="flex items-start gap-3">
                        <CheckCircle className="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" />
                        <span className="text-gray-600 text-sm">{criterion}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </Card>
            ))}
          </div>
        </Container>
      </Section>

      {/* Application Form Section */}
      <Section id="application-form" background="white" padding="xl" className="bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600 text-white">
        <Container>
          <div className="max-w-4xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold mb-6">
                Start Your Application Today
              </h2>
              <p className="text-xl text-white/90 leading-relaxed">
                Ready to join our educational community? Choose how you&apos;d like to begin your application process.
              </p>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {/* Contact Form Option */}
              <Card className="p-8 bg-white/10 backdrop-blur-sm border border-white/20">
                <div className="text-center mb-6">
                  <div className="w-16 h-16 bg-white/20 rounded-full flex items-center justify-center mx-auto mb-4">
                    <FileText className="w-8 h-8 text-white" />
                  </div>
                  <h3 className="text-xl font-bold text-white mb-3">
                    Online Application
                  </h3>
                  <p className="text-white/90">
                    Submit your application through our form
                  </p>
                </div>

                <AdmissionsForm className="text-gray-900" />
              </Card>

              {/* Direct Contact Options */}
              <Card className="p-8 bg-white/10 backdrop-blur-sm border border-white/20">
                <div className="text-center mb-6">
                  <div className="w-16 h-16 bg-white/20 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Phone className="w-8 h-8 text-white" />
                  </div>
                  <h3 className="text-xl font-bold text-white mb-3">
                    Direct Contact
                  </h3>
                  <p className="text-white/90">
                    Prefer to speak with our admissions team directly?
                  </p>
                </div>

                <div className="space-y-6">
                  {/* Phone Contact */}
                  <a 
                    href="tel:+93785675793"
                    className="flex items-center gap-4 p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors duration-300 group"
                  >
                    <div className="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                      <Phone className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <div className="font-semibold text-white">Call Us</div>
                      <div className="text-white/90 text-sm">+93 785 675 793</div>
                    </div>
                  </a>

                  {/* WhatsApp Contact */}
                  <a 
                    href="https://wa.me/93785675793?text=I am interested in admission to Setaragan Zirak School. Please provide me with information about the application process."
                    target="_blank"
                    rel="noopener noreferrer"
                    className="flex items-center gap-4 p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors duration-300 group"
                  >
                    <div className="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                      <svg className="w-6 h-6 text-white" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.890-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.488"/>
                      </svg>
                    </div>
                    <div>
                      <div className="font-semibold text-white">WhatsApp</div>
                      <div className="text-white/90 text-sm">Quick response chat</div>
                    </div>
                  </a>

                  {/* Email Contact */}
                  <a 
                    href="mailto:setarganzirak98@gmail.com?subject=Admission Inquiry&body=I am interested in learning more about admission to Setaragan Zirak School. Please provide me with information about the application process and requirements."
                    className="flex items-center gap-4 p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors duration-300 group"
                  >
                    <div className="w-12 h-12 bg-primary-400 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                      <Mail className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <div className="font-semibold text-white">Email Us</div>
                      <div className="text-white/90 text-sm">setarganzirak98@gmail.com</div>
                    </div>
                  </a>

                  {/* Visit Us */}
                  <Link 
                    href="/contact"
                    className="flex items-center gap-4 p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors duration-300 group"
                  >
                    <div className="w-12 h-12 bg-secondary-500 rounded-full flex items-center justify-center group-hover:scale-110 transition-transform duration-300">
                      <MapPin className="w-6 h-6 text-white" />
                    </div>
                    <div>
                      <div className="font-semibold text-white">Visit Our Campus</div>
                      <div className="text-white/90 text-sm">Schedule a tour</div>
                    </div>
                  </Link>
                </div>
              </Card>
            </div>
          </div>
        </Container>
      </Section>

      {/* FAQ Section */}
      <Section background="white" padding="xl">
        <Container>
          <div className="text-center mb-12 sm:mb-16">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Frequently Asked{" "}
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                Questions
              </span>
            </h2>
            <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
              Get answers to common questions about our admission process and school policies.
            </p>
          </div>

          <div className="max-w-4xl mx-auto space-y-6">
            <Card className="p-6 sm:p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-3">
                When can I apply for admission?
              </h3>
              <p className="text-gray-600 leading-relaxed">
                We accept applications year-round, though we recommend applying at least 2-3 months before your desired start date. 
                Our primary enrollment periods are before the start of each academic year (typically March-April) and mid-year (August-September).
              </p>
            </Card>

            <Card className="p-6 sm:p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-3">
                What is the typical timeline for the admission process?
              </h3>
              <p className="text-gray-600 leading-relaxed">
                From initial application to final decision, the process typically takes 2-3 weeks. This includes document review, 
                assessment, interview, and committee decision. We&apos;ll keep you informed at each step of the process.
              </p>
            </Card>

            <Card className="p-6 sm:p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-3">
                Are there entrance examinations for all grade levels?
              </h3>
              <p className="text-gray-600 leading-relaxed">
                Assessment requirements vary by grade level. Younger students (grades 1-3) have informal assessments focusing on 
                readiness skills, while older students (grades 7-12) take more formal academic assessments. All assessments are 
                designed to ensure proper grade placement, not to exclude students.
              </p>
            </Card>

            <Card className="p-6 sm:p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-3">
                What financial aid options are available?
              </h3>
              <p className="text-gray-600 leading-relaxed">
                We offer merit-based scholarships, need-based financial aid, and sibling discounts. Financial aid decisions are 
                based on academic potential, financial need, and available resources. We encourage all families to apply regardless 
                of their financial situation.
              </p>
            </Card>

            <Card className="p-6 sm:p-8">
              <h3 className="text-xl font-bold text-gray-900 mb-3">
                Can students transfer from other schools mid-year?
              </h3>
              <p className="text-gray-600 leading-relaxed">
                Yes, we accept transfer students throughout the academic year, subject to space availability. Transfer students 
                go through the same admission process, and we work closely with families to ensure a smooth transition.
              </p>
            </Card>
          </div>
        </Container>
      </Section>

      {/* Final CTA */}
      <Section background="white" padding="lg" className="bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600 text-white">
        <Container>
          <div className="text-center max-w-4xl mx-auto">
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold mb-6">
              Ready to Begin This Journey?
            </h2>
            <p className="text-xl text-white/90 mb-8 leading-relaxed">
              Join the hundreds of families who have trusted Setaragan Zirak School with their children&apos;s education. 
              Your child&apos;s bright future starts here.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <a
                href="#application-form"
                className="inline-flex items-center gap-3 bg-white text-primary-600 px-8 py-4 rounded-xl font-bold text-lg hover:bg-gray-100 transition-all duration-200 hover:scale-105 shadow-lg hover:shadow-xl"
              >
                <FileText className="w-5 h-5" />
                Start Application
              </a>
              <Link
                href="/facilities"
                className="inline-flex items-center gap-3 border-2 border-white/30 text-white hover:border-white hover:bg-white/10 px-8 py-4 rounded-xl font-bold text-lg transition-all duration-200 hover:scale-105"
              >
                <MapPin className="w-5 h-5" />
                Schedule Campus Tour
              </Link>
            </div>
          </div>
        </Container>
      </Section>
      </div>
    </AppLayout>
  );
}