import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import * as Yup from 'yup';
import { requireAdminAuth } from '../../auth';

// Validation schema for manual donation
const manualDonationSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(2),
  email: Yup.string().email().required('Email is required'),
  phone: Yup.string().optional(),
  country: Yup.string().optional(),
  amount: Yup.number().required('Amount is required').positive(),
  currency: Yup.string().required('Currency is required'),
  category: Yup.string().optional(),
  paymentMethod: Yup.string().required('Payment method is required'),
  paymentStatus: Yup.string()
    .required('Payment status is required')
    .oneOf(['pending', 'completed', 'failed']),
  anonymous: Yup.boolean().optional(),
  message: Yup.string().optional(),
  transactionId: Yup.string().optional(),
});

export async function POST(request: NextRequest) {
  try {
    // Check admin auth
    const authCheck = await requireAdminAuth();
    if (authCheck.error) {
      return authCheck.response;
    }

    const body = await request.json();

    // Validate request body
    const validatedData = await manualDonationSchema.validate(body);

    // Create donation record
    const donation = await prisma.donation.create({
      data: {
        name: validatedData.name,
        email: validatedData.email,
        phone: validatedData.phone || null,
        country: validatedData.country || null,
        amount: validatedData.amount,
        currency: validatedData.currency,
        category: validatedData.category || 'General',
        paymentMethod: validatedData.paymentMethod,
        paymentStatus: validatedData.paymentStatus as 'pending' | 'completed' | 'failed',
        anonymous: validatedData.anonymous || false,
        message: validatedData.message || null,
        transactionId: validatedData.transactionId || null,
      },
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Donation added successfully',
        donation,
      },
      { status: 201 }
    );
  } catch (error) {
    if (error instanceof Yup.ValidationError) {
      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors: error.errors,
        },
        { status: 400 }
      );
    }

    console.error('Error creating manual donation:', error);
    return NextResponse.json(
      {
        success: false,
        message: error instanceof Error ? error.message : 'Failed to create donation',
      },
      { status: 500 }
    );
  }
}
