import { NextRequest } from 'next/server';
import { prisma } from '@/lib/prisma';
import { withPermission, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

// GET /api/admin/hero-slides - List all hero slides
export async function GET() {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('settings', 'view');
    if (auth.error) return auth.response!;

    const slides = await prisma.heroSlide.findMany({
      orderBy: { order: 'asc' },
    });

    return createSuccessResponse(slides);
  } catch (error) {
    console.error('Error fetching hero slides:', error);
    return createErrorResponse('Failed to fetch hero slides', 500);
  }
}

// POST /api/admin/hero-slides - Create new hero slide
export async function POST(request: NextRequest) {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('settings', 'manage');
    if (auth.error) return auth.response!;

    const body = await request.json();
    const { title, description, imageUrl, alt, order, active } = body;

    // Validation
    if (!title || !imageUrl || !alt) {
      return createErrorResponse('Title, image URL, and alt text are required', 400);
    }

    const slide = await prisma.heroSlide.create({
      data: {
        title,
        description: description || '',
        imageUrl,
        alt,
        order: order || 0,
        active: active !== undefined ? active : true,
      },
    });

    return createSuccessResponse(slide, 'Hero slide created successfully');
  } catch (error) {
    console.error('Error creating hero slide:', error);
    return createErrorResponse('Failed to create hero slide', 500);
  }
}

// DELETE /api/admin/hero-slides - Bulk delete hero slides
export async function DELETE(request: NextRequest) {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('settings', 'manage');
    if (auth.error) return auth.response!;

    const { searchParams } = new URL(request.url);
    const ids = searchParams.get('ids')?.split(',') || [];

    if (ids.length === 0) {
      return createErrorResponse('No IDs provided', 400);
    }

    await prisma.heroSlide.deleteMany({
      where: { id: { in: ids } },
    });

    return createSuccessResponse({ message: `Deleted ${ids.length} slide(s)` });
  } catch (error) {
    console.error('Error deleting hero slides:', error);
    return createErrorResponse('Failed to delete hero slides', 500);
  }
}
