import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { leadershipSchema } from '@/lib/schemas';
import { withAdminAuth, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

// GET /api/admin/leadership - Fetch all leadership members
export async function GET() {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const leadership = await prisma.leadership.findMany({
      orderBy: { order: 'asc' },
    });

    return createSuccessResponse(leadership, 'Leadership members fetched successfully');
  } catch (error) {
    console.error('Leadership GET error:', error);
    return createErrorResponse(
      error instanceof Error ? error.message : 'Failed to fetch leadership',
      500
    );
  }
}

// POST /api/admin/leadership - Create new leadership member
export async function POST(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const body = await request.json();

    // Validate data
    let validatedData;
    try {
      validatedData = await leadershipSchema.validate(body, { abortEarly: false });
    } catch (validationError) {
      const yupError = validationError as unknown as { inner?: Array<{ path: string; message: string }> };
      const errors = yupError.inner?.reduce((acc: Record<string, string>, err: { path: string; message: string }) => ({
        ...acc,
        [err.path]: err.message,
      }), {}) || {};
      
      return createErrorResponse('Validation failed', 400, errors);
    }

    // Get the highest order value
    const lastLeader = await prisma.leadership.findFirst({
      orderBy: { order: 'desc' },
      select: { order: true },
    });

    const nextOrder = (lastLeader?.order || 0) + 1;

    // Create leadership member
    const leadership = await prisma.leadership.create({
      data: {
        ...validatedData,
        order: nextOrder,
      },
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Leadership member created successfully',
        data: leadership,
      },
      { status: 201 }
    );
  } catch (error) {
    console.error('Leadership POST error:', error);
    return NextResponse.json(
      {
        success: false,
        message: error instanceof Error ? error.message : 'Failed to create leadership member',
      },
      { status: 500 }
    );
  }
}
