import { prisma } from "@/lib/prisma";
import { NextRequest, NextResponse } from "next/server";
import { withAdminAuth, createSuccessResponse, createErrorResponse } from "@/lib/admin/security";

export async function GET(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const { searchParams } = new URL(request.url);
    const page = parseInt(searchParams.get("page") || "1");
    const limit = parseInt(searchParams.get("limit") || "10");
    const search = searchParams.get("search") || "";
    const status = searchParams.get("status") || "all";

    const skip = (page - 1) * limit;

    // Build where clause
    const where: Record<string, unknown> = {};
    if (search) {
      where.OR = [
        { title: { contains: search } },
        { content: { contains: search } },
      ];
    }
    if (status !== "all") {
      where.status = status;
    }

    // Get total count
    const total = await prisma.newsArticle.count({ where });

    // Get articles
    const articles = await prisma.newsArticle.findMany({
      where,
      skip,
      take: limit,
      orderBy: { createdAt: "desc" },
    });

    return createSuccessResponse(
      {
        articles,
        pagination: {
          page,
          limit,
          total,
          pages: Math.ceil(total / limit),
        },
      },
      "News articles fetched successfully"
    );
  } catch (error) {
    console.error("Fetch news error:", error);
    return createErrorResponse("Internal server error", 500);
  }
}

export async function POST(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    // Get admin user's name
    const adminUser = await prisma.adminUser.findUnique({
      where: { id: auth.userId },
      select: { name: true },
    });

    if (!adminUser) {
      return createErrorResponse("Admin user not found", 400);
    }

    const { title, content, excerpt, status, slug, category, theme, readTime, featuredImage, tags, author, authorId } = await request.json();

    // Validate input
    if (!title || !content || !excerpt || !category) {
      return createErrorResponse("Title, content, excerpt, and category are required", 400);
    }

    // Use provided author name or default to current user's name
    const authorName = author || adminUser.name;
    
    // Use provided authorId if given, otherwise use current user's ID
    const createdByUserId = authorId || auth.userId;

    const article = await prisma.newsArticle.create({
      data: {
        title,
        content,
        excerpt,
        status: status || "draft",
        slug: slug || title.toLowerCase().replace(/\s+/g, "-"),
        author: authorName,
        authorId: createdByUserId,
        category,
        theme: theme || "primary",
        readTime: readTime || 3,
        image: featuredImage || null,
        tags: tags && Array.isArray(tags) && tags.length > 0 ? JSON.stringify(tags) : null,
      },
    });

    return createSuccessResponse(article, "Article created successfully", 201);
  } catch (error) {
    console.error("Create news error:", error);
    return NextResponse.json(
      { message: "Internal server error" },
      { status: 500 }
    );
  }
}

export async function PUT(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const { searchParams } = new URL(request.url);
    const id = searchParams.get("id");

    if (!id) {
      return createErrorResponse("Article ID is required", 400);
    }

    const { title, content, excerpt, status, category, theme, readTime, featuredImage, tags, author, authorId } = await request.json();
    console.log('Received data:', { title, content, excerpt, status, category, theme, readTime, featuredImage, tags, author, authorId });
    // Validate input
    if (!title || !content || !excerpt || !category) {
      return createErrorResponse("Title, content, excerpt, and category are required", 400);
    }

    // Build update data
    const updateData: Record<string, unknown> = {
      title,
      content,
      excerpt,
      status: status || "draft",
      category,
      theme: theme || "primary",
      readTime: readTime || 3,
    };
    
    // Update author if provided
    if (author !== undefined) {
      updateData.author = author;
    }
    
    // Update authorId if provided
    if (authorId !== undefined) {
      updateData.authorId = authorId || null;
    }
    
    console.log('Featured Image:', featuredImage);
    // Only update image if featuredImage is provided
    if (featuredImage !== undefined) {
      console.log('Updating image to:', featuredImage);
      updateData.image = featuredImage;
    }

    // Only update tags if tags array is provided
    if (tags !== undefined) {
      updateData.tags = Array.isArray(tags) && tags.length > 0 ? JSON.stringify(tags) : null;
    }

    const article = await prisma.newsArticle.update({
      where: { id },
      data: updateData,
    });

    return NextResponse.json(
      {
        message: "Article updated successfully",
        article,
      },
      { status: 200 }
    );
  } catch (error) {
    console.error("Update news error:", error);
    return NextResponse.json(
      { message: "Internal server error" },
      { status: 500 }
    );
  }
}
