import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { withPermission, createErrorResponse } from '@/lib/admin/security';
import { Prisma } from '@prisma/client';

// POST /api/admin/newsletter/export - Export subscribers to CSV
export async function POST(request: NextRequest) {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('newsletter', 'view');
    if (auth.error) return auth.response!;

    const body = await request.json();
    const { status = 'all', ids = [] } = body;

    // Build where clause
    const where: Prisma.NewsletterSubscriberWhereInput = {};

    // If specific IDs provided, only export those
    if (ids.length > 0) {
      where.id = { in: ids };
    } else if (status !== 'all') {
      // Otherwise filter by status
      where.subscribed = status === 'subscribed';
    }

    // Fetch subscribers
    const subscribers = await prisma.newsletterSubscriber.findMany({
      where,
      orderBy: { createdAt: 'desc' },
    });

    if (subscribers.length === 0) {
      return createErrorResponse('No subscribers to export', 404);
    }

    // Generate CSV
    const csvHeaders = ['Email', 'First Name', 'Subscribed', 'Subscribed Date', 'Unsubscribed Date'];
    const csvRows = subscribers.map((sub) => [
      sub.email,
      sub.firstName || 'N/A',
      sub.subscribed ? 'Yes' : 'No',
      new Date(sub.subscribedAt).toLocaleDateString(),
      sub.unsubscribedAt ? new Date(sub.unsubscribedAt).toLocaleDateString() : 'N/A',
    ]);

    const csv = [
      csvHeaders.join(','),
      ...csvRows.map((row) => row.map((cell) => `"${cell}"`).join(',')),
    ].join('\n');

    // Return CSV file
    return new NextResponse(csv, {
      status: 200,
      headers: {
        'Content-Type': 'text/csv',
        'Content-Disposition': `attachment; filename="newsletter-subscribers-${new Date().toISOString().split('T')[0]}.csv"`,
      },
    });
  } catch (error) {
    console.error('Newsletter Export Error:', error);
    return createErrorResponse(
      error instanceof Error ? error.message : 'Failed to export subscribers',
      500
    );
  }
}
