import { NextRequest } from 'next/server';
import { prisma } from '@/lib/prisma';
import { withPermission, createErrorResponse, createSuccessResponse } from '@/lib/admin/security';
import { Prisma } from '@prisma/client';

// GET /api/admin/newsletter - List subscribers with filters
export async function GET(request: NextRequest) {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('newsletter', 'view');
    if (auth.error) return auth.response!;

    // Get query parameters
    const { searchParams } = new URL(request.url);
    const page = parseInt(searchParams.get('page') || '1');
    const limit = parseInt(searchParams.get('limit') || '10');
    const search = searchParams.get('search') || '';
    const status = searchParams.get('status') || 'all'; // all, subscribed, unsubscribed
    const sortBy = searchParams.get('sortBy') || 'createdAt';
    const sortOrder = searchParams.get('sortOrder') || 'desc';

    // Build where clause
    const where: Prisma.NewsletterSubscriberWhereInput = {};

    // Search filter
    if (search) {
      where.OR = [
        { email: { contains: search } },
        { firstName: { contains: search } },
      ];
    }

    // Status filter
    if (status !== 'all') {
      where.subscribed = status === 'subscribed';
    }

    // Calculate pagination
    const skip = (page - 1) * limit;

    // Fetch subscribers with pagination
    const [subscribers, total] = await Promise.all([
      prisma.newsletterSubscriber.findMany({
        where,
        skip,
        take: limit,
        orderBy: {
          [sortBy]: sortOrder,
        },
      }),
      prisma.newsletterSubscriber.count({ where }),
    ]);

    // Calculate stats
    const [totalSubscribers, activeCount] = await Promise.all([
      prisma.newsletterSubscriber.count(),
      prisma.newsletterSubscriber.count({ where: { subscribed: true } }),
    ]);

    const unsubscribedCount = totalSubscribers - activeCount;

    return createSuccessResponse({
      subscribers,
      pagination: {
        page,
        limit,
        total,
        totalPages: Math.ceil(total / limit),
      },
      stats: {
        total: totalSubscribers,
        active: activeCount,
        unsubscribed: unsubscribedCount,
      },
    });
  } catch (error) {
    console.error('Newsletter GET Error:', error);
    return createErrorResponse(
      error instanceof Error ? error.message : 'Failed to fetch subscribers',
      500
    );
  }
}

// DELETE /api/admin/newsletter - Bulk delete subscribers
export async function DELETE(request: NextRequest) {
  try {
    // Check admin authentication and permissions
    const auth = await withPermission('newsletter', 'manage');
    if (auth.error) return auth.response!;

    const body = await request.json();
    const { ids } = body;

    if (!ids || !Array.isArray(ids) || ids.length === 0) {
      return createErrorResponse('No subscriber IDs provided', 400);
    }

    // Delete multiple subscribers
    const result = await prisma.newsletterSubscriber.deleteMany({
      where: {
        id: { in: ids },
      },
    });

    return createSuccessResponse(
      { deletedCount: result.count },
      `Successfully deleted ${result.count} subscriber(s)`
    );
  } catch (error) {
    console.error('Newsletter DELETE Error:', error);
    return createErrorResponse(
      error instanceof Error ? error.message : 'Failed to delete subscribers',
      500
    );
  }
}
