import { prisma } from "@/lib/prisma";
import bcrypt from "bcryptjs";
import { NextRequest } from "next/server";
import { withAdminAuthAndRateLimit, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';
import { RateLimitPresets } from '@/lib/admin/rate-limiter';

export async function PUT(request: NextRequest) {
  try {
    // Rate limit password changes (3 per 5 minutes per user)
    const auth = await withAdminAuthAndRateLimit(
      request,
      RateLimitPresets.PASSWORD_CHANGE.limit,
      RateLimitPresets.PASSWORD_CHANGE.windowMs
    );
    if (auth.error) return auth.response!;

    const { currentPassword, newPassword } = await request.json();

    // Validate input
    if (!currentPassword || !newPassword) {
      return createErrorResponse('Current password and new password are required', 400);
    }

    if (newPassword.length < 8) {
      return createErrorResponse('New password must be at least 8 characters', 400);
    }

    // Get user from database
    const user = await prisma.adminUser.findUnique({
      where: { id: auth.userId },
    });

    if (!user) {
      return createErrorResponse('User not found', 404);
    }

    // Verify current password
    const passwordMatch = await bcrypt.compare(currentPassword, user.password);

    if (!passwordMatch) {
      return createErrorResponse('Current password is incorrect', 400);
    }

    // Hash new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update password
    await prisma.adminUser.update({
      where: { id: auth.userId },
      data: {
        password: hashedPassword,
      },
    });

    return createSuccessResponse({}, 'Password changed successfully');
  } catch (error) {
    console.error("Password change error:", error);
    return createErrorResponse('Failed to change password', 500);
  }
}
