import { prisma } from "@/lib/prisma";
import { NextRequest } from "next/server";
import { withAdminAuth, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

export async function PUT(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const { name, email, avatar, bio, phone, department } = await request.json();

    // Validate input
    if (!name || !email) {
      return createErrorResponse('Name and email are required', 400);
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return createErrorResponse('Invalid email format', 400);
    }

    // Check if email is already taken by another user
    const existingUser = await prisma.adminUser.findUnique({
      where: { email },
    });

    if (existingUser && existingUser.id !== auth.userId) {
      return createErrorResponse('Email is already in use', 400);
    }

    // Validate phone if provided
    if (phone && phone.length < 10) {
      return createErrorResponse('Phone number must be at least 10 digits', 400);
    }

    // Update user with all profile fields
    const updatedUser = await prisma.adminUser.update({
      where: { id: auth.userId },
      data: {
        name,
        email,
        ...(avatar && { avatar }),
        ...(bio && { bio }),
        ...(phone && { phone }),
        ...(department && { department }),
      },
    });

    return createSuccessResponse({
      id: updatedUser.id,
      name: updatedUser.name,
      email: updatedUser.email,
      avatar: updatedUser.avatar,
      bio: updatedUser.bio,
      phone: updatedUser.phone,
      department: updatedUser.department,
      role: updatedUser.role,
    }, 'Profile updated successfully');
  } catch (error) {
    console.error("Profile update error:", error);
    return createErrorResponse('Failed to update profile', 500);
  }
}

