import { NextRequest } from 'next/server';
import { prisma } from '@/lib/prisma';
import { withPermission, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

// GET /api/admin/site-settings - Get all site settings or specific settings by keys
export async function GET(request: NextRequest) {
  try {
    const auth = await withPermission('settings', 'view');
    if (auth.error) return auth.response!;

    const { searchParams } = new URL(request.url);
    const keys = searchParams.get('keys')?.split(',');

    const settings = keys
      ? await prisma.siteSettings.findMany({
          where: { key: { in: keys } },
        })
      : await prisma.siteSettings.findMany();

    // Convert to object format { key: value }
    const settingsObj = settings.reduce((acc, setting) => {
      acc[setting.key] = setting.value;
      return acc;
    }, {} as Record<string, string>);

    return createSuccessResponse(settingsObj);
  } catch (error) {
    console.error('Error fetching site settings:', error);
    return createErrorResponse('Failed to fetch site settings', 500);
  }
}

// POST /api/admin/site-settings - Update multiple settings at once
export async function POST(request: NextRequest) {
  try {
    const auth = await withPermission('settings', 'manage');
    if (auth.error) return auth.response!;

    const body = await request.json();
    const { settings } = body as { settings: Record<string, string> };

    if (!settings || typeof settings !== 'object') {
      return createErrorResponse('Invalid settings format', 400);
    }

    // Update or create each setting
    const updates = Object.entries(settings).map(([key, value]) =>
      prisma.siteSettings.upsert({
        where: { key },
        update: { value },
        create: { key, value },
      })
    );

    await Promise.all(updates);

    return createSuccessResponse({ message: 'Settings updated successfully' });
  } catch (error) {
    console.error('Error updating site settings:', error);
    return createErrorResponse('Failed to update site settings', 500);
  }
}
