'use server';

import { NextRequest } from 'next/server';
import { prisma } from '@/lib/prisma';
import * as Yup from 'yup';
import { withAdminAuth, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

const studentInNeedSchema = Yup.object().shape({
  name: Yup.string().required('Name is required'),
  grade: Yup.string().optional(),
  year: Yup.string().optional(),
  story: Yup.string().required('Story is required').min(50),
  background: Yup.string().optional(),
  specificNeed: Yup.string().optional(),
  achievements: Yup.string().optional(),
  image: Yup.string().optional(),
  needCategory: Yup.string().required('Need category is required'),
  contactInfo: Yup.string().optional(),
  featured: Yup.boolean().optional(),
});

export async function GET() {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const students = await prisma.studentInNeed.findMany({
      orderBy: { createdAt: 'desc' },
    });

    return createSuccessResponse(students, 'Students in need fetched successfully');
  } catch (error) {
    console.error('Error fetching students in need:', error);
    return createErrorResponse('Failed to fetch students in need', 500);
  }
}

export async function POST(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const body = await request.json();

    // Validate
    const validatedData = await studentInNeedSchema.validate(body);

    const student = await prisma.studentInNeed.create({
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      data: validatedData as any,
    });

    return createSuccessResponse(student, 'Student in need created successfully');
  } catch (error) {
    if (error instanceof Yup.ValidationError) {
      const fieldErrors: Record<string, string> = {};
      error.inner.forEach((err) => {
        if (err.path) {
          fieldErrors[err.path] = err.message;
        }
      });

      return createErrorResponse('Validation failed', 400);
    }

    console.error('Error creating student in need:', error);
    return createErrorResponse('Failed to create student in need', 500);
  }
}
