import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import * as Yup from 'yup';
import { withAdminAuth, createSuccessResponse } from '@/lib/admin/security';

// Mark route as dynamic - no static generation
export const dynamic = 'force-dynamic';

// Helper function to parse JSON fields in student data
function parseStudentData(student: Record<string, unknown>) {
  return {
    ...student,
    achievements: student.achievements ? JSON.parse(student.achievements as string) : null,
    images: student.images ? JSON.parse(student.images as string) : null,
  };
}

// Validation schema
const studentSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  title: Yup.string().max(200).nullable(),
  grade: Yup.string().required('Grade is required').min(2).max(50),
  year: Yup.string().max(4).nullable(),
  story: Yup.string().required('Story is required').min(50).max(50000),
  openingQuote: Yup.string().max(500).nullable(),
  closingQuote: Yup.string().max(500).nullable(),
  achievements: Yup.string().max(2000).nullable(),
  images: Yup.array()
    .of(Yup.string().required('Image path is required').min(1, 'Image path cannot be empty'))
    .max(3, 'Maximum 3 images allowed')
    .nullable(),
  featured: Yup.boolean(),
  category: Yup.string().oneOf(['academic', 'sports', 'arts', 'other', ''], 'Invalid category').nullable(),
  supportNeeded: Yup.string().max(500).nullable(),
});

// GET all students
export async function GET() {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const students = await prisma.student.findMany({
      orderBy: { createdAt: 'desc' },
    });

    // Parse JSON fields for each student
    const parsedStudents = students.map(parseStudentData);

    return createSuccessResponse(parsedStudents, 'Students fetched successfully');
  } catch (error) {
    console.error('Error fetching students:', error);
    return NextResponse.json(
      { success: false, message: 'Failed to fetch students' },
      { status: 500 }
    );
  }
}

// POST create student
export async function POST(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const body = await request.json();

    // Validate input
    const validatedData = await studentSchema.validate(body);

    // Create student
    const student = await prisma.student.create({
      data: {
        name: validatedData.name,
        title: validatedData.title || null,
        grade: validatedData.grade,
        year: validatedData.year || null,
        story: validatedData.story,
        openingQuote: validatedData.openingQuote || null,
        closingQuote: validatedData.closingQuote || null,
        achievements: validatedData.achievements || null,
        images: validatedData.images ? JSON.stringify(validatedData.images) : null,
        featured: validatedData.featured || false,
        category: validatedData.category || null,
      },
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Student created successfully',
        data: parseStudentData(student),
      },
      { status: 201 }
    );
  } catch (error) {
    console.error('Error creating student:', error);

    if (error instanceof Yup.ValidationError) {
      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors: {
            [error.path || 'general']: error.message,
          },
        },
        { status: 400 }
      );
    }

    return NextResponse.json(
      { success: false, message: 'Failed to create student' },
      { status: 500 }
    );
  }
}
