import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import * as Yup from 'yup';
import { withAdminAuth, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';

// Validation schema
const teacherSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  title: Yup.string().required('Title is required').min(2).max(50),
  position: Yup.string().required('Position/Subject is required').min(2).max(100),
  subject: Yup.string().max(100).nullable(),
  gender: Yup.string().oneOf(['male', 'female', ''], 'Invalid gender').nullable(),
  bio: Yup.string().max(2000).nullable(),
  qualifications: Yup.string().max(2000).nullable(),
  experience: Yup.number().min(0).max(70).nullable(),
  image: Yup.string().nullable().test('is-valid-image-url', 'Invalid image URL', (value) => {
    if (!value) return true; // Allow empty
    // Allow both relative paths (/uploads/...) and absolute URLs
    return value.startsWith('/') || value.startsWith('http://') || value.startsWith('https://');
  }),
  email: Yup.string().email('Invalid email').nullable(),
  phone: Yup.string().max(20).nullable(),
  active: Yup.boolean(),
});

// GET all teachers
export async function GET() {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const teachers = await prisma.teacher.findMany({
      orderBy: { createdAt: 'desc' },
    });

    return createSuccessResponse(teachers, 'Teachers fetched successfully');
  } catch (error) {
    console.error('Error fetching teachers:', error);
    return createErrorResponse('Failed to fetch teachers', 500);
  }
}

// POST create teacher
export async function POST(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    const body = await request.json();
    console.log("Received Data:", body);
    // Validate input
    const validatedData = await teacherSchema.validate(body);

    // Generate slug from name
    const slug = validatedData.name
      .toLowerCase()
      .replace(/\s+/g, '-')
      .replace(/[^\w-]/g, '');

    // Check if slug already exists
    const existingTeacher = await prisma.teacher.findUnique({
      where: { slug },
    });

    if (existingTeacher) {
      return createErrorResponse('A teacher with this name already exists', 400);
    }

    // Create teacher
    const teacher = await prisma.teacher.create({
      data: {
        ...validatedData,
        slug,
      },
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Teacher created successfully',
        data: teacher,
      },
      { status: 201 }
    );
  } catch (error) {
    console.error('Error creating teacher:', error);
    if (error instanceof Yup.ValidationError) {
      const fieldErrors: Record<string, string> = {};
      error.inner.forEach((err) => {
        if (err.path) {
          fieldErrors[err.path] = err.message;
        }
      });
      return NextResponse.json(
        { success: false, message: 'Validation failed', errors: fieldErrors },
        { status: 400 }
      );
    }
    return NextResponse.json(
      { success: false, message: error instanceof Error ? error.message : 'Failed to create teacher' },
      { status: 500 }
    );
  }
}
