import { NextRequest } from 'next/server';
import { ensureDir, writeFileAsync, deleteImageFile, generateFilename, isValidImageType, isValidFileSize, buildUploadPath, buildPublicUrl } from '@/lib/admin/utils';
import { withAdminAuthAndRateLimit, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';
import { RateLimitPresets } from '@/lib/admin/rate-limiter';
import { join } from 'path';

const UPLOAD_DIR = 'public/uploads/leadership';

export async function POST(request: NextRequest) {
  try {
    // Rate limit uploads (10 per 5 minutes per user)
    const auth = await withAdminAuthAndRateLimit(
      request,
      RateLimitPresets.FILE_UPLOAD.limit,
      RateLimitPresets.FILE_UPLOAD.windowMs
    );
    if (auth.error) return auth.response!;

    const formData = await request.formData();
    const file = formData.get('file') as File;
    const gender = formData.get('gender') as string;
    const oldImagePath = formData.get('oldImagePath') as string;

    // Validate inputs
    if (!file) {
      return createErrorResponse('No file provided', 400);
    }

    if (!gender || !['male', 'female'].includes(gender)) {
      return createErrorResponse('Valid gender (male/female) is required', 400);
    }

    // Validate file
    if (!isValidImageType(file.type)) {
      return createErrorResponse('Only image files (JPG, PNG, GIF, WebP) are allowed', 400);
    }

    if (!isValidFileSize(file.size)) {
      return createErrorResponse('File size must be less than 5MB', 400);
    }

    // Generate unique filename
    const filename = generateFilename(file.name);

    // Create gender-based directory path
    const genderDir = buildUploadPath(UPLOAD_DIR, gender);
    const filePath = join(genderDir, filename);

    // Ensure directory exists
    try {
      await ensureDir(genderDir);
    } catch (error) {
      return createErrorResponse(error instanceof Error ? error.message : 'Failed to create upload directory', 500);
    }

    // Convert file to buffer and write
    try {
      const bytes = await file.arrayBuffer();
      const buffer = Buffer.from(bytes);
      await writeFileAsync(filePath, buffer);
    } catch (error) {
      return createErrorResponse(error instanceof Error ? error.message : 'Failed to write file', 500);
    }

    // Delete old image if provided
    if (oldImagePath && oldImagePath.startsWith('/uploads/leadership/')) {
      await deleteImageFile(oldImagePath);
    }

    // Return the public path to the uploaded image
    const imageUrl = buildPublicUrl('uploads/leadership', gender, filename);

    return createSuccessResponse({ imageUrl }, 'Image uploaded successfully');
  } catch (error) {
    console.error('Leadership image upload error:', error);
    return createErrorResponse(error instanceof Error ? error.message : 'Failed to upload image', 500);
  }
}
