import { writeFile, mkdir } from "fs/promises";
import { join } from "path";
import { NextRequest } from "next/server";
import { randomBytes } from "crypto";
import { withAdminAuthAndRateLimit, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';
import { RateLimitPresets } from '@/lib/admin/rate-limiter';

export async function POST(request: NextRequest) {
  try {
    // Rate limit uploads (10 per 5 minutes per user)
    const auth = await withAdminAuthAndRateLimit(
      request,
      RateLimitPresets.FILE_UPLOAD.limit,
      RateLimitPresets.FILE_UPLOAD.windowMs
    );
    if (auth.error) return auth.response!;

    const formData = await request.formData();
    const file = formData.get("file") as File;
    const folder = (formData.get("folder") as string) || "general";

    if (!file) {
      return createErrorResponse('No file provided', 400);
    }

    // Validate file type
    const validTypes = ["image/jpeg", "image/png", "image/webp", "image/jpg"];
    if (!validTypes.includes(file.type)) {
      return createErrorResponse('Invalid file type. Only JPEG, PNG, and WebP allowed', 400);
    }

    // Validate file size (max 2MB)
    const maxSize = 2 * 1024 * 1024;
    if (file.size > maxSize) {
      return createErrorResponse('File size must be less than 2MB', 400);
    }

    // Generate unique filename with timestamp
    const ext = file.type.split("/")[1];
    const randomName = randomBytes(8).toString("hex");
    const timestamp = Date.now();
    const filename = `${timestamp}-${randomName}.${ext}`;

    // Create upload directory path: public/uploads/{folder}/
    const uploadDir = join(process.cwd(), "public", "uploads", folder);
    
    // Ensure directory exists
    try {
      await mkdir(uploadDir, { recursive: true });
    } catch (err) {
      console.error("Failed to create directory:", err);
    }

    // Write file to disk
    const filepath = join(uploadDir, filename);
    const bytes = await file.arrayBuffer();
    await writeFile(filepath, Buffer.from(bytes));

    // Return the public URL
    const publicUrl = `/uploads/${folder}/${filename}`;

    return createSuccessResponse({
      url: publicUrl,
      filename: filename,
      folder: folder,
    }, 'File uploaded successfully');
  } catch (error) {
    console.error("Upload error:", error);
    return createErrorResponse('Failed to upload file', 500);
  }
}
