import { NextRequest } from 'next/server';
import { ensureDir, writeFileAsync, deleteImageFile, generateFilename, isValidFileSize, buildUploadPath, buildPublicUrl } from '@/lib/admin/utils';
import { withAdminAuthAndRateLimit, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';
import { RateLimitPresets } from '@/lib/admin/rate-limiter';
import { join } from 'path';

export async function POST(request: NextRequest) {
  try {
    // Rate limit uploads (10 per 5 minutes per user)
    const auth = await withAdminAuthAndRateLimit(
      request,
      RateLimitPresets.FILE_UPLOAD.limit,
      RateLimitPresets.FILE_UPLOAD.windowMs
    );
    if (auth.error) return auth.response!;

    const formData = await request.formData();
    const file = formData.get('file') as File;
    const gender = (formData.get('gender') as string) || 'other';
    const oldImage = (formData.get('oldImage') as string) || null;

    if (!file) {
      return createErrorResponse('No file provided', 400);
    }

    // Validate file type
    const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    if (!allowedTypes.includes(file.type)) {
      return createErrorResponse('Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed', 400);
    }

    // Validate file size (5MB max)
    if (!isValidFileSize(file.size)) {
      return createErrorResponse('File too large. Maximum size is 5MB', 400);
    }

    // Generate unique filename
    const filename = generateFilename(file.name);

    // Determine gender folder
    const genderFolder = ['male', 'female'].includes(gender.toLowerCase()) 
      ? gender.toLowerCase() 
      : 'other';

    // Ensure uploads/teachers/[gender] directory exists
    const uploadsDir = buildUploadPath('public', 'uploads', 'teachers', genderFolder);
    try {
      await ensureDir(uploadsDir);
    } catch (error) {
      return createErrorResponse(error instanceof Error ? error.message : 'Failed to create upload directory', 500);
    }

    // Write file
    const filepath = join(uploadsDir, filename);
    try {
      const buffer = await file.arrayBuffer();
      await writeFileAsync(filepath, Buffer.from(buffer));
    } catch (error) {
      return createErrorResponse(error instanceof Error ? error.message : 'Failed to upload image', 500);
    }

    const imageUrl = buildPublicUrl('uploads/teachers', genderFolder, filename);

    // Delete old image if it exists and is different from new one
    if (oldImage && oldImage !== imageUrl) {
      await deleteImageFile(oldImage);
    }

    return createSuccessResponse({ url: imageUrl, filename }, 'Image uploaded successfully');
  } catch (error) {
    console.error('❌ Error uploading teacher image:', error);
    return createErrorResponse(error instanceof Error ? error.message : 'Failed to upload image', 500);
  }
}
