import { prisma } from "@/lib/prisma";
import { withAdminAuth, createSuccessResponse, createErrorResponse } from '@/lib/admin/security';
import { NextRequest } from "next/server";

/**
 * GET /api/admin/users?status=pending&role=editor&search=john
 * Get list of admin users with optional filtering
 * Query params:
 *   - status: pending | approved | rejected | suspended
 *   - role: admin | editor | viewer
 *   - search: search by name or email
 *   - limit: number of results (default: 100)
 *   - offset: pagination offset (default: 0)
 */
export async function GET(request: NextRequest) {
  try {
    const auth = await withAdminAuth();
    if (auth.error) return auth.response!;

    // Get query parameters
    const searchParams = request.nextUrl.searchParams;
    const status = searchParams.get("status");
    const role = searchParams.get("role");
    const search = searchParams.get("search");
    const limit = Math.min(parseInt(searchParams.get("limit") || "100"), 1000);
    const offset = parseInt(searchParams.get("offset") || "0");

    // Build where clause
    const where: Record<string, unknown> = {};

    if (status) {
      where.status = status;
    }

    if (role) {
      where.role = role;
    }

    if (search) {
      where.OR = [
        { name: { contains: search, mode: "insensitive" } },
        { email: { contains: search, mode: "insensitive" } },
      ];
    }

    // Get total count
    const total = await prisma.adminUser.count({ where });

    // Get filtered users
    const users = await prisma.adminUser.findMany({
      where,
      select: {
        id: true,
        name: true,
        email: true,
        role: true,
        status: true,
        active: true,
        permissions: true, // ✅ Include permissions field
        lastLogin: true,
        createdAt: true,
        approvalDate: true,
        rejectionReason: true,
      },
      orderBy: [
        { status: "asc" },
        { createdAt: "desc" },
      ],
      take: limit,
      skip: offset,
    });

    return createSuccessResponse(
      {
        users,
        pagination: {
          total,
          limit,
          offset,
          pages: Math.ceil(total / limit),
        },
      },
      "Users fetched successfully"
    );
  } catch (error) {
    console.error("Fetch admin users error:", error);
    return createErrorResponse("Failed to fetch users", 500);
  }
}
