import { prisma } from "@/lib/prisma";
import bcrypt from "bcryptjs";
import { NextRequest, NextResponse } from "next/server";
import { checkRateLimit, RateLimitPresets, createRateLimitResponse, getClientIp, getLoginRateLimitKey } from "@/lib/admin/rate-limiter";
import { verifyCsrfToken, createCsrfErrorResponse } from "@/lib/admin/csrf";

export async function POST(request: NextRequest) {
  try {
    // Verify CSRF token first (before rate limiting to prevent token exhaustion)
    const body = await request.json();
    const token = body._csrf || request.headers.get('x-csrf-token');

    if (!token || !verifyCsrfToken(token)) {
      return createCsrfErrorResponse();
    }

    // Rate limit signup by IP (5 attempts per minute)
    const ip = getClientIp(request);
    const rateLimitKey = getLoginRateLimitKey(ip);
    const rateLimit = checkRateLimit(rateLimitKey, RateLimitPresets.LOGIN.limit, RateLimitPresets.LOGIN.windowMs);

    if (!rateLimit.allowed) {
      return createRateLimitResponse(RateLimitPresets.LOGIN, rateLimit.resetTime - Date.now());
    }

    const { name, email, password } = body;

    // Validation
    if (!name || !email || !password) {
      return NextResponse.json(
        { message: "Missing required fields" },
        { status: 400 }
      );
    }

    // Check if user already exists
    const existingUser = await prisma.adminUser.findUnique({
      where: { email },
    });

    if (existingUser) {
      return NextResponse.json(
        { message: "Email already exists" },
        { status: 409 }
      );
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Create user with pending status (requires admin approval)
    const user = await prisma.adminUser.create({
      data: {
        name,
        email,
        password: hashedPassword,
        role: "editor", // Default role for new signups
        status: "pending", // NEW: User must be approved by admin
        active: false, // NEW: Cannot login until approved
      },
    });

    return NextResponse.json(
      {
        success: true,
        message: "Account created successfully. Please wait for admin approval.",
        email: user.email,
        status: "pending",
      },
      { status: 201 }
    );
  } catch (error) {
    console.error("Registration error:", error);
    return NextResponse.json(
      { message: "Registration failed" },
      { status: 500 }
    );
  }
}
