import { NextRequest, NextResponse } from "next/server";
import { testSMTPConnection, sendContactNotification, sendContactConfirmation } from "@/lib/email";
import { auth } from "@/lib/auth";

/**
 * Test SMTP Connection
 * GET /api/email/test-smtp
 */
export async function GET(request: NextRequest) {
  try {
    // Optional: Check admin auth
    const session = await auth();
    if (!session?.user?.id) {
      return NextResponse.json(
        { message: "Unauthorized" },
        { status: 401 }
      );
    }

    const result = await testSMTPConnection();
    
    return NextResponse.json(result, {
      status: result.success ? 200 : 500,
    });
  } catch (error) {
    console.error("SMTP test error:", error);
    return NextResponse.json(
      { 
        success: false,
        message: error instanceof Error ? error.message : "SMTP test failed"
      },
      { status: 500 }
    );
  }
}

/**
 * Send Test Email
 * POST /api/email/test-send
 */
export async function POST(request: NextRequest) {
  try {
    // Optional: Check admin auth
    const session = await auth();
    if (!session?.user?.id) {
      return NextResponse.json(
        { message: "Unauthorized" },
        { status: 401 }
      );
    }

    const { testEmail } = await request.json();

    if (!testEmail) {
      return NextResponse.json(
        { message: "Test email address is required" },
        { status: 400 }
      );
    }

    // Send test contact notification
    const result = await sendContactNotification({
      name: "Test User",
      email: testEmail,
      phone: "+1234567890",
      subject: "Test Contact Form",
      message: "This is a test message to verify SMTP configuration is working correctly.",
    });

    if (!result.success) {
      return NextResponse.json(
        { 
          success: false,
          message: "Failed to send test email",
          error: result.error
        },
        { status: 500 }
      );
    }

    return NextResponse.json(
      { 
        success: true,
        message: "Test email sent successfully",
        messageId: result.messageId
      },
      { status: 200 }
    );
  } catch (error) {
    console.error("Email send test error:", error);
    return NextResponse.json(
      { 
        success: false,
        message: error instanceof Error ? error.message : "Failed to send test email"
      },
      { status: 500 }
    );
  }
}
