import { NextRequest, NextResponse } from 'next/server';
import * as Yup from 'yup';
import { prisma } from '@/lib/prisma';
import { sendAdmissionsConfirmation } from '@/lib/email';

// Define the request body schema for validation
const admissionsFormSchema = Yup.object().shape({
  studentFirstName: Yup.string().required('Student first name is required').min(2).max(100),
  studentLastName: Yup.string().required('Student last name is required').min(2).max(100),
  gradeLevel: Yup.string().required('Grade level is required'),
  birthDate: Yup.string().required('Birth date is required'),
  parentName: Yup.string().required('Parent name is required').min(2).max(100),
  parentEmail: Yup.string().email('Invalid email').required('Parent email is required'),
  parentPhone: Yup.string().required('Phone is required').matches(/^[\d\s\-\+\(\)]*$/, 'Valid phone required'),
  message: Yup.string().optional().max(5000),
});

export async function POST(request: NextRequest) {
  try {
    // Parse request body
    const body = await request.json();

    // Extract fields
    const { studentFirstName, studentLastName, gradeLevel, birthDate, parentName, parentEmail, parentPhone, message } = body;

    // Validate fields
    await admissionsFormSchema.validate({
      studentFirstName,
      studentLastName,
      gradeLevel,
      birthDate,
      parentName,
      parentEmail,
      parentPhone,
      message,
    });
    console.log("Validated Data:", {
      studentFirstName,
      studentLastName})
    // Save to database
    const inquiry = await prisma.admissionsInquiry.create({
      data: {
        studentName: `${studentFirstName} ${studentLastName}`,
        gradeLevel,
        parentName,
        parentEmail,
        parentPhone,
        message: message || null,
        status: 'new',
        reviewed: false,
      },
    });

    // Send confirmation email to parent
    await sendAdmissionsConfirmation(parentEmail, parentName, `${studentFirstName} ${studentLastName}`, gradeLevel);

    return NextResponse.json(
      {
        success: true,
        message: 'Thank you! Your application has been received. We will contact you soon.',
        inquiryId: inquiry.id,
      },
      { status: 200 }
    );
  } catch (error) {
    console.error('Admissions form submission error:', error);

    // Handle validation errors
    if (error instanceof Yup.ValidationError) {
      const errors = error.inner.reduce(
        (acc, err) => {
          if (err.path) {
            acc[err.path] = err.message;
          }
          return acc;
        },
        {} as Record<string, string>
      );

      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors,
        },
        { status: 400 }
      );
    }

    // Handle database errors
    if (error instanceof Error && error.message.includes('Prisma')) {
      console.error('Database error:', error);
      return NextResponse.json(
        {
          success: false,
          message: 'Database error. Please try again later.',
        },
        { status: 500 }
      );
    }

    // Handle email errors
    if (error instanceof Error && error.message.includes('SMTP')) {
      console.error('Email error:', error);
      return NextResponse.json(
        {
          success: false,
          message: 'Your application was saved, but confirmation email could not be sent.',
        },
        { status: 500 }
      );
    }

    return NextResponse.json(
      {
        success: false,
        message: 'An error occurred while processing your application. Please try again later.',
      },
      { status: 500 }
    );
  }
}

export async function GET() {
  return NextResponse.json(
    {
      success: false,
      message: 'Method not allowed',
    },
    { status: 405 }
  );
}
