import { NextRequest, NextResponse } from 'next/server';
import * as Yup from 'yup';
import { prisma } from '@/lib/prisma';
import { sendContactConfirmation, sendContactNotification } from '@/lib/email';

// Define the request body schema
const contactFormSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(2).max(100),
  email: Yup.string().email('Invalid email').required('Email is required'),
  phone: Yup.string().optional().matches(/^[\d\s\-\+\(\)]*$/, 'Valid phone required'),
  subject: Yup.string().required('Subject is required'),
  message: Yup.string().required('Message is required').min(20).max(5000),
});

export async function POST(request: NextRequest) {
  try {
    // Parse request body
    const body = await request.json();

    // Validate the request body
    await contactFormSchema.validate(body, { abortEarly: false });

    const { name, email, phone, subject, message } = body;

    // Save contact submission to database
    const contact = await prisma.contact.create({
      data: {
        name,
        email,
        phone: phone || null,
        subject,
        message,
        read: false,
      },
    });

    // Send confirmation email to user
    await sendContactConfirmation(email, name, subject);

    // Send notification email to admin
    await sendContactNotification({
      name,
      email,
      phone,
      subject,
      message,
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Your message has been sent successfully. We will contact you soon.',
        contactId: contact.id,
      },
      { status: 200 }
    );
  } catch (error) {
    console.error('Contact form submission error:', error);

    // Handle validation errors
    if (error instanceof Yup.ValidationError) {
      const errors = error.inner.reduce(
        (acc, err) => {
          if (err.path) {
            acc[err.path] = err.message;
          }
          return acc;
        },
        {} as Record<string, string>
      );

      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors,
        },
        { status: 400 }
      );
    }

    // Handle email sending errors
    if (error instanceof Error && error.message.includes('SMTP')) {
      return NextResponse.json(
        {
          success: false,
          message:
            'Unable to send email. The message was saved, but notification emails could not be sent. Please try again later.',
        },
        { status: 500 }
      );
    }

    // Handle database errors
    if (error instanceof Error && error.message.includes('Prisma')) {
      return NextResponse.json(
        {
          success: false,
          message: 'Database error. Please try again later.',
        },
        { status: 500 }
      );
    }

    return NextResponse.json(
      {
        success: false,
        message: 'An error occurred while processing your request. Please try again later.',
      },
      { status: 500 }
    );
  }
}

export async function GET() {
  return NextResponse.json(
    {
      success: false,
      message: 'Method not allowed',
    },
    { status: 405 }
  );
}
