import { NextRequest, NextResponse } from 'next/server';
import * as Yup from 'yup';
import { prisma } from '@/lib/prisma';
import { sendEmail } from '@/lib/email';

const DONATION_SCHEMA = Yup.object().shape({
  donorName: Yup.string().required('Name is required').min(2).max(100),
  email: Yup.string().email('Invalid email').required('Email is required'),
  phone: Yup.string().optional().matches(/^[\d\s\-\+\(\)]*$/, 'Invalid phone'),
  amount: Yup.number().required('Amount is required').positive().min(1),
  currency: Yup.string().required('Currency is required').oneOf(['USD', 'AFN', 'EUR']),
  category: Yup.string().required('Category is required').oneOf([
    'general',
    'scholarships',
    'facilities',
    'technology',
    'programs',
    'emergency',
  ]),
  message: Yup.string().optional().max(5000),
  paymentMethod: Yup.string().required('Payment method is required').oneOf([
    'credit_card',
    'paypal',
    'bank_transfer',
    'cryptocurrency',
  ]),
  anonymous: Yup.boolean(),
});

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();

    // Validate input
    const validatedData = await DONATION_SCHEMA.validate(body);

    // Create donation record
    const donation = await prisma.donation.create({
      data: {
        name: validatedData.anonymous ? 'Anonymous Donor' : validatedData.donorName,
        email: validatedData.email,
        phone: validatedData.phone || null,
        amount: validatedData.amount,
        currency: validatedData.currency,
        category: validatedData.category,
        message: validatedData.message || null,
        paymentMethod: validatedData.paymentMethod,
        anonymous: validatedData.anonymous,
        paymentStatus: 'pending', // Set to pending until payment is processed
      },
    });

    // Send thank you email
    await sendEmail({
      to: validatedData.email,
      subject: 'Thank You for Your Donation to Setaragan Zirak School',
      html: `
        <div style="font-family: Arial, sans-serif; color: #333;">
          <h2>Thank You for Your Generous Donation!</h2>
          <p>Dear ${validatedData.anonymous ? 'Friend' : validatedData.donorName},</p>
          <p>We are deeply grateful for your generous donation to Setaragan Zirak School.</p>
          
          <div style="background-color: #f5f5f5; padding: 15px; border-radius: 5px; margin: 20px 0;">
            <h3 style="margin-top: 0;">Donation Details</h3>
            <p><strong>Amount:</strong> ${validatedData.currency} ${validatedData.amount.toFixed(2)}</p>
            <p><strong>Category:</strong> ${validatedData.category}</p>
            <p><strong>Transaction ID:</strong> ${donation.id}</p>
            <p><strong>Status:</strong> ${donation.paymentStatus}</p>
          </div>

          <p>Your donation will help us:</p>
          <ul>
            <li>Provide quality education to students from all backgrounds</li>
            <li>Support scholarships for deserving students</li>
            <li>Improve our facilities and learning resources</li>
            <li>Expand our programs and community impact</li>
          </ul>

          <p>For donation receipts and tax information, please contact us at <strong>donations@setaraganzirak.com</strong></p>
          
          <p>Thank you for making a difference in the lives of our students!</p>
          <p>Best regards,<br>
          <strong>Setaragan Zirak School</strong></p>
        </div>
      `,
    });

    // Send notification email to admin
    await sendEmail({
      to: process.env.ADMIN_EMAIL || 'admin@setaraganzirak.com',
      subject: `New Donation Received - ${validatedData.currency} ${validatedData.amount.toFixed(2)}`,
      html: `
        <div style="font-family: Arial, sans-serif; color: #333;">
          <h2>New Donation Received</h2>
          <p><strong>Donor Name:</strong> ${validatedData.anonymous ? 'Anonymous' : validatedData.donorName}</p>
          <p><strong>Email:</strong> ${validatedData.email}</p>
          <p><strong>Phone:</strong> ${validatedData.phone || 'Not provided'}</p>
          <p><strong>Amount:</strong> ${validatedData.currency} ${validatedData.amount.toFixed(2)}</p>
          <p><strong>Category:</strong> ${validatedData.category}</p>
          <p><strong>Payment Method:</strong> ${validatedData.paymentMethod}</p>
          <p><strong>Message:</strong> ${validatedData.message || 'None'}</p>
          <p><strong>Transaction ID:</strong> ${donation.id}</p>
          <p><strong>Status:</strong> ${donation.paymentStatus}</p>
          <p><a href="${process.env.ADMIN_URL || 'http://localhost:3000'}/admin/donations/${donation.id}">View Donation</a></p>
        </div>
      `,
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Thank you for your donation! Check your email for a receipt and next steps.',
        donationId: donation.id,
        paymentUrl: null, // Will be populated when payment gateway is integrated
      },
      { status: 201 }
    );
  } catch (error) {
    if (error instanceof Yup.ValidationError) {
      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors: { [error.path || 'form']: error.message },
        },
        { status: 400 }
      );
    }

    console.error('Donation submission error:', error);

    return NextResponse.json(
      {
        success: false,
        message: error instanceof Error ? error.message : 'Failed to process donation',
      },
      { status: 500 }
    );
  }
}
