import { NextRequest, NextResponse } from 'next/server';
import * as Yup from 'yup';
import { prisma } from '@/lib/prisma';
import { sendEmail } from '@/lib/email';

const NEWSLETTER_SCHEMA = Yup.object().shape({
  email: Yup.string().email('Invalid email').required('Email is required'),
  firstName: Yup.string().optional().max(100),
});

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();

    // Validate input
    const validatedData = await NEWSLETTER_SCHEMA.validate(body);

    // Check if already subscribed
    const existingSubscriber = await prisma.newsletterSubscriber.findUnique({
      where: { email: validatedData.email },
    });

    if (existingSubscriber && existingSubscriber.subscribed) {
      return NextResponse.json(
        {
          success: false,
          message: 'You are already subscribed to our newsletter.',
        },
        { status: 400 }
      );
    }

    // Create or update subscription
    const subscription = await prisma.newsletterSubscriber.upsert({
      where: { email: validatedData.email },
      update: {
        firstName: validatedData.firstName || null,
        subscribed: true,
        subscribedAt: new Date(),
        unsubscribedAt: null, // Re-subscribe if previously unsubscribed
      },
      create: {
        email: validatedData.email,
        firstName: validatedData.firstName || null,
        subscribed: true,
        subscribedAt: new Date(),
      },
    });

    // Send welcome email
    await sendEmail({
      to: validatedData.email,
      subject: 'Welcome to Setaragan Zirak School Newsletter',
      html: `
        <div style="font-family: Arial, sans-serif; color: #333;">
          <h2>Welcome to Our Newsletter!</h2>
          <p>Hi ${validatedData.firstName ? validatedData.firstName + ',' : 'there,'},</p>
          <p>Thank you for subscribing to the Setaragan Zirak School newsletter. We're excited to keep you updated on:</p>
          <ul>
            <li>Latest news and announcements</li>
            <li>Student achievements and success stories</li>
            <li>Upcoming events and programs</li>
            <li>Educational insights and resources</li>
          </ul>
          <p>Look out for our weekly updates in your inbox!</p>
          <p>Best regards,<br>
          <strong>Setaragan Zirak School</strong></p>
          <p style="font-size: 12px; color: #666; margin-top: 20px;">
            To unsubscribe from this newsletter, reply with "UNSUBSCRIBE".
          </p>
        </div>
      `,
    });

    return NextResponse.json(
      {
        success: true,
        message: 'Thank you for subscribing! Check your email for confirmation.',
        subscriptionId: subscription.id,
      },
      { status: 201 }
    );
  } catch (error) {
    if (error instanceof Yup.ValidationError) {
      return NextResponse.json(
        {
          success: false,
          message: 'Validation failed',
          errors: { [error.path || 'form']: error.message },
        },
        { status: 400 }
      );
    }

    console.error('Newsletter subscription error:', error);

    return NextResponse.json(
      {
        success: false,
        message: error instanceof Error ? error.message : 'Failed to subscribe to newsletter',
      },
      { status: 500 }
    );
  }
}
