'use client';

import React from 'react';
import Link from 'next/link';
import { Section, ResponsiveGrid, AppLayout } from '@/components/common/layout';
import Badge from '@/components/common/ui/badge';
import Card from '@/components/common/ui/card';
import ContactForm from '@/components/common/forms/ContactForm';
import { MapPin, Phone, Mail, Clock, MessageSquare } from 'lucide-react';
import classNames from 'classnames';
import { usePublicSiteSettings } from '@/hooks/useSiteSettings';

const ContactPage = () => {
  // Fetch contact settings
  const { data: settings } = usePublicSiteSettings([
    'contact_address_line1',
    'contact_address_line2',
    'contact_phone_primary',
    'contact_phone_whatsapp',
    'contact_email',
    'contact_hours_weekday',
    'contact_hours_saturday',
    'contact_hours_sunday',
    'social_facebook',
    'social_linkedin',
    'social_whatsapp',
    'contact_response_time',
    'contact_best_time',
    'contact_map_embed'
  ]);

  const contactInfo = [
    {
      icon: MapPin,
      title: "Address",
      details: [
        settings?.contact_address_line1 || "Market Safid Street, 2nd Phase, Omid-e-Sabz Township",
        settings?.contact_address_line2 || "Darullaman, Kabul, Afghanistan",
        ""
      ],
      color: "text-blue-600"
    },
    {
      icon: Phone,
      title: "Phone",
      details: [
        settings?.contact_phone_primary || "+93747500432",
        `WhatsApp: ${settings?.contact_phone_whatsapp || "+93785675793"}`
      ],
      color: "text-green-600"
    },
    {
      icon: Mail,
      title: "Email",
      details: [
        settings?.contact_email || "setaraganzirak98@gmail.com"
      ],
      color: "text-purple-600"
    },
    {
      icon: Clock,
      title: "Office Hours",
      details: [
        settings?.contact_hours_weekday || "Monday - Friday: 8:00 AM - 4:00 PM",
        settings?.contact_hours_saturday || "Saturday: 9:00 AM - 1:00 PM",
        settings?.contact_hours_sunday || "Sunday: Closed"
      ],
      color: "text-orange-600"
    }
  ];

  return (
    <AppLayout>
      <div className="min-h-screen">
      {/* Hero Section */}
      <Section background="primary" padding="xl" className="text-center bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600">
        <div className="max-w-4xl mx-auto">
          <Badge className="mb-4">Get In Touch</Badge>
          <h1 className="text-4xl sm:text-5xl lg:text-6xl font-black text-white mb-6">
            <span className="bg-gradient-to-r from-white to-gray-200 bg-clip-text text-transparent">Connect</span> With Us
          </h1>
          <p className="text-xl sm:text-2xl text-white/90 max-w-3xl mx-auto leading-relaxed">
            We&apos;re here to help you learn more about our educational programs and answer any questions you may have.
          </p>
        </div>
      </Section>

      {/* Contact Information Cards */}
      <Section background="white" padding="lg">
        <div className="text-center mb-12 sm:mb-16">
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Get In <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">Touch</span>
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto">
            Multiple ways to reach us for inquiries, admissions, and support.
          </p>
        </div>

        <ResponsiveGrid cols={{ default: 1, sm: 2, lg: 4 }} gap="lg">
          {contactInfo.map((info, index) => {
            const IconComponent = info.icon;
            return (
              <Card key={index} className="p-6 sm:p-8 text-center h-full hover:shadow-xl transition-all duration-300 hover:scale-105">
                <div className={`w-16 h-16 mx-auto mb-6 bg-gradient-to-br from-gray-100 to-gray-50 rounded-2xl flex items-center justify-center shadow-lg`}>
                  <IconComponent className={`w-8 h-8 ${info.color}`} />
                </div>
                <h3 className="text-xl sm:text-2xl font-bold text-gray-900 mb-4">{info.title}</h3>
                <div className="space-y-2 ">
                  {info.details.map((detail, detailIndex) => (
                    <p key={detailIndex} className={classNames("text-gray-600 leading-relaxed text-sm",
                      (index === 0 || index === 3) && "text-left"
                    )}>
                      {detail}
                    </p>
                  ))}
                </div>
              </Card>
            );
          })}
        </ResponsiveGrid>
      </Section>

      {/* Social Media Section */}
      <Section background="gray" padding="lg">
        <div className="text-center mb-12">
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Follow Us
          </h2>
          <p className="text-lg text-gray-600 max-w-2xl mx-auto">
            Stay connected with Setaragan Zirak School through our social media channels.
          </p>
        </div>

        <div className="flex justify-center gap-8">
          <a
            href={settings?.social_facebook || "https://facebook.com/profile.php?id=100063987887508"}
            target="_blank"
            rel="noopener noreferrer"
            className="w-16 h-16 bg-blue-600 hover:bg-blue-700 text-white rounded-full flex items-center justify-center transition-all duration-300 hover:scale-110 shadow-lg hover:shadow-xl"
            aria-label="Follow us on Facebook"
          >
            <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 24 24">
              <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
            </svg>
          </a>

          <a
            href={settings?.social_linkedin || "https://www.linkedin.com/in/azizllah-paeyndah-5048a1250"}
            target="_blank"
            rel="noopener noreferrer"
            className="w-16 h-16 bg-blue-700 hover:bg-blue-800 text-white rounded-full flex items-center justify-center transition-all duration-300 hover:scale-110 shadow-lg hover:shadow-xl"
            aria-label="Connect with us on LinkedIn"
          >
            <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 24 24">
              <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/>
            </svg>
          </a>

          <a
            href={settings?.social_whatsapp || "https://wa.me/93785675793"}
            target="_blank"
            rel="noopener noreferrer"
            className="w-16 h-16 bg-green-600 hover:bg-green-700 text-white rounded-full flex items-center justify-center transition-all duration-300 hover:scale-110 shadow-lg hover:shadow-xl"
            aria-label="Contact us on WhatsApp"
          >
            <svg className="w-8 h-8" fill="currentColor" viewBox="0 0 24 24">
              <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.885 3.488"/>
            </svg>
          </a>
        </div>
      </Section>

      {/* Contact Form & Map Section */}
      <Section background="gray" padding="lg">
        <ResponsiveGrid cols={{ default: 1, lg: 2 }} gap="xl">
          {/* Contact Form */}
          <Card className="p-6 sm:p-8 lg:p-10">
            <div className="mb-8">
              <h3 className="text-2xl sm:text-3xl font-bold text-gray-900 mb-4">Send Us a Message</h3>
              <p className="text-gray-600 leading-relaxed">
                Have questions about admissions, programs, or our school? We&apos;d love to hear from you.
              </p>
            </div>

            <ContactForm />
          </Card>

          {/* Map & Additional Info */}
          <div className="space-y-8">
            {/* Map Placeholder */}
            <Card className="p-6 sm:p-8">
              <h3 className="text-2xl font-bold text-gray-900 mb-6">Our Location</h3>
              <div className="aspect-video bg-gradient-to-br from-gray-100 to-gray-200 rounded-lg flex items-center justify-center">
                <iframe
                  src={settings?.contact_map_embed || "https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3290.1915100927563!2d69.07643857664463!3d34.478862595081864!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x38d1676a53ac71b1%3A0xdffb8f70f4074636!2sSetaragan%20Zirak%20Private%20High%20School!5e0!3m2!1sen!2s!4v1757971730858!5m2!1sen!2s"}
                  width="600"
                  height="450"
                  style={{ border: 0 }}
                  allowFullScreen
                  loading="lazy"
                  referrerPolicy="no-referrer-when-downgrade"
                  title="Setaragan Zirak Private High School Location"
                />
                {/* <div className="text-center">
                  <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-600 font-medium">Our Location</p>
                  <p className="text-sm text-gray-500">Kabul – Darullaman – Omide-Sabz Township</p>
                  <p className="text-sm text-gray-500">2nd Phase – Market Safid Street</p>
                  <p className="text-sm text-gray-500">Afghanistan</p>
                </div> */}
              </div>
            </Card>

            {/* Quick Contact */}
            <Card className="p-6 sm:p-8">
              <h3 className="text-2xl font-bold text-gray-900 mb-6">Quick Contact</h3>
              <div className="space-y-4">
                <div className="flex items-start gap-4">
                  <MessageSquare className="w-6 h-6 text-primary-600 mt-1 flex-shrink-0" />
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-1">Response Time</h4>
                    <p className="text-gray-600 text-sm">We typically respond within 24-48 hours during business days.</p>
                  </div>
                </div>
                <div className="flex items-start gap-4">
                  <Clock className="w-6 h-6 text-secondary-600 mt-1 flex-shrink-0" />
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-1">Best Time to Call</h4>
                    <p className="text-gray-600 text-sm">Monday to Friday, 9:00 AM - 3:00 PM (Local Time)</p>
                  </div>
                </div>
              </div>
            </Card>
          </div>
        </ResponsiveGrid>
      </Section>

      {/* Call to Action */}
      <Section background="white" padding="lg">
        <div className="text-center">
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
            Ready to Join Our Community?
          </h2>
          <p className="text-lg sm:text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
            Take the next step in your educational journey. Contact us today to learn more about admissions and programs.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link
              href="/admissions"
              className="inline-flex items-center gap-3 bg-gradient-to-r from-primary-600 to-secondary-600 text-white px-8 py-4 rounded-xl font-bold text-lg hover:from-primary-700 hover:to-secondary-700 transition-all duration-200 hover:scale-105 shadow-lg hover:shadow-xl"
            >
              <MessageSquare className="w-5 h-5" />
              Apply Now
            </Link>
            <Link
              href="/about"
              className="inline-flex items-center gap-3 border-2 border-gray-300! text-gray-700 hover:border-primary-300! hover:text-primary-600 px-8 py-4 rounded-xl font-bold text-lg transition-all duration-200 hover:scale-105"
            >
              <MapPin className="w-5 h-5" />
              Learn More About Us
            </Link>
          </div>
        </div>
      </Section>
    </div>
    </AppLayout>
  );
};

export default ContactPage;
