import React from "react";
import { Metadata } from "next";
import {
  Heart,
  Users,
  BookOpen,
  Phone,
  Mail,
  ArrowRight,
  Star,
} from "lucide-react";
import Button from "@/components/common/ui/button";
import Card from "@/components/common/ui/card";
import Link from "next/link";
import AppLayout from "@/components/common/layout/AppLayout";

export const metadata: Metadata = {
  title: "Donate - Support Education | Setaragan Zirak School",
  description:
    "Help support students in need at Setaragan Zirak School. Your donation makes a direct impact on students who struggle financially but remain determined to pursue education.",
  keywords:
    "donate, support education, Afghanistan school, student support, financial aid, Setaragan Zirak",
};

export default function DonatePage() {
  const studentStories = [
    {
      name: "Hadi",
      story:
        "My name is Hadi, a student at Setaragan Zirak School. Coming from a family of eight, life is a daily struggle. My father's small business barely covers our needs, and my brother and I collect scraps to help. Rent consumes a large part of our income, making life challenging. Despite hardships, I value education. Setaragan Zirak School and its teachers offer unwavering support, providing hope and a lifeline. With their belief in me, I aim to create a better future for my family and community through education.",
      image: "/images/students/hadi.jpg",
      need: "School fees and educational materials",
      background: "Family of 8, father runs small business",
    },
    {
      name: "Zahra",
      story:
        "Zahra, a bright student at Setaragan Zirak School, embodies perseverance and ambition. Growing up in difficult circumstances, Zahra's passion for learning shines through adversity. Your contributions enable students like Zahra to access quality education and break barriers. With your support, we can nurture talents like Zahra's and pave the way for a promising future.",
      image: "/images/students/zahra.jpg",
      need: "Educational resources and support",
      background: "Bright student facing difficult circumstances",
    },
    {
      name: "Masadullah",
      story:
        "My name is Asadullah, a student at Setaragan Zirak Private School. I live in a family of ten, facing many hardships. My father works tirelessly in construction to provide for us, while my sisters and I weave carpets to help. Despite the struggles, our school gives us hope. Compassionate teachers support and motivate us, helping us strive for a better future. Education is our lifeline, but we desperately need financial support to continue.",
      image: "/images/students/masadullah.jpg",
      need: "School fees and basic educational needs",
      background:
        "Family of 10, father works construction, helps weave carpets",
    },
    {
      name: "Zarina",
      story:
        "I am Mohammad Ali, father of Zarina, a student at Setaragan Zirak School. Our family of five struggles financially, relying on my limited income as a construction laborer. With no stable job for Zarina's mother, affording school fees and basic needs is challenging. Despite hardships, Zarina remains eager to learn, but financial difficulties limit her access to essential educational materials. I kindly seek your support in covering her school fees.",
      image: "/images/students/zarina.jpg",
      need: "School fees and educational materials",
      background: "Father works as construction laborer, family of 5",
      contact: "+93748605069",
    },
  ];

  const impactAreas = [
    {
      icon: BookOpen,
      title: "Educational Materials",
      description: "Books, notebooks, and supplies for students in need",
      impact: "200+ students supported annually",
    },
    {
      icon: Users,
      title: "School Fees Support",
      description: "Direct assistance with tuition for struggling families",
      impact: "150+ families helped this year",
    },
    {
      icon: Heart,
      title: "Teacher Training",
      description: "Professional development for our dedicated educators",
      impact: "25+ teachers trained annually",
    },
  ];

  return (
    <AppLayout>
      <div className="min-h-screen bg-white">
        {/* Hero Section */}
        <section className="relative bg-gradient-to-br from-primary-50 via-white to-secondary-50 py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18">
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center">
              <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full text-xs sm:text-sm font-semibold tracking-wide mb-6">
                ❤️ Because Only Together We Can
              </div>

              <h1 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-black tracking-tight leading-none text-gray-900 mb-6">
                Build A World Where All Youth Are
                <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent">
                  {" "}
                  Safe, Strong & Valued
                </span>
              </h1>

              <p className="text-base sm:text-lg md:text-xl leading-relaxed text-gray-600 max-w-3xl mx-auto mb-12">
                Your support directly impacts students who face financial
                challenges but remain determined to pursue education for a
                brighter future. Together, we can make a meaningful difference.
              </p>

              <div className="flex flex-col sm:flex-row gap-6 justify-center">
                <Link href="#donate-now">
                  <Button className="bg-gradient-to-r from-primary-600 to-secondary-600 text-white px-8 py-4 font-semibold text-lg hover:shadow-lg transition-all duration-300">
                    Donate Now
                    <Heart className="w-5 h-5 ml-2" />
                  </Button>
                </Link>
                <Link href="#learn-more">
                  <Button
                    variant="outlined"
                    className="border-primary-300 text-primary-700 px-8 py-4 font-semibold text-lg hover:bg-primary-50"
                  >
                    Learn More
                    <ArrowRight className="w-5 h-5 ml-2" />
                  </Button>
                </Link>
              </div>
            </div>
          </div>
        </section>

        {/* Who Needs Help Section */}
        <section
          id="learn-more"
          className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gray-50"
        >
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-16">
              <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
                Who Needs Help?
              </h2>
              <p className="text-base sm:text-lg md:text-xl leading-relaxed text-gray-600 max-w-3xl mx-auto">
                Meet some of our students who need your support to continue
                their education journey
              </p>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
              {studentStories.map((student, index) => (
                <Card
                  key={index}
                  className="overflow-hidden shadow-xl border-0 bg-white hover:shadow-2xl transition-all duration-500"
                >
                  <div className="p-8">
                    <div className="flex items-start gap-6 mb-6">
                      <div className="w-20 h-20 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center flex-shrink-0">
                        <span className="text-2xl text-white font-bold">
                          {student.name[0]}
                        </span>
                      </div>
                      <div>
                        <h3 className="text-2xl font-bold text-gray-900 mb-2">
                          {student.name}
                        </h3>
                        <p className="text-primary-600 font-medium">
                          {student.background}
                        </p>
                        {student.contact && (
                          <p className="text-gray-600 text-sm mt-1">
                            Contact: {student.contact}
                          </p>
                        )}
                      </div>
                    </div>

                    <div className="mb-6">
                      <p className="text-gray-700 leading-relaxed text-sm lg:text-base">
                        {student.story}
                      </p>
                    </div>

                    <div className="bg-primary-50 p-4 rounded-lg mb-6">
                      <h4 className="font-semibold text-primary-800 mb-2">
                        Specific Need:
                      </h4>
                      <p className="text-primary-700 text-sm">{student.need}</p>
                    </div>

                    <a
                      href={`https://wa.me/93785675793?text=I would like to support ${student.name} with their education. Please provide me with more details about how I can help.`}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="w-full inline-flex items-center justify-center bg-gradient-to-r from-primary-600 to-secondary-600 text-white hover:from-primary-500 hover:to-secondary-500 px-6 py-3 rounded-lg font-semibold transition-all duration-300"
                    >
                      Support {student.name}
                      <Heart className="w-4 h-4 ml-2" />
                    </a>
                  </div>
                </Card>
              ))}
            </div>
          </div>
        </section>

        {/* Impact Areas */}
        <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-16">
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
                Your Impact
              </h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                See how your donations create meaningful change in our
                students&apos; lives
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
              {impactAreas.map((area, index) => (
                <Card
                  key={index}
                  className="text-center p-8 shadow-xl border-0 bg-white hover:shadow-2xl transition-all duration-500 hover:-translate-y-2"
                >
                  <div className="w-16 h-16 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center mx-auto mb-6">
                    <area.icon className="w-8 h-8 text-white" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-4">
                    {area.title}
                  </h3>
                  <p className="text-gray-600 mb-4 leading-relaxed">
                    {area.description}
                  </p>
                  <div className="bg-primary-50 p-3 rounded-lg">
                    <p className="text-primary-700 font-semibold text-sm">
                      {area.impact}
                    </p>
                  </div>
                </Card>
              ))}
            </div>
          </div>
        </section>

        {/* Donation Methods */}
        <section
          id="donate-now"
          className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gradient-to-br from-gray-50 to-primary-50"
        >
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-16">
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
                Global Support Options
              </h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Multiple ways to make a difference from anywhere in the world
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card className="p-6 text-center shadow-xl border-0 bg-white">
                <div className="w-16 h-16 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Mail className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-bold text-gray-900 mb-3">
                  Bank Transfer
                </h3>
                <p className="text-gray-600 mb-3 text-sm">
                  International wire transfer or local banking
                </p>
                <a
                  href="mailto:setarganzirak98@gmail.com?subject=Bank Transfer Details&body=Please provide bank transfer details for donation"
                  className="text-blue-600 font-semibold hover:underline text-sm"
                >
                  Get Bank Details
                </a>
              </Card>

              <Card className="p-6 text-center shadow-xl border-0 bg-white">
                <div className="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Phone className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-bold text-gray-900 mb-3">
                  WhatsApp Contact
                </h3>
                <p className="text-gray-600 mb-3 text-sm">
                  Direct contact for donation coordination
                </p>
                <a
                  href="https://wa.me/93785675793?text=I would like to make a donation to support students at Setaragan Zirak School. Please provide me with donation options."
                  className="text-green-600 font-semibold hover:underline text-sm"
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  +93 785 675 793
                </a>
              </Card>

              <Card className="p-6 text-center shadow-xl border-0 bg-white">
                <div className="w-16 h-16 bg-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-white font-bold text-sm">💳</span>
                </div>
                <h3 className="text-lg font-bold text-gray-900 mb-3">
                  Digital Payment
                </h3>
                <p className="text-gray-600 mb-3 text-sm">
                  PayPal, Wise, or other online methods
                </p>
                <a
                  href="mailto:setarganzirak98@gmail.com?subject=Digital Payment Options&body=Please provide digital payment options for international donations"
                  className="text-purple-600 font-semibold hover:underline text-sm"
                >
                  Request Options
                </a>
              </Card>

              <Card className="p-6 text-center shadow-xl border-0 bg-white">
                <div className="w-16 h-16 bg-orange-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-white font-bold text-sm">🏪</span>
                </div>
                <h3 className="text-lg font-bold text-gray-900 mb-3">
                  Local Agents
                </h3>
                <p className="text-gray-600 mb-3 text-sm">
                  Money transfer services worldwide
                </p>
                <a
                  href="mailto:setarganzirak98@gmail.com?subject=Money Transfer Services&body=Please provide information about local money transfer agents in my area"
                  className="text-orange-600 font-semibold hover:underline text-sm"
                >
                  Find Agent
                </a>
              </Card>
            </div>

            <div className="mt-12 text-center">
              <div className="bg-gradient-to-r from-blue-50 to-green-50 p-8 rounded-2xl shadow-lg">
                <h3 className="text-2xl font-bold text-gray-900 mb-4">
                  Direct Online Payments (Coming Soon)
                </h3>
                <p className="text-lg text-gray-800 mb-8 max-w-2xl mx-auto">
                  We&apos;re working to bring you secure online donation options
                  through Stripe and PayPal. In the meantime, please use one of
                  the proven payment methods to support our students.
                </p>
                <div className="text-center">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    <div className="bg-white p-4 rounded-lg shadow-sm">
                      <h4 className="font-semibold text-gray-800 mb-2">
                        💳 Credit Card
                      </h4>
                      <p className="text-gray-600 text-sm">
                        Secure payments via Stripe
                      </p>
                      <span className="text-xs text-blue-600 font-medium">
                        Coming Soon
                      </span>
                    </div>
                    <div className="bg-white p-4 rounded-lg shadow-sm">
                      <h4 className="font-semibold text-gray-800 mb-2">
                        🅿️ PayPal
                      </h4>
                      <p className="text-gray-600 text-sm">
                        One-click PayPal donations
                      </p>
                      <span className="text-xs text-blue-600 font-medium">
                        Coming Soon
                      </span>
                    </div>
                  </div>
                  <div className="bg-white p-4 rounded-lg shadow-sm mb-8  border-blue-600 max-w-2xl mx-auto">
                    <h3 className="font-semibold text-gray-900 mb-3 text-left">
                      Why the delay?
                    </h3>
                    <p className="text-gray-600 text-sm text-left">
                      We&apos;re implementing robust payment processing with
                      industry-leading security and transparency. This ensures
                      your donations are processed safely and you receive proper
                      receipts and tax documentation.
                    </p>
                  </div>
                  <a
                    href="mailto:setarganzirak98@gmail.com?subject=Payment Integration Updates&body=Please notify me when direct online payment options become available"
                    className="inline-flex items-center text-blue-600 font-semibold hover:underline text-sm"
                  >
                    Get notified when available
                    <ArrowRight className="w-3 h-3 ml-1" />
                  </a>
                </div>
              </div>

              <div className="mt-8">
                <div className="bg-gradient-to-r from-blue-50 to-green-50 p-8 rounded-2xl shadow-lg">
                  <h3 className="text-2xl font-bold text-gray-900 mb-4">
                    International Donation Options
                  </h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6 text-sm">
                    <div>
                      <h4 className="font-semibold text-gray-800 mb-2">
                        🌍 From Europe/USA
                      </h4>
                      <p className="text-gray-600">
                        Bank transfer, PayPal, Wise, Western Union
                      </p>
                    </div>
                    <div>
                      <h4 className="font-semibold text-gray-800 mb-2">
                        🌏 From Asia/Middle East
                      </h4>
                      <p className="text-gray-600">
                        Hawala, MoneyGram, local banking networks
                      </p>
                    </div>
                    <div>
                      <h4 className="font-semibold text-gray-800 mb-2">
                        📱 Digital Options
                      </h4>
                      <p className="text-gray-600">
                        Mobile money, online platforms, digital wallets
                      </p>
                    </div>
                  </div>
                  <p className="text-gray-600 mt-4 text-sm">
                    Contact us to discuss the best option for your location and
                    preferred method
                  </p>
                </div>
              </div>

              <div className="mt-16 text-center">
                <Card className="bg-gradient-to-r from-primary-600 to-secondary-600 text-white p-12 shadow-2xl border-0">
                  <div className="max-w-3xl mx-auto">
                    <h3 className="text-3xl md:text-4xl font-bold mb-6 text-white">
                      Every Contribution Matters
                    </h3>
                    <p className="text-xl text-white mb-8 leading-relaxed opacity-95">
                      Whether large or small, your donation creates ripple
                      effects of positive change through education. Join us in
                      empowering the next generation of Afghanistan&apos;s
                      leaders.
                    </p>
                    <div className="flex flex-col sm:flex-row gap-6 justify-center">
                      <a
                        href="mailto:setarganzirak98@gmail.com?subject=International Donation Inquiry&body=I would like to make a donation to Setaragan Zirak School. Please provide me with the best donation method for my location and preferred payment type."
                        className="inline-flex items-center bg-white text-primary-600 px-8 py-4 rounded-xl font-semibold text-lg hover:bg-gray-100 transition-colors duration-300"
                      >
                        Start Donation Process
                        <ArrowRight className="w-5 h-5 ml-2" />
                      </a>
                      <a
                        href="https://wa.me/93785675793?text=I'm interested in making an international donation to support your students. What are my options?"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="inline-flex items-center border-2 border-white text-white px-8 py-4 rounded-xl font-semibold text-lg hover:bg-white/10 transition-colors duration-300"
                      >
                        WhatsApp Inquiry
                        <Phone className="w-5 h-5 ml-2" />
                      </a>
                    </div>
                  </div>
                </Card>
              </div>
            </div>
          </div>
        </section>

        {/* Success Stories Testimonials */}
        <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-16">
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
                Donor Impact Stories
              </h2>
              <p className="text-xl text-gray-600 max-w-3xl mx-auto">
                Hear from donors and see the difference your support makes
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
              <Card className="p-8 shadow-xl border-0 bg-gradient-to-br from-primary-50 to-white">
                <div className="flex items-center mb-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center mr-4">
                    <Star className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <h4 className="font-bold text-gray-900">
                      Community Supporter
                    </h4>
                    <p className="text-gray-600 text-sm">Monthly Donor</p>
                  </div>
                </div>
                <p className="text-gray-700 leading-relaxed">
                  &quot;Supporting Setaragan Zirak School has been incredibly
                  rewarding. Knowing that my contributions help students like
                  Hadi and Zahra continue their education gives me hope for
                  Afghanistan&apos;s future. The transparency and regular
                  updates make me confident my donations are making a real
                  difference.&quot;
                </p>
              </Card>

              <Card className="p-8 shadow-xl border-0 bg-gradient-to-br from-secondary-50 to-white">
                <div className="flex items-center mb-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-secondary-500 to-primary-500 rounded-full flex items-center justify-center mr-4">
                    <Heart className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <h4 className="font-bold text-gray-900">
                      Education Advocate
                    </h4>
                    <p className="text-gray-600 text-sm">Annual Donor</p>
                  </div>
                </div>
                <p className="text-gray-700 leading-relaxed">
                  &quot;Education is the most powerful tool for change. When I
                  learned about the challenges these students face and their
                  determination to learn despite hardships, I knew I had to
                  help. Every child deserves the opportunity to reach their
                  potential through education.&quot;
                </p>
              </Card>
            </div>
          </div>
        </section>

        {/* Coming Soon - Online Payment Processing */}
        <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gradient-to-br from-primary-50 via-white to-secondary-50">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <Card className="p-8 lg:p-12 shadow-xl bg-gradient-to-r from-blue-50 to-indigo-50 border-2 border-blue-200">
              <div className="text-center">
                <div className="inline-flex items-center justify-center w-16 h-16 bg-blue-100 rounded-full mb-6">
                  <Heart className="w-8 h-8 text-blue-600" />
                </div>
                <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                  Direct Online Donations Coming Soon
                </h2>
                <p className="text-lg text-gray-700 mb-8 max-w-2xl mx-auto">
                  We&apos;re working to bring you secure online donation options
                  through Stripe and PayPal. In the meantime, please use one of
                  the proven payment methods above to support our students.
                </p>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-8">
                  <div className="p-4 bg-white rounded-lg">
                    <div className="text-3xl mb-2">💳</div>
                    <h4 className="font-semibold text-gray-900 mb-1">
                      Stripe Integration
                    </h4>
                    <p className="text-sm text-gray-600">
                      International cards, secure checkout
                    </p>
                  </div>
                  <div className="p-4 bg-white rounded-lg">
                    <div className="text-3xl mb-2">🅿️</div>
                    <h4 className="font-semibold text-gray-900 mb-1">
                      PayPal Support
                    </h4>
                    <p className="text-sm text-gray-600">
                      One-click checkout, buyer protection
                    </p>
                  </div>
                </div>

                <a
                  href="mailto:setarganzirak98@gmail.com?subject=Online Donation Payment - Coming Soon&body=I'm interested in making an online donation. Please notify me when direct payment options become available."
                  className="inline-flex items-center bg-blue-600 text-white px-8 py-3 rounded-lg font-semibold hover:bg-blue-700 transition-colors duration-300"
                >
                  Get Notified When Available
                  <Mail className="w-5 h-5 ml-2" />
                </a>
              </div>
            </Card>
          </div>
        </section>
      </div>
    </AppLayout>
  );
}
