import Link from "next/link";
import Image from "next/image";
import { Metadata } from "next";
import {
  UltraModernHero,
  NextGenStats,
  AdvancedTestimonials,
  LatestNews,
  DynamicLeadership,
} from "@/components/pages/home";
import { Sparkles, Zap, Target, Users, BookOpen, Award } from "lucide-react";
import { AppLayout } from "@/components/common/layout";

export const metadata: Metadata = {
  title:
    "Setaragan Zirak Private School - Excellence in Education | Kabul, Afghanistan",
  description:
    "Join Setaragan Zirak Private School in Kabul, Afghanistan. Premier education for 500+ students with expert teachers, modern facilities including science labs, computer labs, and digital library. Enroll today!",
  keywords:
    "Setaragan Zirak School, private school Kabul, Afghanistan education, science laboratory, computer lab, digital library, quality education, student enrollment",
  openGraph: {
    title: "Setaragan Zirak Private School - Excellence in Education",
    description:
      "Premier education in Kabul, Afghanistan with modern facilities and expert teachers serving 500+ students.",
    type: "website",
    locale: "en_US",
  },
};

export default function HomePage() {
  const stats = [
    {
      value: 500,
      label: "Active Students",
      suffix: "+",
      description: "Enrolled across all programs",
      color: "primary" as const,
    },
    {
      value: 50,
      label: "Expert Teachers",
      suffix: "+",
      description: "Qualified and experienced educators",
      color: "secondary" as const,
    },
    {
      value: 95,
      label: "Success Rate",
      suffix: "%",
      description: "Students achieving their goals",
      color: "primary" as const,
    },
    {
      value: 6,
      label: "Years of Excellence",
      suffix: "+",
      description: "Serving the community with distinction",
      color: "secondary" as const,
    },
  ];

  const testimonials = [
    {
      id: 1,
      name: "Sarah Ahmad",
      role: "Parent",
      company: "Class of 2024",
      content:
        "The quality of education and personal attention my daughter receives here is exceptional. The teachers genuinely care about each student's growth and development.",
      rating: 5,
    },
    {
      id: 2,
      name: "Mohammad Hassan",
      role: "Graduate",
      company: "Engineering Student",
      content:
        "Setaragan Zirak prepared me incredibly well for university. The strong foundation in sciences and critical thinking has been invaluable in my engineering studies.",
      rating: 5,
    },
    {
      id: 3,
      name: "Fatima Karimi",
      role: "Current Student",
      company: "Grade 12",
      content:
        "I love the supportive environment here. The computer lab and science facilities are amazing, and the teachers always encourage us to explore and ask questions.",
      rating: 5,
    },
  ];

  return (
    <AppLayout>
    <div className="min-h-screen bg-white">
      {/* Ultra Modern Hero */}
      <UltraModernHero
        title="Today's Stars, Tomorrow's Leaders"
        subtitle=""
        description="Excellence in education since 2019. Empowering students to become the heroes capable of keeping our society away from darkness."
        primaryCTA={{
          text: "Apply Now",
          href: "/admissions",
        }}
        secondaryCTA={{
          text: "Take a Tour",
          href: "/facilities",
        }}
      />

      {/* World-Class Facilities - Interactive Showcase */}
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 relative overflow-hidden">
        {/* Light Dynamic Background */}
        <div className="absolute inset-0 bg-gradient-to-br from-gray-50 via-white to-primary-50/30" />
        <div className="absolute inset-0 bg-gradient-to-t from-transparent via-white/10 to-transparent" />

        {/* Animated Background Elements - Reduced on mobile for performance */}
        <div className="hidden md:block absolute top-10 sm:top-20 left-10 sm:left-20 w-32 sm:w-64 h-32 sm:h-64 bg-secondary-400/10 rounded-full blur-2xl md:blur-3xl animate-pulse" />
        <div
          className="hidden md:block absolute bottom-10 sm:bottom-20 right-10 sm:right-20 w-40 sm:w-80 h-40 sm:h-80 bg-primary-400/10 rounded-full blur-2xl md:blur-3xl animate-bounce"
          style={{ animationDuration: "4s" }}
        />
        <div className="hidden lg:block absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-48 lg:w-96 h-48 lg:h-96 bg-secondary-400/5 rounded-full blur-3xl" />

        <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Enhanced Header */}
          <div className="text-center mb-8 sm:mb-12 lg:mb-16">
            <div className="inline-flex items-center gap-2 sm:gap-3 bg-gradient-to-r from-secondary-100 to-primary-100 text-secondary-700 px-4 sm:px-6 py-2 sm:py-3 rounded-full font-semibold text-xs sm:text-sm uppercase tracking-wider mb-6 sm:mb-8 border border-secondary-200/50">
              <Zap className="w-3 sm:w-4 h-3 sm:h-4" />
              <span>State-of-the-Art Learning Environment</span>
            </div>
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6 sm:mb-8 leading-tight">
              <span className="text-transparent bg-clip-text bg-gradient-to-r from-gray-900 via-secondary-600 to-primary-600">
                Where Dreams Take Flight
              </span>
            </h2>
            <p className="text-base sm:text-lg lg:text-xl xl:text-xl text-gray-600 max-w-4xl mx-auto leading-relaxed px-4">
              Step into tomorrow&apos;s classroom today. Our cutting-edge
              facilities are designed to inspire, innovate, and ignite the spark
              of discovery in every student.
            </p>
          </div>

          {/* Interactive Facilities Grid */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
            {/* Science Laboratory - Interactive Card */}
            <div className="group relative overflow-hidden rounded-2xl lg:rounded-3xl bg-white/80 backdrop-blur-xl border border-primary-200/50 hover:border-primary-400/60 transition-all duration-500 hover:scale-105 hover:shadow-2xl hover:shadow-primary-500/20 shadow-lg">
              {/* Background Pattern */}
              <div className="absolute inset-0 bg-gradient-to-br from-primary-50/80 to-emerald-100/60" />
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-primary-200/40 rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700"></div>

              <div className="relative z-10 p-6 sm:p-8 lg:p-10">
                <div className="flex flex-col items-center justify-center">
                  {/* Icon */}
                  <div className="w-16 sm:w-18 lg:w-20 h-16 sm:h-18 lg:h-20 bg-gradient-to-br from-primary-500 to-emerald-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-6 sm:mb-8 group-hover:scale-110  transition-all duration-300 shadow-lg">
                    <span className="text-2xl sm:text-3xl">🔬</span>
                  </div>

                  {/* Content */}
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4 sm:mb-6 group-hover:text-primary-700 transition-colors duration-300">
                    Science Laboratory
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed mb-6 sm:mb-8 group-hover:text-gray-700 transition-colors duration-300">
                  Teaching practical subjects is essential for enhancing
                  academic efficiency. Our laboratory facilities create an
                  environment that enables students to apply theories in
                  practice.
                </p>

                {/* Feature Points */}
                <div className="space-y-2 sm:space-y-3 mb-6 sm:mb-8">
                  {[
                    "Advanced microscopes and laboratory equipment",
                    "Chemistry and physics experiment stations",
                    "Safety equipment and protocols",
                    "Expert supervision and guidance",
                  ].map((feature, index) => (
                    <div
                      key={index}
                      className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-primary-700"
                    >
                      <div className="w-1.5 sm:w-2 h-1.5 sm:h-2 bg-primary-500 rounded-full flex-shrink-0" />
                      <span>{feature}</span>
                    </div>
                  ))}
                </div>

                {/* Action Button */}
                <div className="transition-opacity duration-300">
                  <Link
                    href="/facilities"
                    className="inline-flex items-center gap-2 bg-primary-500 hover:bg-primary-600 text-white px-3 sm:px-4 py-2 rounded-lg sm:rounded-xl font-semibold text-xs sm:text-sm transition-colors duration-200"
                  >
                    <span>Explore Science Lab</span>
                    <Target className="w-3 sm:w-4 h-3 sm:h-4" />
                  </Link>
                </div>
              </div>
            </div>

            {/* Computer Laboratory - Interactive Card */}
            <div className="group relative overflow-hidden rounded-2xl lg:rounded-3xl bg-white/80 backdrop-blur-xl border border-secondary-200/50 hover:border-secondary-400/60 transition-all duration-500 hover:scale-105 hover:shadow-2xl hover:shadow-secondary-500/20 shadow-lg">
              {/* Background Pattern */}
              <div className="absolute inset-0 bg-gradient-to-br from-secondary-50/80 to-primary-100/60" />
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-secondary-200/40 rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />

              <div className="relative z-10 p-6 sm:p-8 lg:p-10">
                <div className="flex flex-col items-center justify-center">
                  {/* Icon */}
                  <div className="w-16 sm:w-18 lg:w-20 h-16 sm:h-18 lg:h-20 bg-gradient-to-br from-secondary-500 to-primary-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-6 sm:mb-8 group-hover:scale-110  transition-all duration-300 shadow-lg">
                    <span className="text-2xl sm:text-3xl">💻</span>
                  </div>

                  {/* Content */}
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4 sm:mb-6 group-hover:text-secondary-700 transition-colors duration-300">
                    Computer Lab
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed mb-6 sm:mb-8 group-hover:text-gray-700 transition-colors duration-300">
                  The use of computers has brought about a great revolution in
                  study methods. With our Computer Lab, students learn skills of
                  the modern world and find new opportunities for creativity and
                  innovation.
                </p>

                {/* Feature Points */}
                <div className="space-y-2 sm:space-y-3 mb-6 sm:mb-8">
                  {[
                    "Modern desktop computers with latest software",
                    "High-speed internet connectivity",
                    "Programming and coding platforms",
                    "Digital design and multimedia tools",
                  ].map((feature, index) => (
                    <div
                      key={index}
                      className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-secondary-700"
                    >
                      <div className="w-1.5 sm:w-2 h-1.5 sm:h-2 bg-secondary-500 rounded-full flex-shrink-0" />
                      <span>{feature}</span>
                    </div>
                  ))}
                </div>

                {/* Action Button */}
                <div className="transition-opacity duration-300">
                  <Link
                    href="/facilities"
                    className="inline-flex items-center gap-2 bg-secondary-500 hover:bg-secondary-600 text-white px-3 sm:px-4 py-2 rounded-lg sm:rounded-xl font-semibold text-xs sm:text-sm transition-colors duration-200"
                  >
                    <span>Visit Computer Lab</span>
                    <Target className="w-3 sm:w-4 h-3 sm:h-4" />
                  </Link>
                </div>
              </div>
            </div>

            {/* Digital Library - Interactive Card */}
            <div className="group relative overflow-hidden rounded-2xl lg:rounded-3xl bg-white/80 backdrop-blur-xl border border-primary-200/50 hover:border-primary-400/60 transition-all duration-500 hover:scale-105 hover:shadow-2xl hover:shadow-primary-500/20 shadow-lg">
              {/* Background Pattern */}
              <div className="absolute inset-0 bg-gradient-to-br from-primary-50/80 to-secondary-100/60" />
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-primary-200/40 rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />

              <div className="relative z-10 p-6 sm:p-8 lg:p-10">
                <div className="flex flex-col items-center justify-center">
                  {/* Icon */}
                  <div className="w-16 sm:w-18 lg:w-20 h-16 sm:h-18 lg:h-20 bg-gradient-to-br from-primary-500 to-secondary-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-6 sm:mb-8 group-hover:scale-110  transition-all duration-300 shadow-lg">
                    <span className="text-2xl sm:text-3xl">📚</span>
                  </div>

                  {/* Content */}
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-4 sm:mb-6 group-hover:text-primary-700 transition-colors duration-300">
                    Library
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed mb-6 sm:mb-8 group-hover:text-gray-700 transition-colors duration-300">
                  Our library is a prime destination for goal attainment with
                  its rich features. Boasting a diverse array of resources and a
                  serene environment that fosters focused study.
                </p>

                {/* Feature Points */}
                <div className="space-y-2 sm:space-y-3 mb-6 sm:mb-8">
                  {[
                    "Diverse collection of books and digital resources",
                    "Quiet study areas and reading spaces",
                    "Cultural and educational programs",
                    "Research assistance and guidance",
                  ].map((feature, index) => (
                    <div
                      key={index}
                      className="flex items-center gap-2 sm:gap-3 text-xs sm:text-sm text-primary-700"
                    >
                      <div className="w-1.5 sm:w-2 h-1.5 sm:h-2 bg-primary-500 rounded-full flex-shrink-0" />
                      <span>{feature}</span>
                    </div>
                  ))}
                </div>

                {/* Action Button */}
                <div className=" transition-opacity duration-300">
                  <Link
                    href="/facilities"
                    className="inline-flex items-center gap-2 bg-primary-500 hover:bg-primary-600 text-white px-3 sm:px-4 py-2 rounded-lg sm:rounded-xl font-semibold text-xs sm:text-sm transition-colors duration-200"
                  >
                    <span>Browse Library</span>
                    <Target className="w-3 sm:w-4 h-3 sm:h-4" />
                  </Link>
                </div>
              </div>
            </div>
          </div>

          {/* Call-to-Action */}
          <div className="text-center mt-8 sm:mt-12">
            <div className="inline-flex flex-col sm:flex-row items-center gap-3 sm:gap-4 bg-white/70 backdrop-blur-sm border border-primary-200/50 rounded-xl sm:rounded-2xl px-4 sm:px-8 py-3 sm:py-4 shadow-lg">
              <div className="flex items-center gap-2 sm:gap-3">
                <Sparkles className="w-5 sm:w-6 h-5 sm:h-6 text-primary-600" />
                <span className="text-sm sm:text-base text-gray-800 font-semibold text-center sm:text-left">
                  Ready to experience world-class facilities?
                </span>
              </div>
              <Link
                href="/facilities"
                className="bg-gradient-to-r from-primary-500 to-secondary-500 hover:from-primary-600 hover:to-secondary-600 text-white px-4 sm:px-6 py-2 rounded-lg sm:rounded-xl font-semibold text-sm sm:text-base transition-colors duration-200 w-full sm:w-auto text-center"
              >
                Schedule a Tour
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Next-Gen Statistics */}
      <NextGenStats
        stats={stats}
        title="Impact Through Numbers"
        subtitle="Our Achievement Story"
      />

      {/* Why Choose Setaragan Zirak */}
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 relative overflow-hidden">
        {/* Background */}
        <div className="absolute inset-0 bg-gradient-to-br from-white via-secondary-50/30 to-primary-50/30" />
        <div className="hidden md:block absolute top-10 sm:top-20 left-5 sm:left-10 w-36 sm:w-72 h-36 sm:h-72 bg-secondary-400/10 rounded-full blur-2xl md:blur-3xl" />
        <div className="hidden md:block absolute bottom-10 sm:bottom-20 right-5 sm:right-10 w-48 sm:w-96 h-48 sm:h-96 bg-primary-400/10 rounded-full blur-2xl md:blur-3xl" />

        <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Header */}
          <div className="text-center mb-8 sm:mb-12 lg:mb-16">
            <div className="inline-flex items-center gap-2 bg-secondary-100 text-secondary-700 px-3 sm:px-4 py-1.5 sm:py-2 rounded-full font-semibold text-xs sm:text-sm uppercase tracking-wider mb-4 sm:mb-6">
              <Award className="w-3 sm:w-4 h-3 sm:h-4" />
              Why Choose Us
            </div>
            <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-4 sm:mb-6">
              <span className="text-transparent bg-clip-text bg-gradient-to-r from-gray-900 via-secondary-600 to-primary-600">
                Why Choose Setaragan Zirak?
              </span>
            </h2>
            <p className="text-base sm:text-lg lg:text-xl text-gray-600 max-w-3xl mx-auto px-4">
              Discover what makes our school the ideal choice for your
              child&apos;s educational journey
            </p>
          </div>

          {/* Features Grid */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
            <div className="group relative overflow-hidden bg-gradient-to-br from-primary-500/5 to-primary-600/10 backdrop-blur-sm border border-primary-200/30 rounded-2xl lg:rounded-3xl p-6 sm:p-8 hover:shadow-2xl hover:shadow-primary-500/20 hover:ring-2 hover:ring-primary-300/50 transition-all duration-500">
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-gradient-to-br from-white/20 to-transparent rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />
              <div className="relative z-10">
                <div className="flex flex-col items-center justify-center">
                  <div className="w-12 sm:w-14 lg:w-16 h-12 sm:h-14 lg:h-16 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-4 sm:mb-6 group-hover:brightness-110 transition-all duration-300">
                    <Users className="w-6 sm:w-7 lg:w-8 h-6 sm:h-7 lg:h-8 text-white" />
                  </div>
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-3 sm:mb-4">
                    Mutual Respect
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed">
                  Setaragan Zirak Private School is committed to mutual respect
                  between staff and students. We strive to create a safe and
                  friendly environment for both students and teachers.
                </p>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-gradient-to-br from-secondary-500/5 to-secondary-600/10 backdrop-blur-sm border border-secondary-200/30 rounded-2xl lg:rounded-3xl p-6 sm:p-8 hover:shadow-2xl hover:shadow-secondary-500/20 hover:ring-2 hover:ring-secondary-300/50 transition-all duration-500">
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-gradient-to-br from-white/20 to-transparent rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />
              <div className="relative z-10">
                <div className="flex flex-col items-center justify-center">
                  <div className="w-12 sm:w-14 lg:w-16 h-12 sm:h-14 lg:h-16 bg-gradient-to-br from-secondary-500 to-secondary-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-4 sm:mb-6 group-hover:brightness-110 transition-all duration-300">
                    <Target className="w-6 sm:w-7 lg:w-8 h-6 sm:h-7 lg:h-8 text-white" />
                  </div>
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-3 sm:mb-4 text-center">
                    Quality Result-Oriented Teaching
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed">
                  We always strive to ensure that teaching is result-oriented.
                  In result-based education, individuals gain the necessary
                  competencies to move on to the next level.
                </p>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-gradient-to-br from-emerald-500/5 to-emerald-600/10 backdrop-blur-sm border border-emerald-200/30 rounded-2xl lg:rounded-3xl p-6 sm:p-8 hover:shadow-2xl hover:shadow-emerald-500/20 hover:ring-2 hover:ring-emerald-300/50 transition-all duration-500">
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-gradient-to-br from-white/20 to-transparent rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />
              <div className="relative z-10">
                <div className="flex flex-col items-center justify-center">
                  <div className="w-12 sm:w-14 lg:w-16 h-12 sm:h-14 lg:h-16 bg-gradient-to-br from-emerald-500 to-emerald-600 rounded-xl lg:rounded-2xl flex items-center justify-center mb-4 sm:mb-6 group-hover:brightness-110 transition-all duration-300">
                    <BookOpen className="w-6 sm:w-7 lg:w-8 h-6 sm:h-7 lg:h-8 text-white" />
                  </div>
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-3 sm:mb-4 text-center">
                    Suitable Teaching Environment
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed">
                  The classes and facilities at Setaragan Zirak provide an
                  appropriate setting for academic and personal development,
                  allowing each individual to grow according to their talents.
                </p>
              </div>
            </div>

            <div className="group relative overflow-hidden bg-gradient-to-br from-primary-800/5 to-primary-900/10 backdrop-blur-sm border border-primary-200/30 rounded-2xl lg:rounded-3xl p-6 sm:p-8 hover:shadow-2xl hover:shadow-primary-800/20 hover:ring-2 hover:ring-primary-300/50 transition-all duration-500">
              <div className="absolute top-0 right-0 w-16 sm:w-24 lg:w-32 h-16 sm:h-24 lg:h-32 bg-gradient-to-br from-white/20 to-transparent rounded-full blur-xl lg:blur-2xl transform translate-x-4 sm:translate-x-8 -translate-y-4 sm:-translate-y-8 group-hover:scale-150 transition-transform duration-700" />
              <div className="relative z-10">
                <div className="flex flex-col items-center justify-center">
                  <div className="w-12 sm:w-14 lg:w-16 h-12 sm:h-14 lg:h-16 bg-gradient-to-br from-primary-800 to-primary-900 rounded-xl lg:rounded-2xl flex items-center justify-center mb-4 sm:mb-6 group-hover:brightness-110 transition-all duration-300">
                    <Sparkles className="w-6 sm:w-7 lg:w-8 h-6 sm:h-7 lg:h-8 text-white" />
                  </div>
                  <h3 className="text-lg sm:text-xl lg:text-2xl font-bold text-gray-900 mb-3 sm:mb-4">
                    Cultural Function
                  </h3>
                </div>
                <p className="text-sm sm:text-base text-gray-600 leading-relaxed">
                  The Setaragan Zirak team prioritizes working on the cultural
                  level of the community alongside educational programs through
                  cultural initiatives and scheduled activities.
                </p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Authentic School Story */}
      <section className="relative bg-gradient-to-br from-gray-50 via-white to-primary-50/20 py-16 sm:py-20 lg:py-24">
        {/* Background glow */}
        <div className="absolute top-10 left-10 w-48 h-48 bg-primary-300/10 rounded-full blur-3xl" />
        <div className="absolute bottom-10 right-10 w-64 h-64 bg-secondary-400/10 rounded-full blur-3xl" />

        <div className="relative z-10 max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Header */}
          <div className="text-center mb-12">
            <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-1.5 rounded-full text-xs font-semibold uppercase tracking-wider mb-5">
              <BookOpen className="w-4 h-4" />
              Our Story
            </div>
            <h2 className="text-3xl sm:text-4xl md:text-5xl font-bold text-gray-900 mb-4">
              Setaragan Zirak&apos;s Journey
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              Since 2019, we’ve been nurturing the next generation of heroes
              through inclusive education in Kabul.
            </p>
          </div>

          {/* Images Masonry */}
          <div className="grid grid-cols-2 sm:grid-cols-3 gap-4 mb-12 aspect-video max-sm:hidden">
            <div className="rounded-xl overflow-hidden shadow-lg order-1 sm:order-none">
              <div className="aspect-[3/4] sm:aspect-square">
                <Image
                  src="/images/home/raihana-768x1024.avif"
                  alt="Student activities"
                  width={0}
                  height={0}
                  className="object-cover object-center h-[400px] sm:h-full w-full"
                />
              </div>
            </div>
            <div className="rounded-xl overflow-hidden shadow-lg row-span-1 max-sm:col-span-2 sm:row-span-2 order-3 sm:order-none">
              <Image
                src="/images/home/IMG_8632-min-scaled.jpg"
                alt="Academic excellence"
                width={0}
                height={0}
                className="object-cover object-center h-full w-full"
              />
            </div>
            <div className="rounded-xl overflow-hidden shadow-lg order-4 sm:order-none">
              <div className="aspect-[3/4]">
                <Image
                  src="/images/home/hom2-768x1024.avif"
                  alt="Learning environment"
                  width={0}
                  height={0}
                  className="object-cover h-full w-full"
                />
              </div>
            </div>
            <div className="rounded-xl overflow-hidden shadow-lg order-5 sm:order-none">
              <div className="aspect-[3/4] sm:aspect-square">
                <Image
                  src="/images/home/sz-bene-768x1024.avif"
                  alt="School campus"
                  width={0}
                  height={0}
                  className="object-cover h-full w-full"
                />
              </div>
            </div>
            <div className="rounded-xl overflow-hidden shadow-lg order-2 sm:order-none">
              <div className="aspect-[3/4]">
                <Image
                  src="/images/home/hom1-edited.avif"
                  alt="Students in classroom"
                  width={0}
                  height={0}
                  className="object-cover object-center h-full w-full"
                />
              </div>
            </div>
          </div>
          <div className="relative sm:hidden mb-12">
            <div className="grid grid-cols-2 gap-3 sm:gap-6">
              <div className="space-y-3 sm:space-y-6">
                <div className="aspect-[3/4] bg-gradient-to-br from-primary-100 to-primary-200 rounded-xl sm:rounded-2xl overflow-hidden shadow-lg">
                  <Image
                    src="/images/home/hom1-edited.avif"
                    alt="Student activities at Setaragan Zirak Private School"
                    width={400}
                    height={533}
                    className="w-full h-full object-cover"
                  />
                </div>
                <div className="aspect-square bg-gradient-to-br from-secondary-100 to-secondary-200 rounded-xl sm:rounded-2xl overflow-hidden shadow-lg">
                  <Image
                    src="/images/home/hom2-768x1024.avif"
                    alt="Learning environment at Setaragan Zirak"
                    width={400}
                    height={400}
                    className="w-full h-full object-cover"
                  />
                </div>
              </div>
              <div className="space-y-3 sm:space-y-6 translate-y-4 sm:translate-y-8">
                <div className="aspect-square bg-gradient-to-br from-emerald-100 to-emerald-200 rounded-xl sm:rounded-2xl overflow-hidden shadow-lg">
                  <Image
                    src="/images/home/sz-bene-768x1024.avif"
                    alt="School campus and facilities"
                    width={400}
                    height={400}
                    className="w-full h-full object-cover"
                  />
                </div>
                <div className="aspect-[3/4] bg-gradient-to-br from-secondary-100 to-secondary-200 rounded-xl sm:rounded-2xl overflow-hidden shadow-lg">
                  <Image
                    src="/images/home/raihana-768x1024.avif"
                    alt="Academic excellence at Setaragan Zirak"
                    width={400}
                    height={533}
                    className="w-full h-full object-cover"
                  />
                </div>
              </div>
            </div>

            {/* Floating decorative elements - Hidden on mobile */}
            <div className="hidden sm:block absolute -top-2 sm:-top-4 -left-2 sm:-left-4 w-8 sm:w-16 h-8 sm:h-16 bg-primary-300/30 rounded-full blur-lg sm:blur-xl" />
            <div className="hidden sm:block absolute -bottom-2 sm:-bottom-4 -right-2 sm:-right-4 w-10 sm:w-20 h-10 sm:h-20 bg-secondary-300/30 rounded-full blur-lg sm:blur-xl" />
          </div>
          {/* Story Text in 2 Cols */}
          <div className="grid md:grid-cols-2 gap-8 text-lg text-gray-700 leading-relaxed">
            <p>
              <strong>
                Setaragan Zirak is a private school established in 2019.
              </strong>{" "}
              While our nation was undergoing a deep silence, we grew stronger
              with the urge to educate, fight, and nurture students as the only
              heroes capable of keeping our society away from darkness.
            </p>
            <p>
              Located in Kabul, Setaragan Zirak Private School is a place for
              growth for both female and male students from different
              backgrounds. Through our nonstop efforts under past circumstances,
              not only did we continue, but we also improved our ability to
              educate the future of Afghanistan.
            </p>
            <p className="md:col-span-2">
              <strong>
                Our student body consists equally of girls and boys.
              </strong>{" "}
              We believe in providing quality education regardless of gender,
              creating an inclusive environment where every student can thrive
              and reach their full potential.
            </p>
          </div>
        </div>
      </section>

      {/* Advanced Testimonials */}
      <AdvancedTestimonials
        testimonials={testimonials}
        title="Voices of Success"
        subtitle="Real Stories, Real Impact"
      />

      {/* Meet Our Core Team - Dynamic */}
      <DynamicLeadership />

      {/* Latest News */}
      <LatestNews
        title="Latest News & Updates"
        subtitle="Stay informed about school events, achievements, and announcements"
        maxPosts={3}
      />

      {/* Donation Section */}
      <section className="ppy-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gradient-to-br from-primary-50 to-secondary-50 relative overflow-hidden">
        {/* Simplified Background Pattern for mobile performance */}
        <div className="hidden md:block absolute inset-0 opacity-3 md:opacity-5">
          <div className="absolute top-0 left-0 w-32 sm:w-48 md:w-72 h-32 sm:h-48 md:h-72 bg-primary-500 rounded-full mix-blend-multiply filter blur-xl"></div>
          <div className="absolute top-0 right-0 w-32 sm:w-48 md:w-72 h-32 sm:h-48 md:h-72 bg-secondary-500 rounded-full mix-blend-multiply filter blur-xl"></div>
          <div className="absolute bottom-0 left-1/2 transform -translate-x-1/2 w-32 sm:w-48 md:w-72 h-32 sm:h-48 md:h-72 bg-emerald-500 rounded-full mix-blend-multiply filter blur-xl"></div>
        </div>

        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 relative">
          <div className="grid lg:grid-cols-2 gap-8 sm:gap-12 md:gap-16 items-center">
            {/* Content */}
            <div className="space-y-4 sm:space-y-6 md:space-y-8 order-2 lg:order-1">
              <div className="space-y-3 sm:space-y-4 md:space-y-6">
                <div className="inline-flex items-center gap-2 bg-gradient-to-r from-primary-100 to-secondary-100 text-primary-700 px-3 sm:px-4 md:px-6 py-1.5 sm:py-2 md:py-3 rounded-full text-xs sm:text-sm font-semibold">
                  <Sparkles className="w-3 sm:w-4 h-3 sm:h-4" />
                  <span>Support Education Excellence</span>
                </div>

                <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight text-transparent bg-clip-text bg-gradient-to-r from-primary-600 via-secondary-600 to-emerald-600">
                  <span className="">Invest in</span>
                  <br />
                  <span className="">Tomorrow&apos;s Leaders</span>
                </h2>

                <p className="text-base sm:text-lg md:text-xl text-gray-600 leading-relaxed">
                  Your generous donation helps us provide world-class education,
                  modern facilities, and opportunities for students who will
                  shape Afghanistan&apos;s future. Every contribution makes a
                  lasting impact.
                </p>
              </div>

              {/* Donation Impact Cards */}
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                <div className="bg-white/70 backdrop-blur-sm rounded-xl lg:rounded-2xl p-4 sm:p-6 border border-white/20 shadow-lg">
                  <div className="flex items-center gap-3 sm:gap-4 mb-3 sm:mb-4">
                    <div className="w-10 sm:w-12 h-10 sm:h-12 bg-gradient-to-r from-primary-500 to-primary-600 rounded-lg sm:rounded-xl flex items-center justify-center">
                      <BookOpen className="w-5 sm:w-6 h-5 sm:h-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-sm sm:text-base">
                        $50
                      </h3>
                      <p className="text-xs sm:text-sm text-gray-600">
                        Monthly Support
                      </p>
                    </div>
                  </div>
                  <p className="text-gray-700 text-xs sm:text-sm">
                    Provides textbooks and learning materials for one student
                    for a full semester
                  </p>
                </div>

                <div className="bg-white/70 backdrop-blur-sm rounded-xl lg:rounded-2xl p-4 sm:p-6 border border-white/20 shadow-lg">
                  <div className="flex items-center gap-3 sm:gap-4 mb-3 sm:mb-4">
                    <div className="w-10 sm:w-12 h-10 sm:h-12 bg-gradient-to-r from-secondary-500 to-secondary-600 rounded-lg sm:rounded-xl flex items-center justify-center">
                      <Users className="w-5 sm:w-6 h-5 sm:h-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-sm sm:text-base">
                        $200
                      </h3>
                      <p className="text-xs sm:text-sm text-gray-600">
                        Quarterly Support
                      </p>
                    </div>
                  </div>
                  <p className="text-gray-700 text-xs sm:text-sm">
                    Sponsors laboratory equipment and hands-on learning
                    experiences for multiple students
                  </p>
                </div>

                <div className="bg-white/70 backdrop-blur-sm rounded-xl lg:rounded-2xl p-4 sm:p-6 border border-white/20 shadow-lg">
                  <div className="flex items-center gap-3 sm:gap-4 mb-3 sm:mb-4">
                    <div className="w-10 sm:w-12 h-10 sm:h-12 bg-gradient-to-r from-emerald-500 to-emerald-600 rounded-lg sm:rounded-xl flex items-center justify-center">
                      <Award className="w-5 sm:w-6 h-5 sm:h-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-sm sm:text-base">
                        $500
                      </h3>
                      <p className="text-xs sm:text-sm text-gray-600">
                        Annual Support
                      </p>
                    </div>
                  </div>
                  <p className="text-gray-700 text-xs sm:text-sm">
                    Funds scholarships for talented students from disadvantaged
                    backgrounds
                  </p>
                </div>

                <div className="bg-white/70 backdrop-blur-sm rounded-xl lg:rounded-2xl p-4 sm:p-6 border border-white/20 shadow-lg">
                  <div className="flex items-center gap-3 sm:gap-4 mb-3 sm:mb-4">
                    <div className="w-10 sm:w-12 h-10 sm:h-12 bg-gradient-to-r from-orange-500 to-orange-600 rounded-lg sm:rounded-xl flex items-center justify-center">
                      <Zap className="w-5 sm:w-6 h-5 sm:h-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-bold text-gray-900 text-sm sm:text-base">
                        Custom
                      </h3>
                      <p className="text-xs sm:text-sm text-gray-600">
                        Your Choice
                      </p>
                    </div>
                  </div>
                  <p className="text-gray-700 text-xs sm:text-sm">
                    Support specific programs, infrastructure development, or
                    teacher training initiatives
                  </p>
                </div>
              </div>

              {/* CTA Buttons */}
              <div className="flex flex-col sm:flex-row gap-3 md:gap-4">
                <Link
                  href="/donate"
                  className="group inline-flex items-center justify-center gap-2 sm:gap-3 bg-gradient-to-r from-primary-500 to-secondary-500 hover:from-primary-600 hover:to-secondary-600 text-white px-4 sm:px-6 md:px-8 py-2.5 sm:py-3 md:py-4 rounded-lg sm:rounded-xl md:rounded-2xl font-bold text-sm sm:text-base md:text-lg transition-colors duration-200 shadow-lg"
                >
                  <span>Donate Now</span>
                  <Sparkles className="w-3 sm:w-4 md:w-5 h-3 sm:h-4 md:h-5" />
                </Link>

                <Link
                  href="/contact"
                  className="group inline-flex items-center justify-center gap-2 sm:gap-3 border-2 border-primary-300 text-primary-700 hover:bg-primary-50 px-4 sm:px-6 md:px-8 py-2.5 sm:py-3 md:py-4 rounded-lg sm:rounded-xl md:rounded-2xl font-bold text-sm sm:text-base md:text-lg transition-colors duration-200"
                >
                  <span>Learn More</span>
                  <Target className="w-3 sm:w-4 md:w-5 h-3 sm:h-4 md:h-5" />
                </Link>
              </div>
            </div>

            {/* Image */}
            <div className="relative order-1 lg:order-2">
              <div className="relative overflow-hidden rounded-xl sm:rounded-2xl md:rounded-3xl shadow-lg sm:shadow-xl md:shadow-2xl transform rotate-1 md:rotate-2 hover:rotate-0 transition-transform duration-300">
                <Image
                  src="/images/home/Science-Lab-new-photo-.avif"
                  alt="Students working in our modern science laboratory"
                  width={600}
                  height={400}
                  className="w-full h-auto object-cover"
                  priority={false}
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/20 to-transparent" />
                <div className="absolute bottom-3 sm:bottom-4 md:bottom-6 left-3 sm:left-4 md:left-6 right-3 sm:right-4 md:right-6">
                  <div className="bg-white/90 backdrop-blur-sm rounded-lg md:rounded-xl p-3 md:p-4">
                    <p className="text-xs md:text-sm font-semibold text-gray-900">
                      🔬 Advanced Science Laboratory
                    </p>
                    <p className="text-xs text-gray-600 mt-1">
                      Where future scientists discover their potential through
                      hands-on learning
                    </p>
                  </div>
                </div>
              </div>

              {/* Floating Stats - Hidden on mobile for better performance */}
              <div className="hidden lg:block absolute -top-4 sm:-top-6 md:-top-8 -left-4 sm:-left-6 md:-left-8 bg-white rounded-lg sm:rounded-xl md:rounded-2xl shadow-lg md:shadow-xl p-3 sm:p-4 md:p-6 border border-gray-100">
                <div className="text-center">
                  <div className="text-lg sm:text-xl md:text-2xl font-bold text-primary-600">
                    500+
                  </div>
                  <div className="text-xs font-semibold text-gray-600">
                    Students Supported
                  </div>
                </div>
              </div>

              <div className="hidden md:block absolute -bottom-4 sm:-bottom-6 md:-bottom-8 -right-4 sm:-right-6 md:-right-8 bg-white rounded-lg sm:rounded-xl md:rounded-2xl shadow-lg md:shadow-xl p-3 sm:p-4 md:p-6 border border-gray-100">
                <div className="text-center">
                  <div className="text-lg sm:text-xl md:text-2xl font-bold text-secondary-600">
                    15+
                  </div>
                  <div className="text-xs font-semibold text-gray-600">
                    Years of Excellence
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Ultimate CTA Section */}
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 relative overflow-hidden bg-gradient-to-br from-primary-400 via-primary-500 to-secondary-600">
        {/* Animated Background */}
        <div className="absolute inset-0">
          <div className="hidden md:block absolute top-10 sm:top-20 left-10 sm:left-20 w-32 sm:w-64 h-32 sm:h-64 bg-white/10 rounded-full blur-2xl md:blur-3xl animate-pulse" />
          <div
            className="hidden md:block absolute bottom-10 sm:bottom-20 right-10 sm:right-20 w-40 sm:w-80 h-40 sm:h-80 bg-white/5 rounded-full blur-2xl md:blur-3xl animate-bounce"
            style={{ animationDuration: "3s" }}
          />
          <div className="hidden lg:block absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-48 sm:w-96 h-48 sm:h-96 bg-secondary-200/10 rounded-full blur-3xl" />
        </div>

        <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <div className="inline-flex items-center gap-2 bg-white/10 backdrop-blur-sm text-white px-3 sm:px-4 py-1.5 sm:py-2 rounded-full font-semibold text-xs sm:text-sm uppercase tracking-wider mb-6 sm:mb-8 border border-white/20">
            <Target className="w-3 sm:w-4 h-3 sm:h-4" />
            Your Future Starts Here
          </div>

          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-white mb-6 sm:mb-8 drop-shadow-lg">
            <span className="text-white">Ready to Begin Your Journey?</span>
          </h2>

          <p className="text-base sm:text-lg lg:text-xl xl:text-2xl text-white/90 mb-8 sm:mb-12 max-w-3xl mx-auto leading-relaxed drop-shadow-sm px-4">
            Join a community where excellence is the standard and every
            student&apos;s potential is unleashed
          </p>

          <div className="flex flex-col sm:flex-row gap-4 sm:gap-6 justify-center">
            <Link
              href="/contact"
              className="group inline-flex items-center gap-2 sm:gap-3 bg-gradient-to-r from-primary-500 to-secondary-500 hover:from-primary-600 hover:to-secondary-600 text-white px-6 sm:px-8 lg:px-10 py-3 sm:py-4 lg:py-5 rounded-xl lg:rounded-2xl font-bold text-sm sm:text-base lg:text-lg transition-all duration-300 shadow-2xl hover:shadow-primary-500/25 transform hover:scale-105"
            >
              <span>Start Your Application</span>
              <Target className="w-4 sm:w-5 h-4 sm:h-5 group-hover:rotate-12 transition-transform duration-300" />
            </Link>

            <Link
              href="/donate"
              className="group inline-flex items-center gap-2 sm:gap-3 border-2 border-white/30 text-white hover:bg-white/10 backdrop-blur-sm px-6 sm:px-8 lg:px-10 py-3 sm:py-4 lg:py-5 rounded-xl lg:rounded-2xl font-bold text-sm sm:text-base lg:text-lg transition-all duration-300"
            >
              <span>Support Our Mission</span>
              <Sparkles className="w-4 sm:w-5 h-4 sm:h-5 group-hover:scale-110 transition-transform duration-300" />
            </Link>
          </div>
        </div>
      </section>
      </div>
      </AppLayout>
  );
}
