# Admin Panel - Migration Guide

## 📋 Overview
This admin panel is designed to be **completely modular and migration-ready**. It can be extracted to a standalone project with minimal effort.

## 🏗️ Architecture

### Directory Structure
```
src/
├── app/(admin)/           # Admin routes (isolated)
│   ├── layout.tsx         # Admin layout wrapper
│   └── admin/             # Admin pages
│       ├── page.tsx       # Dashboard
│       ├── news/          # News management
│       ├── teachers/      # Teacher management
│       ├── students/      # Student management
│       ├── contacts/      # Contact management
│       ├── donations/     # Donation management
│       └── settings/      # Settings
│
├── components/admin/      # Admin-only components (isolated)
│   ├── layout/
│   │   └── AdminLayout.tsx
│   └── ui/                # Admin UI components
│
├── lib/admin/             # Admin utilities (isolated)
│   └── utils.ts
│
└── types/                 # Type definitions
    └── admin.ts           # Admin types (isolated)
```

## 🔄 Migration Process

### Step 1: Copy Admin Code
Copy these directories to new project:
```bash
# From current project
src/app/(admin)/
src/components/admin/
src/lib/admin/
src/types/admin.ts
```

### Step 2: Update Imports
In new project, update import paths:
```typescript
// Before (same project)
import { AdminLayout } from '@/components/admin/layout/AdminLayout';

// After (standalone)
import { AdminLayout } from '@/components/layout/AdminLayout';
```

### Step 3: Add Dependencies
Install required packages in new project:
```bash
npm install next@latest react@latest react-dom@latest
npm install typescript @types/react @types/node
npm install tailwindcss postcss autoprefixer
npm install lucide-react clsx tailwind-merge
npm install prisma @prisma/client
```

### Step 4: Configure Database
Copy `prisma/schema.prisma` and update connection:
```prisma
datasource db {
  provider = "mysql"
  url      = env("DATABASE_URL")
}
```

### Step 5: Environment Variables
Create `.env` in new project:
```env
DATABASE_URL="mysql://user:password@localhost:3306/school_admin"
NEXTAUTH_SECRET="your-secret-key"
NEXTAUTH_URL="http://localhost:3001"
```

### Step 6: Run Migrations
```bash
npx prisma generate
npx prisma db push
```

## 🚀 Running Standalone

After migration:
```bash
npm run dev  # Runs on separate port (e.g., 3001)
```

## 🔐 Authentication

### Current Implementation (Same Project)
- Shared session with public site
- Middleware protects `/admin` routes

### Standalone Implementation
Add `src/middleware.ts`:
```typescript
import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';

export function middleware(request: NextRequest) {
  const token = request.cookies.get('admin-token');
  
  if (!token && request.nextUrl.pathname.startsWith('/admin')) {
    return NextResponse.redirect(new URL('/login', request.url));
  }
  
  return NextResponse.next();
}

export const config = {
  matcher: '/admin/:path*',
};
```

## 📡 API Integration

### Current Setup (Same Project)
API routes at: `src/app/api/admin/*`

### Standalone Setup
- Keep same API structure
- Update base URL in admin utilities
- Add CORS if public site needs to access admin APIs

## 🎨 Styling

Admin panel uses:
- Tailwind CSS (independent configuration)
- No dependency on public site styles
- Separate color scheme

## 📦 Zero Dependencies on Public Code

### ✅ Good (Isolated)
```typescript
import { cn } from '@/lib/admin/utils';
import { AdminUser } from '@/types/admin';
```

### ❌ Bad (Coupled)
```typescript
import { Button } from '@/components/common/ui/button';  // DON'T DO THIS
import { formatDate } from '@/lib/utils';                 // DON'T DO THIS
```

## 🔧 Configuration Files

### next.config.js
```javascript
/** @type {import('next').NextConfig} */
const nextConfig = {
  // Admin-specific config
};

export default nextConfig;
```

### tailwind.config.js
```javascript
/** @type {import('tailwindcss').Config} */
module.exports = {
  content: [
    './src/**/*.{js,ts,jsx,tsx,mdx}',
  ],
  theme: {
    extend: {
      // Admin-specific theme
    },
  },
};
```

## 📝 Checklist Before Migration

- [ ] No imports from `@/components/common/`
- [ ] No imports from `@/lib/utils` (use `@/lib/admin/utils`)
- [ ] All admin types in `@/types/admin.ts`
- [ ] All admin utilities in `@/lib/admin/`
- [ ] Environment variables documented
- [ ] Database schema exported
- [ ] API routes independent
- [ ] Authentication self-contained

## 🎯 Benefits of This Architecture

1. **Easy Migration**: Copy 4 directories, update imports
2. **No Code Duplication**: Clean separation from day one
3. **Independent Deployment**: Can run on different server/port
4. **Security**: Admin can be on private network
5. **Scalability**: Admin and public scale independently
6. **Maintenance**: Changes don't affect each other

## ⚠️ Important Notes

1. **Never** import from public components in admin code
2. **Always** use admin-specific utilities
3. **Keep** admin types separate
4. **Document** any shared database schema changes
5. **Test** migration process periodically

## 🆘 Support

If migration is needed:
1. Follow this guide step-by-step
2. Test authentication first
3. Verify database connections
4. Check all API endpoints
5. Test file uploads (if any)

## 📅 Estimated Migration Time

- **Setup new project**: 30 minutes
- **Copy & update imports**: 1-2 hours
- **Configure database**: 30 minutes
- **Testing**: 1-2 hours
- **Total**: 3-5 hours

---

**Last Updated**: October 22, 2025  
**Version**: 1.0.0  
**Migration Status**: ✅ Ready for standalone deployment
