'use client';

import { useState } from 'react';
import { useFormik } from 'formik';
import { useRouter } from 'next/navigation';
import { Upload, ArrowLeft } from 'lucide-react';
import {
  useCreateLeadership,
  useUpdateLeadership,
  leadershipSchema,
  type Leadership,
} from '@/hooks/useLeadership';
import RichTextEditor from '@/components/common/ui/rich-text-editor';
import Image from 'next/image';

interface LeadershipFormProps {
  leadership?: Leadership;
}

const POSITION_OPTIONS = [
  'Founder & Director',
  'Principal',
  'Vice Principal',
  'Communication Coordinator',
  'Finance Manager',
  'Academic Lead',
  'Other',
];

const GENDER_OPTIONS = ['male', 'female'];

export default function LeadershipForm({ leadership }: LeadershipFormProps) {
  const router = useRouter();
  const isEditing = !!leadership;
  const createMutation = useCreateLeadership();
  const updateMutation = useUpdateLeadership();
  const [uploadingImage, setUploadingImage] = useState(false);
  const [selectedGender, setSelectedGender] = useState<'male' | 'female'>('male');

  const isLoading = createMutation.isPending || updateMutation.isPending || uploadingImage;
  const error = createMutation.error || updateMutation.error;

  const parseExpertise = (expertise: string | string[] | undefined): string => {
    if (!expertise) return '';
    if (typeof expertise === 'string') {
      try {
        const parsed = JSON.parse(expertise);
        if (Array.isArray(parsed)) return parsed.join(', ');
      } catch {
        return expertise;
      }
    }
    if (Array.isArray(expertise)) return expertise.join(', ');
    return '';
  };

  const stringifyExpertise = (expertise: string): string => {
    if (!expertise.trim()) return '[]';
    const items = expertise
      .split(',')
      .map((item) => item.trim())
      .filter((item) => item.length > 0);
    return JSON.stringify(items);
  };

  const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    setUploadingImage(true);
    try {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('gender', selectedGender);
      if (leadership?.image) {
        formData.append('oldImagePath', leadership.image);
      }

      const response = await fetch('/api/admin/upload/leadership', {
        method: 'POST',
        body: formData,
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.message || 'Upload failed');
      }

      formik.setFieldValue('image', data.imageUrl);
    } catch (err) {
      alert(err instanceof Error ? err.message : 'Failed to upload image');
    } finally {
      setUploadingImage(false);
    }
  };

  const formik = useFormik({
    initialValues: {
      name: leadership?.name || '',
      position: leadership?.position || '',
      shortDescription: leadership?.shortDescription || '',
      bio: leadership?.bio || '',
      image: leadership?.image || '',
      expertise: parseExpertise(leadership?.expertise),
      active: leadership?.active ?? true,
    },
    validationSchema: leadershipSchema,
    onSubmit: async (values) => {
      try {
        const submitValues = {
          ...values,
          expertise: stringifyExpertise(values.expertise || ''),
        };

        if (isEditing && leadership) {
          await updateMutation.mutateAsync({
            id: leadership.id,
            values: submitValues,
          });
        } else {
          await createMutation.mutateAsync(submitValues);
        }

        formik.resetForm();
        router.push('/admin/leadership');
      } catch (err) {
        console.error('Form submission error:', err);
      }
    },
    enableReinitialize: true,
  });

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="sticky top-0 bg-white border-b border-gray-200 shadow-sm z-10">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center gap-4">
            <button
              onClick={() => router.back()}
              className="p-2 hover:bg-gray-100 rounded-lg transition"
              title="Go back"
            >
              <ArrowLeft size={20} className="text-gray-600" />
            </button>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">
                {isEditing ? 'Edit Leadership Member' : 'Add Leadership Member'}
              </h1>
              <p className="text-sm text-gray-600 mt-1">
                {isEditing ? 'Update member information' : 'Add a new leadership member'}
              </p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <form onSubmit={formik.handleSubmit} className="space-y-8">
          {error && (
            <div className="p-4 bg-red-50 text-red-700 rounded-lg border border-red-200">
              {error instanceof Error ? error.message : 'An error occurred'}
            </div>
          )}

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Profile Image</h3>
            <div className="flex gap-4 items-start">
              <div className="flex-shrink-0">
                {formik.values.image ? (
                  <Image
                    src={formik.values.image}
                    alt="Preview"
                    width={96}
                    height={96}
                    className="w-24 h-24 rounded-lg object-cover border-2 border-gray-200"
                  />
                ) : (
                  <div className="w-24 h-24 rounded-lg bg-gray-200 flex items-center justify-center border-2 border-gray-300">
                    <Upload className="w-8 h-8 text-gray-400" />
                  </div>
                )}
              </div>

              <div className="flex-1">
                <div className="flex flex-col gap-2 mb-3">
                  <label className="block text-xs font-medium text-gray-600 mb-1">
                    Gender (for organizing images)
                  </label>
                  <select
                    value={selectedGender}
                    onChange={(e) => setSelectedGender(e.target.value as 'male' | 'female')}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-primary-500"
                  >
                    {GENDER_OPTIONS.map((g) => (
                      <option key={g} value={g}>
                        {g.charAt(0).toUpperCase() + g.slice(1)}
                      </option>
                    ))}
                  </select>
                </div>

                <input
                  type="file"
                  accept="image/*"
                  onChange={handleImageUpload}
                  disabled={uploadingImage || isLoading}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-primary-500 disabled:bg-gray-100"
                />
                <p className="text-xs text-gray-500 mt-2">Max 2MB • JPG, PNG, GIF, WebP</p>
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Basic Information</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Name *</label>
                <input
                  type="text"
                  name="name"
                  value={formik.values.name}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className={`w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 ${
                    formik.touched.name && formik.errors.name ? 'border-red-500' : 'border-gray-300'
                  }`}
                  placeholder="Full name"
                  disabled={isLoading}
                />
                {formik.touched.name && formik.errors.name && (
                  <p className="text-xs text-red-600 mt-1">{formik.errors.name}</p>
                )}
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Position *</label>
                <select
                  name="position"
                  value={formik.values.position}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  className={`w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 ${
                    formik.touched.position && formik.errors.position
                      ? 'border-red-500'
                      : 'border-gray-300'
                  }`}
                  disabled={isLoading}
                >
                  <option value="">Select position</option>
                  {POSITION_OPTIONS.map((pos) => (
                    <option key={pos} value={pos}>
                      {pos}
                    </option>
                  ))}
                </select>
                {formik.touched.position && formik.errors.position && (
                  <p className="text-xs text-red-600 mt-1">{formik.errors.position}</p>
                )}
              </div>
            </div>

            <div>
              <label className="flex items-center gap-2">
                <input
                  type="checkbox"
                  name="active"
                  checked={formik.values.active}
                  onChange={formik.handleChange}
                  disabled={isLoading}
                  className="w-4 h-4 rounded border-gray-300 focus:ring-2 focus:ring-primary-500"
                />
                <span className="text-sm font-medium text-gray-900">Active</span>
              </label>
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Short Description</h3>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Description
              <span className="text-xs text-gray-500 font-normal ml-2">(1-2 sentences for home page)</span>
            </label>
            <textarea
              name="shortDescription"
              value={formik.values.shortDescription}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              rows={2}
              disabled={isLoading}
              className={`w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 resize-none ${
                formik.touched.shortDescription && formik.errors.shortDescription
                  ? 'border-red-500'
                  : 'border-gray-300'
              }`}
              placeholder="Brief description for home page..."
            />
            <div className="flex justify-between items-center mt-1">
              <span className={`text-xs ${
                (formik.values.shortDescription?.length || 0) > 300 ? 'text-red-600' : 'text-gray-500'
              }`}>
                {formik.values.shortDescription?.length || 0}/300
              </span>
              {formik.touched.shortDescription && formik.errors.shortDescription && (
                <p className="text-xs text-red-600">{formik.errors.shortDescription}</p>
              )}
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Full Biography</h3>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Biography
              <span className="text-xs text-gray-500 font-normal ml-2">(Detailed bio for team page)</span>
            </label>
            <RichTextEditor
              value={formik.values.bio || ''}
              onChange={(content) => formik.setFieldValue('bio', content)}
              readOnly={isLoading}
            />
            {formik.touched.bio && formik.errors.bio && (
              <p className="text-xs text-red-600 mt-1">{formik.errors.bio}</p>
            )}
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Areas of Expertise</h3>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Expertise
              <span className="text-xs text-gray-500 font-normal ml-2">(Separate with commas)</span>
            </label>
            <textarea
              name="expertise"
              value={formik.values.expertise}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              rows={3}
              disabled={isLoading}
              className={`w-full px-3 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 resize-none ${
                formik.touched.expertise && formik.errors.expertise ? 'border-red-500' : 'border-gray-300'
              }`}
              placeholder="e.g., Educational Leadership, Curriculum Development, Academic Administration"
            />

            {formik.values.expertise && (
              <div className="mt-3 flex flex-wrap gap-2">
                {formik.values.expertise
                  .split(',')
                  .map((item) => item.trim())
                  .filter((item) => item.length > 0)
                  .map((item, idx) => (
                    <span
                      key={idx}
                      className="inline-flex items-center gap-1 bg-primary-100 text-primary-700 px-3 py-1 rounded-full text-sm"
                    >
                      {item}
                    </span>
                  ))}
              </div>
            )}

            {formik.touched.expertise && formik.errors.expertise && (
              <p className="text-xs text-red-600 mt-1">{formik.errors.expertise}</p>
            )}
          </div>

          <div className="flex flex-col sm:flex-row gap-3 justify-end sticky bottom-0 bg-white border-t border-gray-200 p-4 -mx-4 sm:-mx-6 lg:-mx-8">
            <button
              type="button"
              onClick={() => router.back()}
              disabled={isLoading}
              className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition disabled:opacity-50 order-2 sm:order-1"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={isLoading}
              className="px-4 py-2 bg-primary-600 text-white rounded-lg hover:bg-primary-700 transition disabled:opacity-50 order-1 sm:order-2"
            >
              {uploadingImage ? 'Uploading...' : isLoading ? 'Saving...' : isEditing ? 'Update' : 'Add'}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}
