"use client";

import { useState } from "react";
import { useFormik } from "formik";
import * as Yup from "yup";
import { useCreateManualDonation } from "@/hooks";
import Button from "@/components/common/ui/button";
import Card from "@/components/common/ui/card";
import InputField from "@/components/common/ui/input-field";
import SelectField from "@/components/common/ui/select-field";
import { Modal } from "@/components/common/ui";

interface ManualDonationFormProps {
  onSuccess?: () => void;
  onCancel?: () => void;
}

const manualDonationSchema = Yup.object().shape({
  name: Yup.string().required("Donor name is required").min(2),
  email: Yup.string().email("Invalid email").required("Email is required"),
  phone: Yup.string(),
  country: Yup.string(),
  amount: Yup.number()
    .required("Amount is required")
    .positive("Amount must be positive"),
  currency: Yup.string().required("Currency is required").default("USD"),
  category: Yup.string(),
  paymentMethod: Yup.string().required("Payment method is required"),
  paymentStatus: Yup.string()
    .required("Payment status is required")
    .oneOf(["pending", "completed", "failed"]),
  anonymous: Yup.boolean().default(false),
  message: Yup.string().max(1000),
  transactionId: Yup.string(),
});

type ManualDonationFormValues = Yup.InferType<typeof manualDonationSchema>;

export default function ManualDonationForm({
  onSuccess,
  onCancel,
}: ManualDonationFormProps) {
  const [showForm, setShowForm] = useState(false);
  const createDonationMutation = useCreateManualDonation();

  const formik = useFormik<ManualDonationFormValues>({
    initialValues: {
      name: "",
      email: "",
      phone: "",
      country: "",
      amount: 0,
      currency: "USD",
      category: "General",
      paymentMethod: "Cash",
      paymentStatus: "completed",
      anonymous: false,
      message: "",
      transactionId: "",
    },
    validationSchema: manualDonationSchema,
    onSubmit: (values) => {
      createDonationMutation.mutate(
        {
          ...values,
          paymentStatus: values.paymentStatus as
            | "pending"
            | "completed"
            | "failed",
        },
        {
          onSuccess: () => {
            formik.resetForm();
            setShowForm(false);
            onSuccess?.();
          },
        }
      );
    },
  });

  return (
    <div className="mb-6">
      <Button onClick={() => setShowForm(true)} color="primary">
        + Add Manual Donation
      </Button>
      <Modal
        isOpen={showForm}
        size="lg"
        onClose={() => {
          setShowForm(false);
          formik.resetForm();
          onCancel?.();
        }}
        title="Add Manual Donation"
      >
        <div className="pt-3">
          <form onSubmit={formik.handleSubmit} className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Donor Name */}
              <InputField
                label="Donor Name *"
                type="text"
                placeholder="Donor full name"
                {...formik.getFieldProps("name")}
                error={formik.touched.name ? formik.errors.name : undefined}
              />

              {/* Email */}
              <InputField
                label="Email *"
                type="email"
                placeholder="donor@example.com"
                {...formik.getFieldProps("email")}
                error={formik.touched.email ? formik.errors.email : undefined}
              />

              {/* Phone */}
              <InputField
                label="Phone"
                type="tel"
                placeholder="+1 234 567 8900"
                {...formik.getFieldProps("phone")}
              />

              {/* Country */}
              <SelectField label="Country" {...formik.getFieldProps("country")}>
                <option value="">Select Country</option>
                {[
                  "Afghanistan",
                  "Pakistan",
                  "Iran",
                  "Turkey",
                  "United Arab Emirates",
                  "Saudi Arabia",
                  "Qatar",
                  "Kuwait",
                  "Bahrain",
                  "Oman",
                  "Malaysia",
                  "India",
                  "Germany",
                  "France",
                  "Italy",
                  "Spain",
                  "Sweden",
                  "Norway",
                  "Netherlands",
                  "Belgium",
                  "United Kingdom",
                  "United States",
                  "Canada",
                  "Australia",
                  "New Zealand",
                  "Other",
                ].map((country) => (
                  <option key={country} value={country}>
                    {country}
                  </option>
                ))}
              </SelectField>

              {/* Amount */}
              <InputField
                label="Amount *"
                type="number"
                step="0.01"
                placeholder="0.00"
                {...formik.getFieldProps("amount")}
                error={formik.touched.amount ? formik.errors.amount : undefined}
              />

              {/* Currency */}
              <SelectField
                label="Currency *"
                {...formik.getFieldProps("currency")}
                error={
                  formik.touched.currency ? formik.errors.currency : undefined
                }
              >
                <option value="USD">USD ($) - United States</option>
                <option value="EUR">EUR (€) - Europe</option>
                <option value="GBP">GBP (£) - United Kingdom</option>
                <option value="PKR">PKR (₨) - Pakistan</option>
                <option value="IRR">IRR (﷼) - Iran</option>
                <option value="AFN">AFN (؋) - Afghanistan</option>
                <option value="AED">AED (د.إ) - United Arab Emirates</option>
              </SelectField>

              {/* Payment Method */}
              <SelectField
                label="Payment Method *"
                {...formik.getFieldProps("paymentMethod")}
                error={
                  formik.touched.paymentMethod
                    ? formik.errors.paymentMethod
                    : undefined
                }
              >
                <option value="Cash">Cash</option>
                <option value="Check">Check</option>
                <option value="Bank Transfer">Bank Transfer</option>
                <option value="Wire Transfer">Wire Transfer</option>
                <option value="Credit Card">Credit Card</option>
                <option value="Mobile Payment">Mobile Payment</option>
              </SelectField>

              {/* Category */}
              <SelectField
                label="Category"
                {...formik.getFieldProps("category")}
              >
                <option value="General">General</option>
                <option value="Scholarships">Scholarships</option>
                <option value="Building">Building</option>
                <option value="Equipment">Equipment</option>
                <option value="Programs">Programs</option>
                <option value="Facilities">Facilities</option>
              </SelectField>

              {/* Payment Status */}
              <SelectField
                label="Payment Status *"
                {...formik.getFieldProps("paymentStatus")}
                error={
                  formik.touched.paymentStatus
                    ? formik.errors.paymentStatus
                    : undefined
                }
              >
                <option value="completed">Completed</option>
                <option value="pending">Pending</option>
                <option value="failed">Failed</option>
              </SelectField>

              {/* Transaction ID */}
              <InputField
                label="Transaction ID"
                type="text"
                placeholder="TXN-001234"
                {...formik.getFieldProps("transactionId")}
              />

              {/* Anonymous */}
              <div className="flex items-center">
                <label className="flex items-center gap-2 cursor-pointer">
                  <input
                    type="checkbox"
                    {...formik.getFieldProps("anonymous")}
                    className="w-4 h-4 rounded border-gray-300"
                  />
                  <span className="text-sm font-medium text-gray-700">
                    Anonymous Donor
                  </span>
                </label>
              </div>
            </div>

            {/* Message */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Message / Notes
              </label>
              <textarea
                {...formik.getFieldProps("message")}
                placeholder="Optional: Add any notes about this donation"
                rows={3}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 resize-none"
              />
              {formik.touched.message && formik.errors.message && (
                <p className="text-sm text-red-600 mt-1">
                  {formik.errors.message}
                </p>
              )}
            </div>

            {/* Form Status Messages */}
            {createDonationMutation.isError && (
              <div className="p-3 bg-red-50 border border-red-200 rounded text-red-700 text-sm">
                {createDonationMutation.error?.message ||
                  "Failed to add donation"}
              </div>
            )}

            {createDonationMutation.isSuccess && (
              <div className="p-3 bg-green-50 border border-green-200 rounded text-green-700 text-sm">
                Donation added successfully!
              </div>
            )}

            {/* Submit Buttons */}
            <div className="flex justify-end gap-3 pt-4 border-t">
              <Button
                variant="outlined"
                onClick={() => {
                  setShowForm(false);
                  formik.resetForm();
                  onCancel?.();
                }}
              >
                Cancel
              </Button>
              <Button
                type="submit"
                disabled={createDonationMutation.isPending || !formik.isValid}
              >
                {createDonationMutation.isPending
                  ? "Adding..."
                  : "Add Donation"}
              </Button>
            </div>
          </form>
        </div>
      </Modal>
    </div>
  );
}
