'use client';

import { useState, useEffect } from 'react';
import {
  ADMIN_PAGES_CONFIG,
  getPagesByCategory,
  type UserPagePermissions,
} from '@/lib/admin/page-permissions';

interface PermissionSelectorProps {
  currentPermissions: UserPagePermissions | null;
  userRole: string;
  onPermissionsChange: (permissions: UserPagePermissions) => void;
  disabled?: boolean;
}

export default function PermissionSelector({
  currentPermissions,
  userRole,
  onPermissionsChange,
  disabled = false,
}: PermissionSelectorProps) {
  const [permissions, setPermissions] = useState<UserPagePermissions>(
    currentPermissions || {}
  );

  // Update local state when currentPermissions prop changes
  useEffect(() => {
    setPermissions(currentPermissions || {});
  }, [currentPermissions]);

  const pagesByCategory = getPagesByCategory();

  // Handle page-level checkbox (select/deselect all actions)
  const handlePageToggle = (pageId: string, allActionIds: string[]) => {
    const newPermissions = { ...permissions };

    if (newPermissions[pageId]?.length === allActionIds.length) {
      // All actions selected - deselect all
      newPermissions[pageId] = [];
    } else {
      // Select all actions
      newPermissions[pageId] = allActionIds;
    }

    setPermissions(newPermissions);
    onPermissionsChange(newPermissions);
  };
  console.log('PermissionSelector permissions state:', permissions);
  // Handle individual action checkbox
  const handleActionToggle = (pageId: string, actionId: string) => {
    const newPermissions = { ...permissions };

    if (!newPermissions[pageId]) {
      newPermissions[pageId] = [];
    }

    if (newPermissions[pageId].includes(actionId)) {
      // Remove action
      newPermissions[pageId] = newPermissions[pageId].filter((a) => a !== actionId);
    } else {
      // Add action
      newPermissions[pageId] = [...newPermissions[pageId], actionId];
    }

    setPermissions(newPermissions);
    onPermissionsChange(newPermissions);
  };

  // Check if all actions are selected for a page
  const isPageFullySelected = (pageId: string, actionsCount: number) => {
    return permissions[pageId]?.length === actionsCount;
  };

  // Check if some (but not all) actions are selected
  const isPagePartiallySelected = (pageId: string, actionsCount: number) => {
    const selectedCount = permissions[pageId]?.length || 0;
    return selectedCount > 0 && selectedCount < actionsCount;
  };

  if (userRole === 'admin') {
    return (
      <div className="rounded-lg border border-blue-200 bg-blue-50 p-4">
        <p className="text-sm text-blue-800">
          <strong>Super Admin</strong> - This user has full access to all pages and features.
        </p>
      </div>
    );
  }

  const renderCategorySection = (
    categoryName: string,
    pages: typeof ADMIN_PAGES_CONFIG
  ) => (
    <div key={categoryName} className="space-y-3">
      <h4 className="text-sm font-semibold text-gray-700 uppercase tracking-wider">
        {categoryName}
      </h4>
      <div className="space-y-2">
        {pages.map((page) => {
          const allActionIds = page.actions.map((a) => a.id);
          const isFullySelected = isPageFullySelected(page.id, allActionIds.length);
          const isPartiallySelected = isPagePartiallySelected(page.id, allActionIds.length);

          return (
            <div
              key={page.id}
              className="rounded-lg border border-gray-200 bg-white p-4 hover:border-gray-300 transition-colors"
            >
              {/* Page Header */}
              <div className="flex items-start justify-between mb-3">
                <div className="flex items-start gap-3">
                  <input
                    type="checkbox"
                    checked={isFullySelected}
                    ref={(input) => {
                      if (input) input.indeterminate = isPartiallySelected;
                    }}
                    onChange={() => handlePageToggle(page.id, allActionIds)}
                    disabled={disabled || page.adminOnly}
                    className="mt-1 h-4 w-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500 disabled:opacity-50"
                  />
                  <div>
                    <label className="block font-medium text-gray-900">
                      {page.name}
                      {page.adminOnly && (
                        <span className="ml-2 text-xs px-2 py-0.5 rounded-full bg-red-100 text-red-700">
                          Admin Only
                        </span>
                      )}
                    </label>
                    <p className="text-sm text-gray-500 mt-0.5">{page.description}</p>
                  </div>
                </div>
                <span className="text-xs text-gray-400 whitespace-nowrap">{page.path}</span>
              </div>

              {/* Action Checkboxes */}
              {!page.adminOnly && (
                <div className="ml-7 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-2 mt-3 pt-3 border-t border-gray-100">
                  {page.actions.map((action) => (
                    <label
                      key={action.id}
                      className="flex items-start gap-2 cursor-pointer group"
                    >
                      <input
                        type="checkbox"
                        checked={permissions[page.id]?.includes(action.id) || false}
                        onChange={() => handleActionToggle(page.id, action.id)}
                        disabled={disabled}
                        className="mt-0.5 h-3.5 w-3.5 text-blue-600 rounded border-gray-300 focus:ring-blue-500 disabled:opacity-50"
                      />
                      <div className="text-sm">
                        <span className="text-gray-700 group-hover:text-gray-900 font-medium">
                          {action.label}
                        </span>
                        <p className="text-xs text-gray-500">{action.description}</p>
                      </div>
                    </label>
                  ))}
                </div>
              )}
            </div>
          );
        })}
      </div>
    </div>
  );

  return (
    <div className="space-y-6">
      <div className="rounded-lg bg-blue-50 border border-blue-200 p-4">
        <p className="text-sm text-blue-800">
          <strong>Custom Permissions:</strong> Select specific pages and actions this user can
          access. Leave unchecked to deny access.
        </p>
      </div>

      {/* Content Pages */}
      {renderCategorySection('Content Management', pagesByCategory.content)}

      {/* Management Pages */}
      {renderCategorySection('Data Management', pagesByCategory.management)}

      {/* System Pages */}
      {renderCategorySection('System', pagesByCategory.system)}

      {/* Permission Summary */}
      <div className="rounded-lg bg-gray-50 border border-gray-200 p-4">
        <h4 className="text-sm font-semibold text-gray-700 mb-2">Permission Summary</h4>
        <div className="text-sm text-gray-600">
          <p>
            <strong>Pages with access:</strong>{' '}
            {Object.keys(permissions).filter((pageId) => permissions[pageId]?.length > 0).length}{' '}
            / {ADMIN_PAGES_CONFIG.length}
          </p>
          <p className="mt-1">
            <strong>Total permissions granted:</strong>{' '}
            {Object.values(permissions).reduce((sum, actions) => sum + actions.length, 0)}
          </p>
        </div>
      </div>
    </div>
  );
}
