'use client';

import { useFormik } from 'formik';
import { useRouter } from 'next/navigation';
import { Student, StudentFormData, studentSchema, useUpdateStudent, useCreateStudent } from '@/hooks';
import { useImageUpload } from '@/hooks/useImageUpload';
import InputField from '@/components/common/ui/input-field';
import RichTextEditor from '@/components/common/ui/rich-text-editor';
import Image from 'next/image';
import { Upload, ArrowLeft } from 'lucide-react';

interface StudentFormProps {
  student?: Student;
}

export default function StudentForm({ student }: StudentFormProps) {
  const router = useRouter();
  const createMutation = useCreateStudent();
  const updateMutation = useUpdateStudent();
  const { uploading, uploadImage } = useImageUpload();
  
  const isLoading = createMutation.isPending || updateMutation.isPending || uploading;
  const error = createMutation.error || updateMutation.error;
  const isEditing = !!student;

  const parseAchievements = (achievements?: string | string[]): string => {
    if (!achievements) return "";
    console.log('Parsing achievements:', achievements);
    try {
      console.log('Parsed achievements:', student?.achievements);
      // const parsed = JSON.parse(achievements);
      if (typeof achievements === 'string') return JSON.parse(achievements).join(", ");
      if (Array.isArray(achievements)) return achievements.join(", ");
      return achievements; // If parsed but not an array, return the original string
    } catch (error) {
      console.log('Parsed achievements:', achievements, error);
      return String(achievements); // On parse error, return the original string
    }
  };

  const stringifyAchievements = (achievements: string): string => {
    if (!achievements.trim()) return JSON.stringify([]);
    const items = achievements
      .split(",")
      .map((item) => item.trim())
      .filter((item) => item.length > 0);
    return JSON.stringify(items);
  };

  const handleImageUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;
    const currentImages = formik.values.images || [];
    if (currentImages.length >= 3) {
      alert('Maximum 3 images allowed');
      return;
    }
    try {
      const imageUrl = await uploadImage(file, {});
      if (imageUrl) {
        const updatedImages = [...currentImages, imageUrl];
        formik.setFieldValue('images', updatedImages);
      }
    } catch (err) {
      alert(err instanceof Error ? err.message : 'Upload failed');
    }
  };

  const handleRemoveImage = (index: number) => {
    const updatedImages = (formik.values.images || []).filter((_, i) => i !== index);
    formik.setFieldValue('images', updatedImages);
  };

  const formik = useFormik<StudentFormData>({
    initialValues: student ? {
      name: student.name,
      title: student.title || '',
      grade: student.grade,
      year: student.year || '',
      story: student.story,
      openingQuote: student.openingQuote || '',
      closingQuote: student.closingQuote || '',
      achievements: parseAchievements(student.achievements),
      images: student.images || [],
      featured: student.featured || false,
      category: student.category || '',
    } : {
      name: '',
      title: '',
      grade: '',
      year: '',
      story: '',
      openingQuote: '',
      closingQuote: '',
      achievements: '',
      images: [],
      featured: false,
      category: '',
    },
    validationSchema: studentSchema,
    onSubmit: async (values) => {
      try {
        const submitValues = {
          ...values,
          achievements: stringifyAchievements(values.achievements || ''),
        };
        if (student?.id) {
          await updateMutation.mutateAsync({ id: student.id, values: submitValues });
        } else {
          await createMutation.mutateAsync(submitValues);
        }
        formik.resetForm();
        router.push('/admin/students');
      } catch (err) {
        console.error('Form submission error:', err);
      }
    },
  });
  console.log('Formik values:', formik.values.achievements);
  return (
    <div className="min-h-screen bg-gray-50">
      <div className="sticky top-0 bg-white border-b border-gray-200 shadow-sm z-10">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center gap-4">
            <button
              onClick={() => router.back()}
              className="p-2 hover:bg-gray-100 rounded-lg transition"
              title="Go back"
            >
              <ArrowLeft size={20} className="text-gray-600" />
            </button>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">
                {isEditing ? 'Edit Student Story' : 'Create New Student Story'}
              </h1>
              <p className="text-sm text-gray-600 mt-1">
                {isEditing ? 'Update student information and story' : 'Add a new student story to showcase'}
              </p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <form onSubmit={formik.handleSubmit} className="space-y-8">
          {error && (
            <div className="p-4 bg-red-50 text-red-700 rounded-lg border border-red-200">
              {error instanceof Error ? error.message : 'An error occurred'}
            </div>
          )}

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Basic Information</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <InputField
                label="Student Name"
                name="name"
                type="text"
                placeholder="Enter student name"
                value={formik.values.name}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.name && formik.errors.name ? formik.errors.name : ''}
                disabled={isLoading}
              />
              <InputField
                label="Story Title (Optional)"
                name="title"
                type="text"
                placeholder="e.g., My Experience in the Math Olympiad"
                value={formik.values.title}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.title && formik.errors.title ? formik.errors.title : ''}
                disabled={isLoading}
              />
              <InputField
                label="Grade"
                name="grade"
                type="text"
                placeholder="e.g., 5th Grade, 10th Grade"
                value={formik.values.grade}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.grade && formik.errors.grade ? formik.errors.grade : ''}
                disabled={isLoading}
              />
              <InputField
                label="Academic Year (Optional)"
                name="year"
                type="text"
                placeholder="e.g., 1403 or 2024"
                value={formik.values.year}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.year && formik.errors.year ? formik.errors.year : ''}
                disabled={isLoading}
              />
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Category</label>
                <select
                  name="category"
                  value={formik.values.category}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  disabled={isLoading}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent disabled:opacity-50"
                >
                  <option value="">Select a category</option>
                  <option value="academic">Academic Achievement</option>
                  <option value="sports">Sports</option>
                  <option value="arts">Arts & Culture</option>
                  <option value="other">Other</option>
                </select>
                {formik.touched.category && formik.errors.category && (
                  <p className="text-red-600 text-sm mt-1">{formik.errors.category}</p>
                )}
              </div>
              <div className="flex items-center gap-3 pt-6">
                <input
                  type="checkbox"
                  name="featured"
                  id="featured"
                  checked={formik.values.featured}
                  onChange={formik.handleChange}
                  disabled={isLoading}
                  className="w-4 h-4 text-primary-600 border-gray-300 rounded focus:ring-primary-500"
                />
                <label htmlFor="featured" className="text-sm font-medium text-gray-700">
                  Featured Story (Show on donations & home page)
                </label>
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Story Content</h3>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">Student Story *</label>
              <RichTextEditor
                value={formik.values.story}
                onChange={(value) => formik.setFieldValue('story', value)}
                readOnly={isLoading}
              />
              {formik.touched.story && formik.errors.story && (
                <p className="text-red-600 text-sm mt-1">{formik.errors.story}</p>
              )}
            </div>
            <div className="grid grid-cols-1 gap-4">
              <InputField
                label="Opening Quote (Optional)"
                name="openingQuote"
                type="textarea"
                placeholder="e.g., This journey has shaped me into who I am today."
                value={formik.values.openingQuote}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.openingQuote && formik.errors.openingQuote ? formik.errors.openingQuote : ''}
                disabled={isLoading}
              />
              <InputField
                label="Closing Quote (Optional)"
                name="closingQuote"
                type="textarea"
                placeholder="e.g., Every challenge became an opportunity."
                value={formik.values.closingQuote}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                error={formik.touched.closingQuote && formik.errors.closingQuote ? formik.errors.closingQuote : ''}
                disabled={isLoading}
              />
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Additional Information</h3>
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
              <div className='col-span-2'>
                <label className="block text-sm font-medium text-gray-700 mb-1">Achievements (comma-separated)</label>
                <textarea
                  name="achievements"
                  value={formik.values.achievements}
                  onChange={formik.handleChange}
                  onBlur={formik.handleBlur}
                  placeholder="Bronze Medal, First Place, Student of the Year"
                  disabled={isLoading}
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent disabled:opacity-50"
                />
                <p className="text-gray-500 text-xs mt-1">Separate multiple achievements with commas</p>
                {formik.touched.achievements && formik.errors.achievements && (
                  <p className="text-red-600 text-sm mt-1">{formik.errors.achievements}</p>
                )}
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 space-y-4">
            <h3 className="text-lg font-semibold text-gray-900">Student Photos</h3>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Student Photos (Max 3 images)</label>
              <label className="flex items-center justify-center w-full px-4 py-6 border-2 border-dashed border-gray-300 rounded-lg cursor-pointer hover:border-primary-500 transition">
                <div className="flex items-center gap-2">
                  <Upload size={20} className="text-gray-600" />
                  <span className="text-gray-600">{uploading ? 'Uploading...' : `Click to upload (${formik.values.images?.length || 0}/3)`}</span>
                </div>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleImageUpload}
                  disabled={isLoading || (formik.values.images?.length || 0) >= 3}
                  className="hidden"
                />
              </label>
              {formik.values.images && formik.values.images.length > 0 && (
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4 mt-4">
                  {formik.values.images.map((image, index) => (
                    <div key={index} className="relative group">
                      <div className="relative aspect-[4/3] rounded-lg overflow-hidden bg-gray-100">
                        <Image src={image} alt={`Upload ${index + 1}`} fill className="object-cover" />
                      </div>
                      <button
                        type="button"
                        onClick={() => handleRemoveImage(index)}
                        className="absolute inset-0 flex items-center justify-center bg-black/50 opacity-0 group-hover:opacity-100 transition rounded-lg"
                      >
                        <span className="text-white text-sm font-medium">Remove</span>
                      </button>
                    </div>
                  ))}
                </div>
              )}
              {formik.touched.images && formik.errors.images && (
                <p className="text-red-600 text-sm mt-1">{formik.errors.images}</p>
              )}
            </div>
          </div>

          <div className="flex flex-col sm:flex-row gap-3 justify-end sticky bottom-0 bg-white border-t border-gray-200 p-4 -mx-4 sm:-mx-6 lg:-mx-8">
            <button
              type="button"
              onClick={() => router.back()}
              disabled={isLoading}
              className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition disabled:opacity-50 order-2 sm:order-1"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={isLoading}
              className="px-4 py-2 bg-primary-600 text-white rounded-lg hover:bg-primary-700 transition disabled:opacity-50 order-1 sm:order-2"
            >
              {isLoading ? 'Saving...' : isEditing ? 'Update Student' : 'Create Student'}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}
