"use client";

import React from "react";
import { useRouter } from "next/navigation";
import { useFormik } from "formik";
import * as Yup from "yup";
import { ArrowLeft, Upload, X, Loader } from "lucide-react";
import Button from "@/components/common/ui/button";
import Card from "@/components/common/ui/card";
import InputField from "@/components/common/ui/input-field";
import { useImageUpload } from "@/hooks/useImageUpload";
import { StudentInNeed } from "@/types/types";
import { useCreateStudentInNeed, useUpdateStudentInNeed } from "@/hooks/useStudentInNeed";

const studentInNeedSchema = Yup.object().shape({
  name: Yup.string().required("Name is required").min(2),
  grade: Yup.string().optional(),
  year: Yup.string().optional(),
  story: Yup.string().required("Story is required").min(50),
  background: Yup.string().optional(),
  specificNeed: Yup.string().required("Specific need is required"),
  needCategory: Yup.string().required("Need category is required"),
  contactInfo: Yup.string().optional(),
  achievements: Yup.string().optional(),
  featured: Yup.boolean().optional(),
});

type StudentInNeedFormValues = Yup.InferType<typeof studentInNeedSchema>;

interface StudentInNeedFormProps {
  student?: StudentInNeed;
  isEditing?: boolean;
}

const needCategories = [
  { value: "education", label: "Education (school fees, books, materials)" },
  { value: "housing", label: "Housing (rent, shelter)" },
  { value: "food", label: "Food & Basic Needs" },
  { value: "medical", label: "Medical Treatment" },
  { value: "special-needs", label: "Special Needs Support" },
];

const parseAchievements = (
  achievementsJson: string | null | undefined
): string => {
  if (!achievementsJson) return "";
  try {
    const arr = JSON.parse(achievementsJson);
    return Array.isArray(arr) ? arr.join(", ") : "";
  } catch {
    return "";
  }
};

const stringifyAchievements = (achievements: string): string => {
  if (!achievements.trim()) return JSON.stringify([]);
  const items = achievements
    .split(",")
    .map((a: string) => a.trim())
    .filter((a: string) => a.length > 0);
  return JSON.stringify(items);
};

export default function StudentInNeedForm({
  student,
  isEditing = false,
}: StudentInNeedFormProps) {
  const router = useRouter();
  const { uploadImage, uploading } = useImageUpload();
  const [image, setImage] = React.useState<string>(student?.image || "");
  
  const createMutation = useCreateStudentInNeed();
  const updateMutation = useUpdateStudentInNeed();
  const isPending = createMutation.isPending || updateMutation.isPending;

  const handleFormSubmit = async (values: StudentInNeedFormValues) => {
    const payload: Omit<StudentInNeed, 'id' | 'createdAt' | 'updatedAt'> = {
      name: values.name,
      grade: values.grade || null,
      year: values.year || null,
      story: values.story,
      background: values.background || null,
      specificNeed: values.specificNeed || null,
      needCategory: values.needCategory,
      contactInfo: values.contactInfo || null,
      image: image || null,
      achievements: stringifyAchievements(values.achievements || ""),
      featured: values.featured || false,
    };

    try {
      if (isEditing && student) {
        await updateMutation.mutateAsync({
          id: student.id,
          student: payload,
        });
      } else {
        await createMutation.mutateAsync(payload);
      }
      formik.resetForm();
      router.push("/admin/students/in-need");
    } catch (err) {
      console.error("Error submitting form:", err);
    }
  };

  const formik = useFormik({
    initialValues: {
      name: student?.name || "",
      grade: student?.grade || "",
      year: student?.year || "",
      story: student?.story || "",
      background: student?.background || "",
      specificNeed: student?.specificNeed || "",
      needCategory: student?.needCategory || "education",
      contactInfo: student?.contactInfo || "",
      achievements: parseAchievements(student?.achievements),
      featured: student?.featured || false,
    } as StudentInNeedFormValues,
    validationSchema: studentInNeedSchema,
    onSubmit: handleFormSubmit,
  });

  const handleImageUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    const url = await uploadImage(file);
    if (url) {
      setImage(url);
    }
  };

  const handleRemoveImage = () => {
    setImage("");
  };

  return (
    <div className="min-h-screen bg-gray-50 py-8 sm:py-12">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="sticky top-0 bg-white shadow-sm rounded-lg p-4 sm:p-6 mb-6 flex items-center justify-between">
          <div className="flex items-center gap-4">
            <button
              onClick={() => router.push("/admin/students/in-need")}
              className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
            >
              <ArrowLeft className="w-5 h-5" />
            </button>
            <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">
              {isEditing ? "Edit Student in Need" : "Add Student in Need"}
            </h1>
          </div>
        </div>

        <form onSubmit={formik.handleSubmit} className="space-y-6">
          {/* Basic Information */}
          <Card className="p-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-4">
              Basic Information
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <InputField
                label="Name *"
                name="name"
                placeholder="Student's full name"
                value={formik.values.name}
                onChange={formik.handleChange}
                error={
                  typeof formik.errors.name === "string"
                    ? formik.errors.name
                    : undefined
                }
              />
              <InputField
                label="Grade"
                name="grade"
                placeholder="e.g., 5th Grade, 10th Grade"
                value={formik.values.grade}
                onChange={formik.handleChange}
                error={
                  typeof formik.errors.grade === "string"
                    ? formik.errors.grade
                    : undefined
                }
              />
              <InputField
                label="Year"
                name="year"
                placeholder="e.g., 1403"
                value={formik.values.year}
                onChange={formik.handleChange}
                error={
                  typeof formik.errors.year === "string"
                    ? formik.errors.year
                    : undefined
                }
              />
              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Need Category *
                </label>
                <select
                  name="needCategory"
                  value={formik.values.needCategory}
                  onChange={formik.handleChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                >
                  {needCategories.map((cat) => (
                    <option key={cat.value} value={cat.value}>
                      {cat.label}
                    </option>
                  ))}
                </select>
                {formik.errors.needCategory && (
                  <p className="text-sm text-red-600 mt-1">
                    {typeof formik.errors.needCategory === "string"
                      ? formik.errors.needCategory
                      : ""}
                  </p>
                )}
              </div>
            </div>
          </Card>

          {/* Story & Background */}
          <Card className="p-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-4">
              Story & Background
            </h2>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Family Background
                </label>
                <textarea
                  name="background"
                  value={formik.values.background}
                  onChange={formik.handleChange}
                  placeholder="Describe the student's family situation and background"
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Student&apos;s Story *
                </label>
                <textarea
                  name="story"
                  value={formik.values.story}
                  onChange={formik.handleChange}
                  placeholder="Tell the student's story - their challenges, dreams, and why they need support"
                  rows={5}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                />
                {formik.errors.story && (
                  <p className="text-sm text-red-600 mt-1">
                    {typeof formik.errors.story === "string"
                      ? formik.errors.story
                      : ""}
                  </p>
                )}
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Specific Need *
                </label>
                <textarea
                  name="specificNeed"
                  value={formik.values.specificNeed}
                  onChange={formik.handleChange}
                  placeholder="What specific help does this student need? (e.g., School fees, books, food assistance)"
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                />
                {formik.errors.specificNeed && (
                  <p className="text-sm text-red-600 mt-1">
                    {typeof formik.errors.specificNeed === "string"
                      ? formik.errors.specificNeed
                      : ""}
                  </p>
                )}
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Contact Information (Phone)
                </label>
                <InputField
                  name="contactInfo"
                  placeholder="+93XXXXXXXXX"
                  value={formik.values.contactInfo}
                  onChange={formik.handleChange}
                />
              </div>
            </div>
          </Card>

          {/* Additional Info */}
          <Card className="p-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-4">
              Additional Information
            </h2>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-900 mb-2">
                  Achievements/Skills (comma-separated)
                </label>
                <textarea
                  name="achievements"
                  value={formik.values.achievements}
                  onChange={formik.handleChange}
                  placeholder="e.g., Good at Math, Plays Football, Creative Writing"
                  rows={2}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                />
              </div>

              <label className="flex items-center gap-3 cursor-pointer">
                <input
                  type="checkbox"
                  name="featured"
                  checked={formik.values.featured}
                  onChange={formik.handleChange}
                  className="w-4 h-4 rounded border-gray-300"
                />
                <span className="text-sm font-medium text-gray-900">
                  Show this student&apos;s story on the donation page
                </span>
              </label>
            </div>
          </Card>

          {/* Photo */}
          <Card className="p-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-4">
              Student Photo
            </h2>
            <div className="space-y-4">
              {!image ? (
                <label className="flex  items-center justify-center w-full px-4 py-6 border-2 border-dashed border-gray-300 rounded-lg cursor-pointer hover:border-primary-500 transition">
                  <div className="flex items-center  gap-2">
                    <Upload className="w-10 h-10 text-gray-400 mx-auto" />
                    <div>
                      <p className="text-sm font-medium text-gray-700 mb-1">
                        Click to upload student photo
                      </p>
                      <p className="text-xs text-gray-500">
                        PNG, JPG or GIF (max 5MB)
                      </p>
                    </div>
                  </div>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={handleImageUpload}
                    disabled={uploading}
                    className="hidden"
                  />
                  {uploading && (
                    <div className="mt-2">
                      <Loader className="w-4 h-4 animate-spin text-primary-600 mx-auto" />
                    </div>
                  )}
                </label>
              ) : (
                <div className="relative inline-block">
                  <img
                    src={image}
                    alt="Student photo"
                    className="w-48 h-48 object-cover rounded-lg border border-gray-200"
                  />
                  <button
                    type="button"
                    onClick={() => handleRemoveImage()}
                    className="absolute -top-3 -right-3 bg-red-500 text-white p-2 rounded-full shadow-lg hover:bg-red-600 transition-colors"
                  >
                    <X className="w-5 h-5" />
                  </button>
                </div>
              )}
            </div>
          </Card>

          {/* Action Buttons */}
          <div className="sticky bottom-0 bg-white shadow-lg rounded-lg p-4 sm:p-6 flex gap-4">
            <Button
              type="button"
              variant="outlined"
              onClick={() => router.push("/admin/students/in-need")}
            >
              Cancel
            </Button>
            <Button
              type="submit"
              disabled={isPending || uploading}
              className="flex items-center gap-2"
            >
              {isPending && (
                <Loader className="w-4 h-4 animate-spin" />
              )}
              {isEditing ? "Update Student" : "Add Student"}
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
}
