'use client';

import { useFormik } from 'formik';
import { Teacher, TeacherFormData, teacherSchema, useUpdateTeacher, useCreateTeacher } from '@/hooks';
import { useImageUpload } from '@/hooks/useImageUpload';
import InputField from '@/components/common/ui/input-field';
import RichTextEditor from '@/components/common/ui/rich-text-editor';
import Modal from '@/components/common/ui/modal';
import { Upload } from 'lucide-react';

interface TeacherFormProps {
  teacher?: Teacher;
  onClose: () => void;
  onSuccess?: () => void;
}

export default function TeacherForm({ teacher, onClose, onSuccess }: TeacherFormProps) {
  const createMutation = useCreateTeacher();
  const updateMutation = useUpdateTeacher();
  const { uploading, uploadImage } = useImageUpload();
  
  const isLoading = createMutation.isPending || updateMutation.isPending || uploading;
  const error = createMutation.error || updateMutation.error;

  const handleImageUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    try {
      const imageUrl = await uploadImage(file, {
        gender: formik.values.gender,
        oldImage: teacher?.image,
      });

      if (imageUrl) {
        formik.setFieldValue('image', imageUrl);
      }
    } catch (err) {
      alert(err instanceof Error ? err.message : 'Upload failed');
    }
  };

  const formik = useFormik<TeacherFormData>({
    initialValues: teacher ? {
      name: teacher.name,
      title: teacher.title,
      position: teacher.position,
      subject: teacher.subject || '',
      gender: teacher.gender || '',
      bio: teacher.bio || '',
      qualifications: teacher.qualifications || '',
      experience: teacher.experience || undefined,
      image: teacher.image || '',
      email: teacher.email || '',
      phone: teacher.phone || '',
      active: teacher.active !== false,
    } : {
      name: '',
      title: '',
      position: '',
      subject: '',
      gender: '',
      bio: '',
      qualifications: '',
      experience: undefined,
      image: '',
      email: '',
      phone: '',
      active: true,
    },
    validationSchema: teacherSchema,
    onSubmit: async (values) => {
      try {
        if (teacher?.id) {
          await updateMutation.mutateAsync({ id: teacher.id, values });
        } else {
          await createMutation.mutateAsync(values);
        }
        formik.resetForm();
        onSuccess?.();
        onClose();
      } catch (err) {
        console.error('Form submission error:', err);
      }
    },
  });

  return (
    <Modal
      isOpen={true}
      onClose={onClose}
      title={teacher ? 'Edit Teacher' : 'Add New Teacher'}
      size="xl"
      placement="center"
      onOutSide
    >
      {/* Form */}
      <form onSubmit={formik.handleSubmit} className="space-y-4 pr-2">
          {error && (
            <div className="p-4 bg-red-50 text-red-700 rounded-lg">
              {error instanceof Error ? error.message : 'An error occurred'}
            </div>
          )}

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Name */}
            <InputField
              name="name"
              label="Full Name"
              placeholder="e.g., Mohammad Ahmad"
              value={formik.values.name}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              error={formik.touched.name && formik.errors.name ? formik.errors.name : undefined}
              disabled={isLoading}
              required
            />

            {/* Title */}
            <div>
              <label className="block text-sm font-semibold text-gray-700 mb-2">Title <span className="text-red-500">*</span></label>
              <select
                name="title"
                value={formik.values.title}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                disabled={isLoading}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent outline-none disabled:bg-gray-50"
              >
                <option value="">Select Title</option>
                <option value="Mr.">Mr.</option>
                <option value="Mrs.">Mrs.</option>
                <option value="Ms.">Ms.</option>
                <option value="Dr.">Dr.</option>
                <option value="Prof.">Prof.</option>
              </select>
              {formik.touched.title && formik.errors.title && (
                <p className="text-red-600 text-sm mt-1">{formik.errors.title}</p>
              )}
            </div>

            {/* Position */}
            <InputField
              name="position"
              label="Position/Subject"
              placeholder="e.g., Mathematics Teacher, Biology"
              value={formik.values.position}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              error={formik.touched.position && formik.errors.position ? formik.errors.position : undefined}
              disabled={isLoading}
              required
            />

            {/* Subject */}
            <div>
              <label className="block text-sm font-semibold text-gray-700 mb-2">Subject/Specialization</label>
              <select
                name="subject"
                value={formik.values.subject}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                disabled={isLoading}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent outline-none disabled:bg-gray-50"
              >
                <option value="">Select Subject</option>
                <option value="Mathematics">Mathematics</option>
                <option value="English">English</option>
                <option value="Science">Science</option>
                <option value="Physics">Physics</option>
                <option value="Chemistry">Chemistry</option>
                <option value="Biology">Biology</option>
                <option value="History">History</option>
                <option value="Geography">Geography</option>
                <option value="Islamic Studies">Islamic Studies</option>
                <option value="Dari">Dari</option>
                <option value="Pashto">Pashto</option>
                <option value="Physical Education">Physical Education</option>
                <option value="Art">Art</option>
                <option value="Music">Music</option>
                <option value="Computer Science">Computer Science</option>
              </select>
              {formik.touched.subject && formik.errors.subject && (
                <p className="text-red-600 text-sm mt-1">{formik.errors.subject}</p>
              )}
            </div>

            {/* Gender */}
            <div>
              <label className="block text-sm font-semibold text-gray-700 mb-2">Gender</label>
              <select
                name="gender"
                value={formik.values.gender}
                onChange={formik.handleChange}
                onBlur={formik.handleBlur}
                disabled={isLoading}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent outline-none disabled:bg-gray-50"
              >
                <option value="">Select Gender</option>
                <option value="male">Male</option>
                <option value="female">Female</option>
              </select>
              {formik.touched.gender && formik.errors.gender && (
                <p className="text-red-600 text-sm mt-1">{formik.errors.gender}</p>
              )}
            </div>

            {/* Experience */}
            <InputField
              name="experience"
              label="Experience (Years)"
              type="number"
              placeholder="e.g., 5"
              value={formik.values.experience?.toString() || ''}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              error={formik.touched.experience && formik.errors.experience ? formik.errors.experience : undefined}
              disabled={isLoading}
            />

            {/* Email */}
            <InputField
              name="email"
              label="Email"
              type="email"
              placeholder="teacher@school.com"
              value={formik.values.email}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              error={formik.touched.email && formik.errors.email ? formik.errors.email : undefined}
              disabled={isLoading}
            />

            {/* Phone */}
            <InputField
              name="phone"
              label="Phone"
              placeholder="+93 70 123 4567"
              value={formik.values.phone}
              onChange={formik.handleChange}
              onBlur={formik.handleBlur}
              error={formik.touched.phone && formik.errors.phone ? formik.errors.phone : undefined}
              disabled={isLoading}
            />
          </div>

          {/* Image Upload */}
          <div>
            <label className="block text-sm font-semibold text-gray-700 mb-2">Teacher Image</label>
            <div className="space-y-3">
              {formik.values.image && (
                <div className="relative w-20 h-20 rounded-lg overflow-hidden border border-gray-300">
                  <img
                    src={formik.values.image}
                    alt="Teacher"
                    className="w-full h-full object-cover"
                  />
                </div>
              )}
              <label className="flex items-center justify-center gap-2 px-4 py-2 border-2 border-dashed border-primary-300 rounded-lg hover:bg-primary-50 cursor-pointer transition">
                <Upload size={18} className="text-primary-600" />
                <span className="text-sm font-medium text-primary-600">
                  {uploading ? 'Uploading...' : 'Upload Image'}
                </span>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleImageUpload}
                  disabled={uploading || isLoading}
                  className="hidden"
                />
              </label>
              <p className="text-xs text-gray-500">
                Max 5MB. JPG, PNG, GIF, or WebP
              </p>
              {formik.touched.image && formik.errors.image && (
                <p className="text-red-600 text-sm">{formik.errors.image}</p>
              )}
            </div>
          </div>

          {/* Bio */}
          <div>
            <label className="block text-sm font-semibold text-gray-700 mb-2">Bio</label>
            <div className="border border-gray-300 rounded-lg overflow-hidden focus-within:ring-2 focus-within:ring-primary-500">
              <RichTextEditor
                value={formik.values.bio || ''}
                onChange={(content) => formik.setFieldValue('bio', content)}
              />
            </div>
            {formik.touched.bio && formik.errors.bio && (
              <p className="text-red-600 text-sm mt-1">{formik.errors.bio}</p>
            )}
          </div>

          {/* Qualifications */}
          <div>
            <label className="block text-sm font-semibold text-gray-700 mb-2">Qualifications</label>
            <div className="border border-gray-300 rounded-lg overflow-hidden focus-within:ring-2 focus-within:ring-primary-500">
              <RichTextEditor
                value={formik.values.qualifications || ''}
                onChange={(content) => formik.setFieldValue('qualifications', content)}
              />
            </div>
            {formik.touched.qualifications && formik.errors.qualifications && (
              <p className="text-red-600 text-sm mt-1">{formik.errors.qualifications}</p>
            )}
          </div>

          {/* Active Status */}
          <div className="flex items-center gap-2">
            <input
              type="checkbox"
              name="active"
              id="active"
              checked={formik.values.active}
              onChange={formik.handleChange}
              disabled={isLoading}
              className="w-4 h-4 text-primary-600 rounded cursor-pointer"
            />
            <label htmlFor="active" className="text-sm font-semibold text-gray-700 cursor-pointer">
              Active
            </label>
          </div>

          {/* Buttons */}
          <div className="flex gap-3 pt-4 border-t">
            <button
              type="button"
              onClick={onClose}
              disabled={isLoading}
              className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 disabled:opacity-50"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={isLoading}
              className="flex-1 px-4 py-2 bg-primary-600 text-white rounded-lg hover:bg-primary-700 disabled:opacity-50 flex items-center justify-center gap-2"
            >
              {isLoading ? 'Saving...' : (teacher ? 'Update Teacher' : 'Add Teacher')}
            </button>
          </div>
        </form>
    </Modal>
  );
}
