'use client';

import { useState } from 'react';
import { Modal } from '@/components/common/ui';

interface UserApprovalModalProps {
  isOpen: boolean;
  onClose: () => void;
  onApprove: (role: 'admin' | 'editor' | 'viewer') => void;
  userEmail: string;
  isPending?: boolean;
}

export default function UserApprovalModal({
  isOpen,
  onClose,
  onApprove,
  userEmail,
  isPending = false,
}: UserApprovalModalProps) {
  const [selectedRole, setSelectedRole] = useState<'admin' | 'editor' | 'viewer'>('viewer');

  const handleApprove = () => {
    onApprove(selectedRole);
    
    // Reset state
    setSelectedRole('viewer');
  };

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Approve User"
      size='lg'
    >
      <div className="space-y-6">
        {/* User Info */}
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
          <p className="text-sm font-medium text-gray-700">Approving User</p>
          <p className="text-lg font-semibold text-gray-900 mt-1">{userEmail}</p>
        </div>

        {/* Important Notice */}
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex gap-3">
            <svg className="w-5 h-5 text-yellow-600 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
            <div>
              <p className="text-sm font-medium text-yellow-900">
                Permissions Required
              </p>
              <p className="text-sm text-yellow-700 mt-1">
                After approval, you must set permissions for this user using the &quot;Edit&quot; button. Without permissions, they can only view their profile page.
              </p>
            </div>
          </div>
        </div>

        {/* Role Selection */}
        <div className="space-y-3">
          <label className="block text-sm font-medium text-gray-700">
            Select User Role
          </label>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
            {/* Admin Role */}
            <button
              type="button"
              onClick={() => setSelectedRole('admin')}
              className={`relative rounded-lg border-2 p-4 text-left transition-all ${
                selectedRole === 'admin'
                  ? 'border-red-500 bg-red-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-semibold text-gray-900">Administrator</span>
                <input
                  type="radio"
                  checked={selectedRole === 'admin'}
                  onChange={() => setSelectedRole('admin')}
                  className="h-4 w-4 text-red-600"
                />
              </div>
              <p className="text-xs text-gray-600">
                Full access to all features, pages, and settings. Can manage users and permissions.
              </p>
            </button>

            {/* Editor Role */}
            <button
              type="button"
              onClick={() => setSelectedRole('editor')}
              className={`relative rounded-lg border-2 p-4 text-left transition-all ${
                selectedRole === 'editor'
                  ? 'border-blue-500 bg-blue-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-semibold text-gray-900">Content Editor</span>
                <input
                  type="radio"
                  checked={selectedRole === 'editor'}
                  onChange={() => setSelectedRole('editor')}
                  className="h-4 w-4 text-blue-600"
                />
              </div>
              <p className="text-xs text-gray-600">
                Can manage content. Permissions must be set after approval.
              </p>
            </button>

            {/* Viewer Role */}
            <button
              type="button"
              onClick={() => setSelectedRole('viewer')}
              className={`relative rounded-lg border-2 p-4 text-left transition-all ${
                selectedRole === 'viewer'
                  ? 'border-gray-500 bg-gray-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-semibold text-gray-900">Viewer</span>
                <input
                  type="radio"
                  checked={selectedRole === 'viewer'}
                  onChange={() => setSelectedRole('viewer')}
                  className="h-4 w-4 text-gray-600"
                />
              </div>
              <p className="text-xs text-gray-600">
                Read-only access. Permissions must be set after approval.
              </p>
            </button>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex justify-end gap-3 pt-4 border-t border-gray-200">
          <button
            type="button"
            onClick={onClose}
            className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50"
            disabled={isPending}
          >
            Cancel
          </button>
          <button
            type="button"
            onClick={handleApprove}
            disabled={isPending}
            className="px-6 py-2 text-sm font-medium text-white bg-green-600 rounded-lg hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {isPending ? 'Approving...' : 'Approve User'}
          </button>
        </div>
      </div>
    </Modal>
  );
}
