'use client';

import React, { useState } from 'react';
import { useFormik } from 'formik';
import { useMutation } from '@tanstack/react-query';
import * as Yup from 'yup';
import { AlertCircle, CheckCircle, Loader2, Save, Lock } from 'lucide-react';
import Button from '@/components/common/ui/button';
import InputField from '@/components/common/ui/input-field';

// Validation schema for admin user
const USER_SCHEMA = Yup.object().shape({
  email: Yup.string().email('Invalid email').required('Email is required'),
  name: Yup.string().required('Name is required').min(3).max(100),
  role: Yup.string().required('Role is required').oneOf(['admin', 'editor', 'viewer']),
  password: Yup.string().when('isNew', {
    is: true,
    then: (schema) => schema.required('Password is required').min(8),
    otherwise: (schema) => schema.optional().min(8),
  }),
  active: Yup.boolean(),
});

type UserFormValues = Yup.InferType<typeof USER_SCHEMA>;

interface UserManagementFormProps {
  initialData?: Partial<UserFormValues & { isNew: boolean }>;
  onSuccess?: () => void;
}

const submitUser = async (values: UserFormValues): Promise<{ success: boolean; message: string }> => {
  const response = await fetch('/api/admin/users', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(values),
  });

  const data = await response.json();
  if (!response.ok) throw new Error(data.message || 'Failed to save user');
  return data;
};

export default function UserManagementForm({
  initialData,
  onSuccess,
}: UserManagementFormProps) {
  const [successMessage, setSuccessMessage] = useState('');
  const [errorMessage, setErrorMessage] = useState('');
  const isNew = initialData?.isNew ?? true;

  const initialValues = {
    email: initialData?.email || '',
    name: initialData?.name || '',
    role: initialData?.role || 'editor',
    password: '',
    active: initialData?.active !== false,
  };

  const mutation = useMutation({
    mutationFn: submitUser,
    onSuccess: (data) => {
      setSuccessMessage(data.message || 'User saved successfully!');
      setErrorMessage('');

      setTimeout(() => {
        setSuccessMessage('');
        onSuccess?.();
      }, 3000);
    },
    onError: (error) => {
      setErrorMessage(error instanceof Error ? error.message : 'Failed to save user');
      setSuccessMessage('');
    },
  });

  const formik = useFormik({
    initialValues,
    validationSchema: USER_SCHEMA,
    onSubmit: (values) => mutation.mutate(values),
  });

  return (
    <form
      onSubmit={formik.handleSubmit}
      className="space-y-6 bg-white p-6 rounded-lg shadow"
    >
      {/* Messages */}
      {successMessage && (
        <div className="flex items-center gap-3 p-4 bg-green-50 border border-green-200 rounded-lg">
          <CheckCircle className="w-5 h-5 text-green-600" />
          <p className="text-green-800">{successMessage}</p>
        </div>
      )}

      {errorMessage && (
        <div className="flex items-center gap-3 p-4 bg-red-50 border border-red-200 rounded-lg">
          <AlertCircle className="w-5 h-5 text-red-600" />
          <p className="text-red-800">{errorMessage}</p>
        </div>
      )}

      {/* Email and Name */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <InputField
          label="Email Address"
          name="email"
          type="email"
          value={formik.values.email}
          onChange={formik.handleChange}
          onBlur={formik.handleBlur}
          error={formik.touched.email ? formik.errors.email : undefined}
          placeholder="user@example.com"
          disabled={!isNew}
        />

        <InputField
          label="Full Name"
          name="name"
          type="text"
          value={formik.values.name}
          onChange={formik.handleChange}
          onBlur={formik.handleBlur}
          error={formik.touched.name ? formik.errors.name : undefined}
          placeholder="John Doe"
        />
      </div>

      {/* Role and Status */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Role
          </label>
          <select
            name="role"
            value={formik.values.role}
            onChange={formik.handleChange}
            onBlur={formik.handleBlur}
            className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
          >
            <option value="admin">Administrator</option>
            <option value="editor">Editor</option>
            <option value="viewer">Viewer</option>
          </select>
          {formik.touched.role && formik.errors.role && (
            <p className="text-sm text-red-600 mt-1">{formik.errors.role}</p>
          )}
        </div>

        <div className="flex items-end">
          <label className="flex items-center cursor-pointer">
            <input
              type="checkbox"
              name="active"
              checked={formik.values.active}
              onChange={formik.handleChange}
              className="w-5 h-5 text-primary-600 rounded focus:ring-2 focus:ring-primary-500"
            />
            <span className="ml-3 text-sm font-medium text-gray-700">
              Active Account
            </span>
          </label>
        </div>
      </div>

      {/* Password */}
      <div>
        <div className="flex items-center gap-2 mb-2">
          <Lock className="w-4 h-4 text-gray-600" />
          <label className="block text-sm font-medium text-gray-700">
            {isNew ? 'Password' : 'New Password (optional)'}
          </label>
        </div>
        <input
          type="password"
          name="password"
          value={formik.values.password}
          onChange={formik.handleChange}
          onBlur={formik.handleBlur}
          className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
          placeholder={isNew ? 'Minimum 8 characters' : 'Leave blank to keep current'}
        />
        {formik.touched.password && formik.errors.password && (
          <p className="text-sm text-red-600 mt-1">{formik.errors.password}</p>
        )}
      </div>

      {/* Info Box */}
      <div className="p-4 bg-blue-50 border border-blue-200 rounded-lg">
        <p className="text-sm text-blue-800">
          <strong>Roles:</strong> Admin (full access), Editor (create/edit content), Viewer (read-only)
        </p>
      </div>

      {/* Submit Button */}
      <div className="flex items-center gap-3 justify-end pt-6 border-t">
        <Button
          type="submit"
          disabled={mutation.isPending}
          className="flex items-center gap-2"
        >
          {mutation.isPending ? (
            <>
              <Loader2 className="w-4 h-4 animate-spin" />
              Saving...
            </>
          ) : (
            <>
              <Save className="w-4 h-4" />
              Save User
            </>
          )}
        </Button>
      </div>
    </form>
  );
}
