"use client";
import {
  Dialog,
  DialogBackdrop,
  DialogPanel,
  TransitionChild,
} from "@headlessui/react";
import classNames from "classnames";
import {
  Brush,
  Contact2,
  DollarSign,
  FileText,
  Globe,
  GraduationCap,
  LayoutDashboard,
  LogOut,
  Mail,
  Newspaper,
  Settings,
  User,
  Users,
} from "lucide-react";
import { signOut } from "next-auth/react";
import Link from "next/link";
import { usePathname } from "next/navigation";
import { useEffect, useState } from "react";
import { LuX } from "react-icons/lu";
import { useAuth } from "@/context/auth-context";
import { usePublicSiteSettings } from "@/hooks";
import Image from "next/image";

const navigation = [
  { name: "Dashboard", href: "/admin", icon: LayoutDashboard },
  { name: "News", href: "/admin/news", icon: FileText },
  { name: "Teachers", href: "/admin/teachers", icon: Users },
  { name: "Leadership", href: "/admin/leadership", icon: User },
  { name: "Students", href: "/admin/students", icon: Contact2 },
  { name: "Contacts", href: "/admin/contacts", icon: Mail },
  { name: "Admissions", href: "/admin/admissions", icon: GraduationCap },
  { name: "Donations", href: "/admin/donations", icon: DollarSign },
  { name: "Newsletter", href: "/admin/newsletter", icon: Newspaper },
  { name: "Site Settings", href: "/admin/site-settings", icon: Globe },
  { name: "Users", href: "/admin/users", icon: Users },
  { name: "Settings", href: "/admin/settings", icon: Settings },
];

const SidebarList = ({ onNavigate }: { onNavigate?: () => void }) => {
  const pathname = usePathname();
  const { user } = useAuth();
  const { data: settings } = usePublicSiteSettings([
    "logo_url",
    "logo_text",
    "logo_initials",
    "show_logo_text",
  ]);

  const logoText = settings?.logo_text || "Setaragan Zirak";
  const logoInitials = settings?.logo_initials || "SZ";
  const showLogoText = settings?.show_logo_text !== "false";
  const logoUrl = settings?.logo_url;

  const [selectedMenu, setSelectedMenu] = useState("");

  const handleActiveMenu = (menu: string) => {
    setSelectedMenu(menu);
    if (onNavigate) onNavigate();
  };

  useEffect(() => {
    if (!pathname) return;
    navigation.forEach((nav) => {
      if (pathname.startsWith(nav.href)) {
        setSelectedMenu(nav.href);
      }
    });
  }, [pathname]);

  const handleSignOut = async () => {
    // next-auth sign out logic here
    await signOut();
    window.location.href = "/auth/signin";
    console.log("User signed out");
  };

  return (
    <>
      <div className="ml-3 sticky top-0 flex h-16 shrink-0 justify-start items-center gap-x-3 border-b border-slate-300 bg-white shadow-xs px-3">
        <div className="w-8 h-8">
          {logoUrl ? (
            <Image
              src={logoUrl}
              alt={logoText}
              width={0}
              height={0}
              className="w-full h-full object-contain"
            />
          ) : (
            <>
              <div className="absolute inset-0 bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600 rounded-xl flex items-center justify-center shadow-lg group-hover:shadow-2xl transition-all duration-300 group-hover:scale-110 group-hover:rotate-3">
                <span className="text-white font-bold text-base drop-shadow-sm">
                  {logoInitials}
                </span>
              </div>
              <div className="absolute inset-0 bg-gradient-to-br from-primary-400 to-secondary-400 rounded-xl opacity-0 group-hover:opacity-20 transition-opacity duration-300"></div>
            </>
          )}
      </div>
          {showLogoText && (
            <div className="flex flex-col">
              <span className="font-bold text-xl hidden sm:block bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent leading-tight">
                {logoText} 
              </span>
              <span className="font-medium text-sm block sm:hidden bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                {logoInitials} School
              </span>
            </div>
          )}
        </div>
      {/*begin::Menu Container*/}
      <div className="h-0 flex-1 mr-2.5 mt-2.5 overflow-y-auto  lg:overflow-y-hidden lg:hover:overflow-y-auto ">
        {/*begin::Menu Nav*/}
        <ul className=" mt-5 flex-1 space-y-2 px-2 ml-3">
          {navigation.map(({ href, name, icon: Icon }) => (
            <li key={name} className={`w-full`}>
              <Link href={href}>
                <div
                  onClick={() => handleActiveMenu(href)}
                  aria-current={selectedMenu === href ? "page" : undefined}
                  className={classNames(
                    `group text-base px-3 py-2 rounded-md font-medium flex self-stretch items-center gap-x-2.5 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-300`,
                    selectedMenu === href
                      ? "bg-primary-500 text-white hover:bg-primary-600"
                      : "text-primary-900 hover:bg-primary-50"
                  )}
                >
                  <span>
                    <Icon
                      className={classNames(`h-5 w-5 text-icon-dark `, {
                        "text-white": selectedMenu === href,
                        "text-primary-700": selectedMenu !== href,
                      })}
                    />
                  </span>
                  <span className={`w-full`}>{name}</span>
                </div>
              </Link>
            </li>
          ))}
        </ul>
      </div>
      <div className="border-t border-blue-700 p-4">
        <div className="flex items-center space-x-3">
          <div className="h-10 w-10 rounded-full bg-gray-100 flex items-center justify-center">
            <User className="h-6 w-6 text-gray-700" />
          </div>
          <div className="flex-1 min-w-0">
            <p className="text-sm font-medium truncate">
              {user?.name || "Admin User"}
            </p>
            <p className="text-xs text-gray-500 truncate">
              {user?.email || "admin@school.com"}
            </p>
          </div>
          <button
            className="text-gray-600 hover:text-gray-800 transition-colors"
            onClick={handleSignOut}
          >
            <LogOut className="h-5 w-5" />
          </button>
        </div>
      </div>
    </>
  );
};

export default function Sidebar({
  sidebarOpen,
  setSidebarOpen,
}: {
  sidebarOpen: boolean;
  setSidebarOpen: (open: boolean) => void;
}) {
  return (
    <>
      {/* Mobile sidebar */}
      <Dialog
        open={sidebarOpen}
        onClose={() => setSidebarOpen(false)}
        className="relative z-50 lg:hidden h-screen"
        as="div"
      >
        {/* Backdrop */}
        <DialogBackdrop
          transition
          className="fixed inset-0 bg-gray-900/80 transition-opacity duration-300 ease-linear data-[closed]:opacity-0"
        />

        {/* Sidebar panel */}
        <div className="fixed inset-0 ">
          <DialogPanel
            transition
            className="relative h-full mr-16 flex w-full max-w-xs flex-1 transform transition duration-300 ease-in-out data-[closed]:-translate-x-full"
          >
            {/* Close button */}
            <TransitionChild>
              <div className="absolute left-full top-0 flex w-16 justify-center pt-5 duration-300 ease-in-out data-[closed]:opacity-0">
                <button
                  type="button"
                  onClick={() => setSidebarOpen(false)}
                  aria-label="Close sidebar"
                  // data-autofocus
                  className="-m-2.5 p-2.5 text-white hover:text-gray-300 focus:outline-none focus:ring-2 focus:ring-white rounded"
                >
                  <span className="sr-only">Close sidebar</span>
                  <LuX aria-hidden="true" className="h-6 w-6" />
                </button>
              </div>
            </TransitionChild>

            {/* Sidebar content */}
            <div className="flex grow flex-col gap-y-5 overflow-y-auto bg-white pb-4 scrollbarLight">
              <SidebarList onNavigate={() => setSidebarOpen(false)} />
            </div>
          </DialogPanel>
        </div>
      </Dialog>

      {/* Static sidebar for desktop */}
      <div className="hidden h-screen lg:fixed lg:inset-y-0 lg:z-50 lg:flex lg:w-64 lg:flex-col">
        {/* Sidebar component, swap this element with another sidebar if you like */}
        <div className="flex grow flex-col gap-y-5 overflow-y-auto scrollbarLight bg-white pb-4 border-r border-slate-300">
          <SidebarList />
        </div>
      </div>
    </>
  );
}
